package eu.dnetlib.data.mapreduce.util;

import com.google.common.base.Splitter;
import com.google.common.collect.Sets;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.protobuf.Descriptors;
import com.google.protobuf.GeneratedMessage;
import eu.dnetlib.data.graph.model.DNGFDecoder;
import eu.dnetlib.data.graph.model.DNGFEntityDecoder;
import eu.dnetlib.data.graph.model.DNGFRelDecoder;
import eu.dnetlib.data.graph.utils.RelDescriptor;
import eu.dnetlib.data.mapreduce.hbase.index.config.ContextDef;
import eu.dnetlib.data.mapreduce.hbase.index.config.ContextMapper;
import eu.dnetlib.data.mapreduce.hbase.index.config.EntityConfigTable;
import eu.dnetlib.data.mapreduce.hbase.index.config.LinkDescriptor;
import eu.dnetlib.data.proto.*;
import eu.dnetlib.data.transform.Ontologies;
import org.apache.commons.lang3.StringUtils;

import javax.xml.transform.TransformerException;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Created by sandro on 1/9/17.
 */
public class JSONRecordFactory extends AbstractRecordFactory {

    public JSONRecordFactory(EntityConfigTable entityConfigTable, ContextMapper contextMapper, Ontologies ontologies, boolean entityDefaults, boolean relDefaults, boolean childDefaults) {
        super(entityConfigTable, contextMapper, ontologies, entityDefaults, relDefaults, childDefaults);
    }

    @Override
    public String build() {
        try {
            final DNGFEntityDecoder entity = mainEntity.decodeEntity();

            final JsonObject result = new JsonObject();

            final TypeProtos.Type type = entity.getType();


            // rels has to be processed before the contexts because they enrich the contextMap with the funding info.
            final JsonArray rels = listRelations();
            result.add("rels", rels);
            buildContexts(type, result);


//          metadata.add(parseDataInfo(mainEntity));
//
//          final String body = templateFactory.buildBody(type, metadata, rels, listChildren(), extraInfo);

            result.add("metadata", decodeType(entity, null, entityDefaults, false));

            return result.toString();

        } catch (final Throwable e) {
            throw new RuntimeException(String.format("error building record '%s'", this.key), e);
        }
    }

    private JsonObject decodeType(final DNGFEntityDecoder decoder, final Set<String> filter, final boolean defaults, final boolean expandingRel) {

        final JsonObject root = new JsonObject();
        addFields(root, decoder.getMetadata(), filter, defaults, expandingRel);
        if ((decoder.getEntity() instanceof PublicationProtos.Publication) && !expandingRel) {
            asJsonElement("bestlicense", "", getBestLicense(), null, root);
            addFields(root, decoder.getEntity(), filter, defaults, expandingRel);
        }
        if ((decoder.getEntity() instanceof PersonProtos.Person) && !expandingRel) {
            addFields(root, decoder.getEntity(), filter, defaults, expandingRel);
        }
        if ((decoder.getEntity() instanceof ProjectProtos.Project) && !expandingRel) {
            addFields(root, decoder.getEntity(), filter, defaults, expandingRel);
        }

        return root;
    }


    private void addFields(final JsonObject metadata, final GeneratedMessage fields, final Set<String> filter, final boolean defaults, final boolean expandingRel) {

        if (fields != null) {
            final Set<String> seen = Sets.newHashSet();
            for (final Map.Entry<Descriptors.FieldDescriptor, Object> e : filterFields(fields, filter)) {
                final String name = e.getKey().getName();
                seen.add(name);
                addFieldValue(metadata, e.getKey(), e.getValue(), expandingRel);
            }
            if (defaults) {
                for (final Descriptors.FieldDescriptor fd : fields.getDescriptorForType().getFields()) {
                    if (!seen.contains(fd.getName())) {
                        addFieldValue(metadata, fd, getDefault(fd), expandingRel);
                    }
                }
            }
        }

    }


    private void addFieldValue(final JsonObject metadata, final Descriptors.FieldDescriptor fd, final Object value, final boolean expandingRel) {
        if ("dateofcollection".equals(fd.getName()) ||
                "dateoftransformation".equals(fd.getName()) ||
                "id".equals(fd.getName()) ||
                (value == null)) return;

        if (fd.getName().equals("datasourcetype")) {
            final String classid = ((FieldTypeProtos.Qualifier) value).getClassid();

            final FieldTypeProtos.Qualifier.Builder q = FieldTypeProtos.Qualifier.newBuilder((FieldTypeProtos.Qualifier) value);
            if (specialDatasourceTypes.contains(classid)) {
                q.setClassid("other").setClassname("other");
            }

            asJsonElement("datasourcetypeui", "", q.build(), null, metadata);
        }

        if (fd.isRepeated() && (value instanceof List<?>)) {
            for (final Object o : (List<Object>) value) {
                guessType(metadata, fd, o, expandingRel);
            }
        } else {
            guessType(metadata, fd, value, expandingRel);
        }
    }

    private void guessType(JsonObject metadata, Descriptors.FieldDescriptor fd, Object object, boolean expandingRel) {

        if (fd.getType().equals(Descriptors.FieldDescriptor.Type.MESSAGE)) {

            if (FieldTypeProtos.Qualifier.getDescriptor().equals(fd.getMessageType())) {
                final FieldTypeProtos.Qualifier qualifier = (FieldTypeProtos.Qualifier) object;
                asJsonElement(fd.getName(), "", qualifier, null, metadata);
            } else if (FieldTypeProtos.StructuredProperty.getDescriptor().equals(fd.getMessageType())) {
                final FieldTypeProtos.StructuredProperty sp = (FieldTypeProtos.StructuredProperty) object;
                asJsonElement(fd.getName(), sp.getValue(), sp.getQualifier(), sp.hasDataInfo() ? sp.getDataInfo() : null, metadata);

                if (!expandingRel && fd.getName().equals("pid")) {
                    if (sp.getQualifier().getClassid().equalsIgnoreCase("doi")) {
                        incrementCounter("doi");
                    }
                }
            } else if (FieldTypeProtos.KeyValue.getDescriptor().equals(fd.getMessageType())) {
                final FieldTypeProtos.KeyValue kv = (FieldTypeProtos.KeyValue) object;

                JsonObject tmp = new JsonObject();
                tmp.addProperty("name", kv.getValue());
                tmp.addProperty("id", removePrefix(kv.getValue()));
                metadata.add(fd.getName(), tmp);

            } else if (FieldTypeProtos.StringField.getDescriptor().equals(fd.getMessageType())) {
                final String fieldName = fd.getName();
                final FieldTypeProtos.StringField sf = (FieldTypeProtos.StringField) object;
                final JsonObject sb = new JsonObject();
                if (sf.hasDataInfo()) {
                    final FieldTypeProtos.DataInfo dataInfo = sf.getDataInfo();
                    dataInfoAsAttributes(sb, dataInfo);
                    if (!StringUtils.isEmpty(sf.getValue()))
                        sb.addProperty("value", sf.getValue());
                    if (sb.entrySet().size() > 0)
                        metadata.add(fieldName, sb);
                } else {
                    if (!StringUtils.isEmpty(sf.getValue()))
                        metadata.addProperty(fd.getName(), sf.getValue());
                }
            } else if (FieldTypeProtos.BoolField.getDescriptor().equals(fd.getMessageType())) {
                final FieldTypeProtos.BoolField bf = (FieldTypeProtos.BoolField) object;
                if (bf.hasDataInfo()) {
                    final JsonObject sb = new JsonObject();
                    final FieldTypeProtos.DataInfo dataInfo = bf.getDataInfo();
                    dataInfoAsAttributes(sb, dataInfo);
                    sb.addProperty("value", bf.getValue());
                    metadata.add(fd.getName(), sb);
                } else
                    metadata.addProperty(fd.getName(), bf.getValue());

            } else if (FieldTypeProtos.Journal.getDescriptor().equals(fd.getMessageType()) && (object != null)) {
                final FieldTypeProtos.Journal j = (FieldTypeProtos.Journal) object;
                JsonObject journal = new JsonObject();
                journal.addProperty("issn", j.getIssnPrinted());
                journal.addProperty("eissn", j.getIssnOnline());
                journal.addProperty("lissn", j.getIssnLinking());
                journal.addProperty("value", j.getName());
                metadata.add("journal", journal);
            } else if (FieldTypeProtos.Context.getDescriptor().equals(fd.getMessageType()) && (object != null)) {
                contextes.add(((FieldTypeProtos.Context) object).getId());
            } else if (FieldTypeProtos.ExtraInfo.getDescriptor().equals(fd.getMessageType()) && (object != null)) {
                final FieldTypeProtos.ExtraInfo e = (FieldTypeProtos.ExtraInfo) object;
                final JsonObject sb = new JsonObject();
                sb.addProperty("name", e.getName());
                sb.addProperty("typology", e.getTypology());
                sb.addProperty("provenance", e.getProvenance());
                sb.addProperty("trust", e.getTrust());
                sb.addProperty("value", e.getValue());
                metadata.add(fd.getName(), sb);
            }


        } else if (fd.getType().equals(Descriptors.FieldDescriptor.Type.ENUM)) {
            if (fd.getFullName().equals("eu.dnetlib.data.proto.DNGFEntity.type")) return;
            asJsonElement(fd.getName(), ((Descriptors.EnumValueDescriptor) object).getName(), null, null, metadata);
        } else {
            asJsonElement(fd.getName(), object.toString(), null, null, metadata);
        }

    }


    private JsonArray listRelations() {

        JsonArray relations = new JsonArray();

        for (final DNGFDecoder decoder : this.relations) {

            JsonObject root = new JsonObject();


            final DNGFProtos.DNGFRel rel = decoder.getDNGFRel();
            final DNGFProtos.DNGFEntity cachedTarget = rel.getCachedTarget();
            final DNGFRelDecoder relDecoder = DNGFRelDecoder.decode(rel);

            // if (!relDecoder.getRelType().equals(RelType.personResult) || relDecoder.getRelTargetId().equals(key)) {
            if (relDecoder.getRelSourceId().equals(key) || relDecoder.getRelTargetId().equals(key)) {


                final TypeProtos.Type targetType = relDecoder.getTargetType();
                final Set<String> relFilter = entityConfigTable.getFilter(targetType, relDecoder.getRelDescriptor());


                addFields(root, relDecoder.getDngfRel(), relFilter, false, true);


                final RelDescriptor relDescriptor = relDecoder.getRelDescriptor();

                if ((cachedTarget != null) && cachedTarget.isInitialized()) {

                    final Set<String> filter = entityConfigTable.getFilter(targetType, relDescriptor);
                    root.add("cachedTarget", decodeType(DNGFEntityDecoder.decode(cachedTarget), filter, relDefaults, true));
                }

                final String semanticclass = ontologies.inverseOf(relDescriptor);
                final String semanticscheme = relDescriptor.getOntologyCode();

                final String rd = semanticscheme + "_" + semanticclass;
                incrementCounter(rd);

                final FieldTypeProtos.DataInfo info = decoder.getDNGF().getDataInfo();

                manageInferred(rd, info);

                final LinkDescriptor ld = entityConfigTable.getDescriptor(relDecoder.getTargetType(), relDescriptor);

                final String relId = (ld != null) && !ld.isSymmetric() ? relDecoder.getRelTargetId() : relDecoder.getRelSourceId();

                relations.add(root);

//                rels.add(templateFactory.getRel(targetType, relId, metadata, semanticclass, semanticscheme, info.getInferred(), info.getTrust(),
//                        info.getInferenceprovenance(), info.getProvenanceaction().getClassid()));
            }
        }
        return relations;
    }


    private void buildContexts(final TypeProtos.Type type, final JsonObject metadata) {

        if ((contextMapper != null) && !contextMapper.isEmpty() && type.equals(TypeProtos.Type.publication)) {
            JsonObject contextRoot = new JsonObject();
            JsonArray contexts = new JsonArray();
            contextRoot.add("contextRoot", contexts);

            for (final String context : contextes) {

                String id = "";
                for (final String token : Splitter.on("::").split(context)) {
                    id += token;

                    final ContextDef def = contextMapper.get(id);

                    if (def == null)
                        throw new IllegalStateException(String.format("cannot find context for id '%s'", id));

                    if (def.getName().equals("context")) {
                        if (!findExistingContext(contexts,def.getId()))
                            addContextDef(contexts,def);
                    }

                    if (def.getName().equals("category")) {
                        final String rootId = StringUtils.substringBefore(def.getId(), "::");
                        addContextDef(contexts, def);
                    }

                    if (def.getName().equals("concept")) {
                        addContextDef(contexts, def);
                    }
                    id += "::";
                }
            }

        }
    }

    private boolean findExistingContext(JsonArray contexes, String id) {
        for(JsonElement item:contexes) {
            if(item.getAsJsonObject().has("context")) {
                if (item.getAsJsonObject().get("context").getAsJsonObject().get("id").getAsString().equals(id)) {
                    return true;
                }
            }
        }
        return false;

    }


    private void addContextDef(final JsonArray tags, final ContextDef def) {

        final JsonObject defContext = new JsonObject();
        defContext.addProperty("id",def.getId());
        defContext.addProperty("label", def.getLabel());
        if ((def.getType() != null) && !def.getType().isEmpty()) {
            defContext.addProperty("type", def.getType());
        }
        final JsonObject result = new JsonObject();
        result.add(def.getName(),defContext);
        tags.add(result);
    }


    private void asJsonElement(final String name, final String value, final FieldTypeProtos.Qualifier q, final FieldTypeProtos.DataInfo dataInfo, final JsonObject metadata) {

        JsonObject root = new JsonObject();
        getAttributes(q, root);
        if (dataInfo != null) {
            dataInfoAsAttributes(root, dataInfo);
        }
        if (value != null && !StringUtils.isEmpty(value))
            root.addProperty("value", value);

        if (root.entrySet().size() > 0)
            metadata.add(name, root);

    }

    private void getAttributes(final FieldTypeProtos.Qualifier q, JsonObject root) {
        if (q == null) return;
        q.getAllFields().entrySet().forEach(e -> root.addProperty(e.getKey().getName(), e.getValue().toString()));


    }

    private void dataInfoAsAttributes(final JsonObject root, final FieldTypeProtos.DataInfo dataInfo) {
        root.addProperty("inferred", dataInfo.getInferred());
        root.addProperty("inferenceprovenance", dataInfo.getInferenceprovenance());
        root.addProperty("provenanceaction", dataInfo.getProvenanceaction().getClassid());
        root.addProperty("trust", dataInfo.getTrust());

    }


}
