package eu.dnetlib.data.mapreduce.hbase.dli;

import java.io.IOException;
import java.util.List;

import com.google.common.collect.Lists;
import com.google.protobuf.InvalidProtocolBufferException;
import com.googlecode.protobuf.format.JsonFormat;
import eu.dnetlib.data.mapreduce.hbase.dli.kv.DliKey;
import eu.dnetlib.data.proto.dli.ScholixObjectProtos.Scholix;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.hbase.io.ImmutableBytesWritable;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.mapreduce.Reducer;

/**
 * Created by claudio on 13/03/2017.
 */
public class PrepareScholixDataReducer2 extends Reducer<DliKey, ImmutableBytesWritable, Text, Text> {

	private static final Log log = LogFactory.getLog(PrepareScholixDataReducer2.class); // NOPMD by marko on 11/24/08 5:02 PM

	private Text outKey;

	private Text outValue;

	@Override
	protected void setup(final Context context) throws IOException, InterruptedException {
		outKey = new Text("");
		outValue = new Text();
	}

	@Override
	protected void reduce(DliKey key, Iterable<ImmutableBytesWritable> values, Context context) throws IOException, InterruptedException {

		final List<Scholix> scholixes = Lists.newArrayList();
		values.forEach(i ->	scholixes.add(parse(key, i).build()));

		System.out.println("reducing key = " + key);

		if (!scholixes.isEmpty()) {

			final Scholix source = getSource(context, scholixes);

			if (source == null) {
				context.getCounter("scholix", "missing source").increment(1);
				System.out.println(String.format("missing source in group of %s: %s", scholixes.size(), key.getId()));
				System.out.println(scholixes);
				return;
			}

			scholixes.forEach(t -> {
				if (!t.hasSource()) {
					final Scholix.Builder rel = Scholix.newBuilder(t);
					if (rel.hasTarget()) {
						rel.setSource(source.getSource());
						emit(context, JsonFormat.printToString(rel.build()));
					} else {
						context.getCounter("scholix", "missing target").increment(1);
					}
				}
			});
		}
	}

	private Scholix getSource(final Context context, final List<Scholix> scholixes) {
		for (int i = 0; i < scholixes.size(); i++) {
			final Scholix s = scholixes.get(i);
			if (s.hasSource()) {
				context.getCounter("scholix", "first source in position " + i).increment(1);
				context.getCounter("scholix", "group size " + scholixes.size()).increment(1);
				return s;
			}
		}
		return null;
	}

	private Scholix.Builder parse(DliKey key, final ImmutableBytesWritable value) {
		try {
			return Scholix.newBuilder(Scholix.parseFrom(value.copyBytes()));
		} catch (InvalidProtocolBufferException e) {
			throw new IllegalArgumentException(String.format("cannot parse Scholix, keytype '%s', id '%s'", key.getKeyType(), key.getId()));
		}
	}

	private void emit(final Context context, final String data) {
		outValue.set(data.getBytes());
		try {
			context.write(outKey, outValue);
		} catch (Exception e) {
			e.printStackTrace();
			throw new RuntimeException(e);
		}
	}

}
