package eu.dnetlib.data.mapreduce.hbase.oai.utils;

import java.text.Normalizer;
import java.util.List;

import org.apache.commons.lang3.StringEscapeUtils;
import org.apache.commons.lang3.StringUtils;

import com.google.common.collect.Lists;
import com.mongodb.BasicDBObject;
import com.mongodb.BasicDBObjectBuilder;
import com.mongodb.DB;
import com.mongodb.DBCollection;
import com.mongodb.DBCursor;
import com.mongodb.DBObject;

public class MongoSetCollection implements SetCollection {

	public static String DEFAULT_SET = "OTHER";
	private DB publisherDB;
	private String setCollection = "sets";
	private String setCountCollection = "setsCount";

	public MongoSetCollection(final DB mongoDB) {
		this.publisherDB = mongoDB;
	}

	public void ensureIndexes() {
		this.ensureIndexesOnSets();
		this.ensureIndexesOnCount();
	}

	@Override
	public List<SetInfo> getAllSets() {
		DBCursor cursor = this.getSetsCollection().find();
		List<SetInfo> res = Lists.newArrayList();
		while (cursor.hasNext()) {
			DBObject obj = cursor.next();
			res.add(this.getSetFromDBObject(obj));
		}
		return res;
	}

	@Override
	public boolean containSet(final String set) {
		DBObject query = new BasicDBObject("spec", set);
		return this.getSetsCollection().find(query).limit(1).size() != 0;
	}

	@Override
	public String getSetQuery(final String set) {
		DBObject query = new BasicDBObject("spec", set);
		DBObject returnField = new BasicDBObject("query", 1);
		DBObject obj = this.getSetsCollection().findOne(query, returnField);
		return (String) obj.get("query");
	}

	@Override
	public int count(final String setSpec, final String mdPrefix) {
		DBObject query = BasicDBObjectBuilder.start("spec", setSpec).add("mdPrefix", mdPrefix).get();
		DBObject returnField = new BasicDBObject("count", 1);
		DBObject obj = this.getSetsCountCollection().findOne(query, returnField);
		return (Integer) obj.get("count");
	}

	public void updateCounts(final String setSpec, final String mdPrefix, final int count) {
		DBObject setCount = BasicDBObjectBuilder.start("spec", setSpec).add("mdPrefix", mdPrefix).get();
		this.getSetsCountCollection().update(setCount, new BasicDBObject("$set", new BasicDBObject("count", count)), true, false);
	}

	public void upsertSet(final SetInfo setInfo, final boolean fromConfiguration) {
		DBObject obj = this.getObjectFromSet(setInfo);
		obj.put("fromConfiguration", fromConfiguration);
		this.getSetsCollection().update(new BasicDBObject("spec", setInfo.getSetSpec()), obj, true, false);
	}

	public String normalizeSetSpec(final String setName) {
		String s = StringEscapeUtils.unescapeXml(setName);
		s = Normalizer.normalize(s, Normalizer.Form.NFD);
		// replace spaces with underscores
		s = s.replaceAll(" ", "_");
		// remove tilde, dots... over letters
		s = s.replaceAll("[\\p{InCombiningDiacriticalMarks}&&[^-_]]", "");
		// change punctuation into an underscore
		s = s.replaceAll("[\\p{Punct}&&[^-_]]", "_");
		// remove all non-word characters
		s = s.replaceAll("[\\W&&[^-_]]", "");
		// Avoiding set '___' generated when we have "strange" set names such as those in cyrillic/ukrain
		// strips _ from the beginning and the end
		String stripped = StringUtils.strip(s, "_ ");
		if (StringUtils.isBlank(stripped)) {
			stripped = DEFAULT_SET;
		}
		return stripped;
	}

	public List<SetInfo> getConfiguredSets() {
		DBObject query = new BasicDBObject("fromConfiguration", true);
		return this.findSets(query);
	}

	public List<SetInfo> getSetsFromData() {
		DBObject query = new BasicDBObject("fromConfiguration", false);
		return this.findSets(query);
	}

	public void dropOAISets() {
		this.getSetsCountCollection().drop();
		this.getSetsCollection().drop();
	}

	public void dropConfigurationSets() {
		this.getSetsCollection().remove(new BasicDBObject("fromConfiguration", true));
	}

	protected List<SetInfo> findSets(final DBObject query) {
		DBCursor cursor = this.getSetsCollection().find(query);
		List<SetInfo> res = Lists.newArrayList();
		while (cursor.hasNext()) {
			DBObject obj = cursor.next();
			res.add(this.getSetFromDBObject(obj));
		}
		return res;
	}

	private SetInfo getSetFromDBObject(final DBObject obj) {
		SetInfo setInfo = new SetInfo();
		setInfo.setEnabled((Boolean) obj.get("enabled"));
		setInfo.setQuery((String) obj.get("query"));
		setInfo.setSetDescription((String) obj.get("description"));
		setInfo.setSetName((String) obj.get("name"));
		setInfo.setSetSpec((String) obj.get("spec"));
		return setInfo;
	}

	private DBObject getObjectFromSet(final SetInfo s) {
		DBObject obj = BasicDBObjectBuilder.start("spec", s.getSetSpec()).add("name", s.getSetName()).add("description", s.getSetDescription())
				.add("query", s.getQuery()).add("enabled", s.isEnabled()).get();
		return obj;
	}

	private void ensureIndexesOnSets() {
		this.getSetsCollection().createIndex(new BasicDBObject("spec", 1));
		this.getSetsCollection().createIndex(new BasicDBObject("fromConfiguration", 1));
	}

	private void ensureIndexesOnCount() {
		DBObject index = BasicDBObjectBuilder.start("spec", 1).add("mdPrefix", 1).get();
		this.getSetsCountCollection().createIndex(index);
	}

	public DBCollection getSetsCollection() {
		return this.getCollection(this.setCollection);
	}

	public DBCollection getSetsCountCollection() {
		return this.getCollection(this.setCountCollection);
	}

	private DBCollection getCollection(final String collectionName) {
		return this.publisherDB.getCollection(collectionName);
	}

	public DB getPublisherDB() {
		return publisherDB;
	}

	public void setPublisherDB(final DB publisherDB) {
		this.publisherDB = publisherDB;
	}

	public String getSetCollection() {
		return setCollection;
	}

	public void setSetCollection(final String setCollection) {
		this.setCollection = setCollection;
	}

	public String getSetCountCollection() {
		return setCountCollection;
	}

	public void setSetCountCollection(final String setCountCollection) {
		this.setCountCollection = setCountCollection;
	}

}
