package eu.dnetlib.data.mapreduce.util;
import com.google.common.base.Predicate;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.protobuf.Descriptors;
import com.google.protobuf.GeneratedMessage;
import eu.dnetlib.data.graph.model.DNGFDecoder;
import eu.dnetlib.data.graph.utils.RelDescriptor;
import eu.dnetlib.data.mapreduce.hbase.index.config.ContextMapper;
import eu.dnetlib.data.mapreduce.hbase.index.config.EntityConfigTable;
import eu.dnetlib.data.mapreduce.hbase.index.config.LinkDescriptor;
import eu.dnetlib.data.proto.DNGFProtos;
import eu.dnetlib.data.proto.FieldTypeProtos;
import eu.dnetlib.data.proto.PublicationProtos;
import eu.dnetlib.data.proto.TypeProtos;
import eu.dnetlib.data.transform.Ontologies;
import org.apache.commons.lang3.StringUtils;

import java.util.List;
import java.util.Map;
import java.util.Set;


public abstract class AbstractRecordFactory {

    protected final Ontologies ontologies;
    protected final Map<String, Integer> relCounters = Maps.newHashMap();
    protected Set<String> specialDatasourceTypes = Sets.newHashSet("scholarcomminfra", "infospace", "pubsrepository::mock", "entityregistry");
    protected DNGFDecoder mainEntity = null;
    protected String key = null;
    protected List<DNGFDecoder> relations = Lists.newLinkedList();
    protected List<DNGFDecoder> children = Lists.newLinkedList();
    protected EntityConfigTable entityConfigTable;
    protected ContextMapper contextMapper;

    protected Map<String, Integer> counters = Maps.newHashMap();
    protected boolean entityDefaults;
    protected boolean relDefaults;
    protected boolean childDefaults;
    protected Set<String> contextes = Sets.newHashSet();

    public AbstractRecordFactory(final EntityConfigTable entityConfigTable, final ContextMapper contextMapper, final Ontologies ontologies,
                                  final boolean entityDefaults, final boolean relDefaults, final boolean childDefaults) {
        this.entityConfigTable = entityConfigTable;
        this.contextMapper = contextMapper;
        this.entityDefaults = entityDefaults;
        this.relDefaults = relDefaults;
        this.childDefaults = childDefaults;
        this.ontologies = ontologies;
    }

    public static String removePrefix(final String s) {
        if (s.contains("|")) return StringUtils.substringAfter(s, "|");
        return s;

    }

    public Map<String, Integer> getRelCounters() {
        return relCounters;
    }

    public void setMainEntity(final DNGFDecoder mainEntity) {
        this.mainEntity = mainEntity;
        this.key = mainEntity.decodeEntity().getId();

    }

    public void addRelation(final TypeProtos.Type type, final DNGFDecoder rel) {
        addRelOrChild(type, relations, rel);
    }

    public void addChild(final TypeProtos.Type type, final DNGFDecoder child) {
        addRelOrChild(type, children, child);
    }

    private void addRelOrChild(final TypeProtos.Type type, final List<DNGFDecoder> list, final DNGFDecoder decoder) {

        final DNGFProtos.DNGFRel oafRel = decoder.getDNGFRel();
        final String relFamily = oafRel.getRelType().getSchemeid();
        final String relType = oafRel.getRelType().getClassid();
        final String relInverse = ontologies.get(relFamily).inverseOf(relType);

        final String rd = relFamily + "_" + relInverse;
        final LinkDescriptor ld = entityConfigTable.getDescriptor(type, new RelDescriptor(rd));

        getRelCounters().putIfAbsent(rd, 0);

        if (ld == null) {
            list.add(decoder);
            return;
        }

        if (ld.getMax() < 0) {
            list.add(decoder);
            return;
        }

        if (getRelCounters().get(rd) < ld.getMax()) {
            getRelCounters().put(rd, getRelCounters().get(rd) + 1);
            list.add(decoder);
        }
    }

    protected Set<Map.Entry<Descriptors.FieldDescriptor, Object>> filterFields(final GeneratedMessage fields, final Set<String> filter) {

        if (filter != null && filter.size()>0) {
            final Predicate<Descriptors.FieldDescriptor> p = descriptor -> {
                if (fields == null) return false;
                final String name = descriptor.getName();
                return filter.contains(name);
            };
            final Map<Descriptors.FieldDescriptor, Object> filtered = Maps.filterKeys(fields.getAllFields(), p);

            return filtered.entrySet();
        }
        return fields.getAllFields().entrySet();
    }

    public String getId() {
        return key;
    }

    public boolean isValid() {
        return mainEntity != null;
    }

    public FieldTypeProtos.Qualifier getQualifier(final String classid, final String classname, final String schemename) {
        return FieldTypeProtos.Qualifier.newBuilder().setClassid(classid).setClassname(classname).setSchemeid(schemename).setSchemename(schemename).build();
    }

    protected Object getDefault(final Descriptors.FieldDescriptor fd) {
        switch (fd.getType()) {
            case BOOL:
                return false;
            case BYTES:
                return "".getBytes();
            case MESSAGE: {
                if (FieldTypeProtos.Qualifier.getDescriptor().equals(fd.getMessageType())) return defaultQualifier();
                if (FieldTypeProtos.StructuredProperty.getDescriptor().equals(fd.getMessageType()))
                    return FieldTypeProtos.StructuredProperty.newBuilder().setValue("").setQualifier(defaultQualifier()).build();
                if (FieldTypeProtos.KeyValue.getDescriptor().equals(fd.getMessageType()))
                    return FieldTypeProtos.KeyValue.newBuilder().setKey("").setValue("").build();
                if (FieldTypeProtos.StringField.getDescriptor().equals(fd.getMessageType()))
                    return FieldTypeProtos.StringField.newBuilder().setValue("").build();
                if (FieldTypeProtos.BoolField.getDescriptor().equals(fd.getMessageType())) return FieldTypeProtos.BoolField.newBuilder().buildPartial();
                return null;
            }
            case SFIXED32:
            case SFIXED64:
            case SINT32:
            case SINT64:
            case INT32:
            case INT64:
            case DOUBLE:
            case FIXED32:
            case FIXED64:
            case FLOAT:
                return 0;
            case STRING:
                return "";
            default:
                return null;
        }
    }

    private FieldTypeProtos.Qualifier defaultQualifier() {
        return FieldTypeProtos.Qualifier.newBuilder().setClassid("").setClassname("").setSchemeid("").setSchemename("").build();
    }

    protected void incrementCounter(final String type) {
        if (!counters.containsKey(type)) {
            counters.put(type, 1);
        } else {
            counters.put(type, counters.get(type) + 1);
        }
    }

    protected FieldTypeProtos.Qualifier getBestLicense() {
        FieldTypeProtos.Qualifier bestLicense = getQualifier("UNKNOWN", "not available", "dnet:access_modes");
        final LicenseComparator lc = new LicenseComparator();
        for (final FieldTypeProtos.Instance instance : ((PublicationProtos.Publication) mainEntity.decodeEntity().getEntity()).getInstanceList()) {
            if (lc.compare(bestLicense, instance.getLicence()) > 0) {
                bestLicense = instance.getLicence();
            }
        }
        return bestLicense;
    }

    protected void manageInferred(String rd, FieldTypeProtos.DataInfo info) {
        if (info.getInferred()) {
            incrementCounter(rd + "_inferred");
        } else if (StringUtils.startsWith(info.getProvenanceaction().getClassid(), "sysimport:crosswalk")) {
            incrementCounter(rd + "_collected");
        } else if (StringUtils.startsWith(info.getProvenanceaction().getClassid(), "user:")) {
            incrementCounter(rd + "_claimed");
        }
    }

    public abstract String build();

}
