package eu.dnetlib.data.mapreduce.hbase.dli;

import java.io.IOException;

import eu.dnetlib.data.graph.model.DNGFRowKeyDecoder;
import eu.dnetlib.data.mapreduce.JobParams;
import eu.dnetlib.data.proto.DNGFProtos.DNGF;
import eu.dnetlib.data.proto.KindProtos.Kind;
import eu.dnetlib.data.proto.TypeProtos.Type;
import eu.dnetlib.data.transform.Ontologies;
import eu.dnetlib.data.transform.OntologyLoader;
import eu.dnetlib.dli.proto.DNGFDLISummaryConverter;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.io.ImmutableBytesWritable;
import org.apache.hadoop.hbase.mapreduce.TableMapper;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.mapreduce.Counter;

import static eu.dnetlib.data.mapreduce.util.dao.HBaseTableDAO.getMetadata;
import static eu.dnetlib.data.mapreduce.util.dao.HBaseTableDAO.rel;

public class PrepareSummaryDataMapper extends TableMapper<Text, Text> {

    private final static String SCHOLIX = "Scholix";

    private Ontologies ontologies = null;

    private Text outKey;

    private Text outValue;

    @Override
    protected void setup(final Context context) throws IOException, InterruptedException {

        ontologies = OntologyLoader.loadOntologies(context.getConfiguration().get(JobParams.ONTOLOGIES));

        outKey = new Text();
        outValue = new Text();
    }

    @Override
    protected void map(final ImmutableBytesWritable keyIn, final Result value, final Context context) throws IOException, InterruptedException {

        final DNGFRowKeyDecoder keyDecoder = DNGFRowKeyDecoder.decode(keyIn.copyBytes());

        final Type type = keyDecoder.getType();
        final DNGF entity = getMetadata(value, type);

        if (isValid(entity)) {

            if (!deletedByInference(entity)) {

                final DNGFDLISummaryConverter converter = new DNGFDLISummaryConverter();

                converter.setMainDNGFEntity(entity);

                incrementCounter(context, Kind.entity.toString(), type.toString(), 1);

                rel(value, "isMergedIn", "merges", "isSimilarTo").forEach((qualifier, r) -> {
                    if (!r.getDataInfo().getDeletedbyinference()) {
                        converter.addRelation(r);
                    }
                });
                if (converter.isValid()) {
                    emit(new String(keyIn.copyBytes()), context, converter.convertAsJson());
                }
            } else {
                incrementCounter(context, SCHOLIX, "Deletedbyinference: " + type.toString(), 1);
            }
        } else {
            incrementCounter(context, SCHOLIX, "Missing body: " + type.toString(), 1);
        }
    }

    private boolean isValid(final DNGF entity) {
        return (entity != null) && entity.isInitialized();
    }

    private boolean deletedByInference(final DNGF oaf) {
        return oaf.getDataInfo().getDeletedbyinference();
    }

    private void emit(final String key, final Context context, final String data) {
        if (data == null)
            return;
        outKey.set(key);
        outValue.set(data.getBytes());
        try {
            context.write(outKey, outValue);
        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        }
    }

    private void incrementCounter(final Context context, final String k, final String t, final int n) {
        getCounter(context, k, t).increment(n);
    }

    private Counter getCounter(final Context context, final String k, final String t) {
        return context.getCounter(k, t);
    }

}
