package eu.dnetlib.data.mapreduce.hbase.index;

import java.io.IOException;
import java.util.Map;
import java.util.Map.Entry;

import com.google.protobuf.InvalidProtocolBufferException;
import eu.dnetlib.data.graph.model.DNGFRowKeyDecoder;
import eu.dnetlib.data.graph.utils.RelDescriptor;
import eu.dnetlib.data.mapreduce.JobParams;
import eu.dnetlib.data.mapreduce.hbase.index.config.EntityConfigTable;
import eu.dnetlib.data.mapreduce.hbase.index.config.IndexConfig;
import eu.dnetlib.data.mapreduce.hbase.index.config.LinkDescriptor;
import eu.dnetlib.data.mapreduce.util.UpdateMerger;
import eu.dnetlib.data.mapreduce.util.dao.HBaseTableDAO;
import eu.dnetlib.data.proto.DNGFProtos.DNGF;
import eu.dnetlib.data.proto.DNGFProtos.DNGFEntity;
import eu.dnetlib.data.proto.DNGFProtos.DNGFRel;
import eu.dnetlib.data.proto.DNGFProtos.DNGFRelOrBuilder;
import eu.dnetlib.data.proto.KindProtos.Kind;
import eu.dnetlib.data.proto.TypeProtos.Type;
import eu.dnetlib.data.transform.Ontologies;
import eu.dnetlib.data.transform.OntologyLoader;
import eu.dnetlib.miscutils.functional.xml.XMLIndenter;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.io.ImmutableBytesWritable;
import org.apache.hadoop.hbase.mapreduce.TableMapper;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.mapreduce.Counter;

public class PrepareFeedMapper extends TableMapper<Text, ImmutableBytesWritable> {

	private EntityConfigTable entityConfigTable;

	private Ontologies ontologies = null;

	private Text outKey;

	private ImmutableBytesWritable ibw;

	@Override
	protected void setup(final Context context) throws IOException, InterruptedException {
		final String json = context.getConfiguration().get(JobParams.INDEX_ENTITY_LINKS);
		System.out.println(JobParams.INDEX_ENTITY_LINKS + ":\n" + json);
		entityConfigTable = IndexConfig.load(json).getConfigMap();

		final String contextMap = context.getConfiguration().get("contextmap");
		System.out.println("contextmap:\n" + XMLIndenter.indent(contextMap));
		ontologies = OntologyLoader.loadOntologies(context.getConfiguration().get(JobParams.ONTOLOGIES));

		outKey = new Text();
		ibw = new ImmutableBytesWritable();
	}

	@Override
	protected void map(final ImmutableBytesWritable keyIn, final Result value, final Context context) throws IOException, InterruptedException {

		final DNGFRowKeyDecoder keyDecoder = DNGFRowKeyDecoder.decode(keyIn.copyBytes());

		final Type type = keyDecoder.getType();
		final DNGF oaf = mergeUpdates(value, context, type, keyDecoder);

		if (isValid(oaf)) {

			if (!deletedByInference(oaf) || entityConfigTable.includeDuplicates(type)) {
				emit(new String(keyIn.copyBytes()), context, oaf);

				incrementCounter(context, Kind.entity.toString(), type.toString(), 1);

				for (final LinkDescriptor ld : entityConfigTable.getDescriptors(type)) {

					final Map<byte[], byte[]> columnMap = value.getFamilyMap(Bytes.toBytes(ld.getRelDescriptor().getCode()));

					if (hasData(columnMap)) {
						emitRelationship(oaf.getEntity(), context, columnMap, ld);
						incrementCounter(context, type.toString(), ld.getRelDescriptor().getCode(), columnMap.size());
					} // else {
						// incrementCounter(context, type.toString(), ld.getRelDescriptor().getIt() + "_empty", 1);
						// }
				}
			} else {
				incrementCounter(context, "deleted by inference", type.toString(), 1);
			}
		} else {
			incrementCounter(context, "missing body (map)", type.toString(), 1);
		}
	}

	private DNGF mergeUpdates(final Result value, final Context context, final Type type, final DNGFRowKeyDecoder keyDecoder)
			throws InvalidProtocolBufferException {
		try {
			return UpdateMerger.mergeBodyUpdates(context, value.getFamilyMap(HBaseTableDAO.cfMetadataByte()), type);
		} catch (final InvalidProtocolBufferException e) {
			System.err.println(String.format("Unable to parse proto (Type: %s) in row: %s", type.toString(), keyDecoder.getKey()));
			throw e;
		}
	}

	private void emitRelationship(final DNGFEntity cachedTarget, final Context context, final Map<byte[], byte[]> columnMap, final LinkDescriptor ld)
			throws IOException, InterruptedException {

		final DNGF.Builder oafBuilder = DNGF.newBuilder().setKind(Kind.relation);

		// iterates the column map
		for (final Entry<byte[], byte[]> e : columnMap.entrySet()) {

			final DNGF oaf = decodeProto(context, e.getValue());
			if (!isValid(oaf)) {
				incrementCounter(context, "invalid oaf rel", ld.getRelDescriptor().getCode(), 1);
			} else if (!deletedByInference(oaf)) {

				final DNGFRel.Builder relBuilder = DNGFRel.newBuilder(oaf.getRel());

				if (ld.isSymmetric()) {
					final RelDescriptor rd = ld.getRelDescriptor();
					relBuilder.setCachedTarget(cachedTarget).setRelType(rd.asQualifier());
				}

				if (ld.getRelDescriptor().getCode().contains("dedup") && isDedupSelf(relBuilder)) {
					incrementCounter(context, "avoid to emit dedup self", ld.getRelDescriptor().getCode(), 1);
					continue;
				}

				final DNGFRel oafRel = relBuilder.setChild(ld.isChild()).build();

				// String rowKey = patchTargetId(target, DNGFRelDecoder.decode(oafRel).getRelTargetId());

				emit(ld.isSymmetric() ? oafRel.getTarget() : oafRel.getSource(), context, merge(oafBuilder, oaf).setRel(oafRel).build());
			} else {
				incrementCounter(context, "deleted by inference", ld.getRelDescriptor().getCode(), 1);
			}
		}
	}

	private String patchTargetId(final Type target, final String id) {
		return id.replaceFirst("^.*\\|", target.getNumber() + "|");
	}

	private DNGF.Builder merge(final DNGF.Builder builder, final DNGF prototype) {
		return builder.setDataInfo(prototype.getDataInfo()).setLastupdatetimestamp(prototype.getLastupdatetimestamp());
	}

	private boolean isDedupSelf(final DNGFRelOrBuilder rel) {
		return rel.getSource().contains(rel.getTarget());
	}

	private boolean hasData(final Map<byte[], byte[]> columnMap) {
		return (columnMap != null) && !columnMap.isEmpty();
	}

	private boolean isValid(final DNGF oaf) {
		return (oaf != null) && oaf.isInitialized();
	}

	private boolean deletedByInference(final DNGF oaf) {
		return oaf.getDataInfo().getDeletedbyinference();
	}

	private DNGF decodeProto(final Context context, final byte[] body) {
		try {
			return DNGF.parseFrom(body);
		} catch (final InvalidProtocolBufferException e) {
			e.printStackTrace(System.err);
			context.getCounter("decodeProto", e.getClass().getName()).increment(1);
		}
		return null;
	}

	// private byte[] prefix(final Type name, final byte[] bytes) {
	// return concat(concat(name.toString().getBytes(), PrepareFeedJob.bSEPARATOR), bytes);
	// }

	private void emit(final String key, final Context context, final DNGF oaf) throws IOException, InterruptedException {
		// Text keyOut = new Text(Hashing.murmur3_128().hashString(key).toString());
		outKey.set(key);
		ibw.set(oaf.toByteArray());

		context.write(outKey, ibw);
	}

	// protected byte[] concat(final byte[] a, final byte[] b) {
	// byte[] c = new byte[a.length + b.length];
	// System.arraycopy(a, 0, c, 0, a.length);
	// System.arraycopy(b, 0, c, a.length, b.length);
	//
	// return c;
	// }

	private void incrementCounter(final Context context, final String k, final String t, final int n) {
		getCounter(context, k, t).increment(n);
	}

	private Counter getCounter(final Context context, final String k, final String t) {
		return context.getCounter(k, t);
	}

}
