package eu.dnetlib.functionality.modular.ui.oai;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringWriter;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;

import com.google.common.collect.Lists;
import eu.dnetlib.functionality.modular.ui.oai.objects.OaiRequest;
import eu.dnetlib.functionality.modular.ui.oai.objects.ResponseDetails;
import eu.dnetlib.miscutils.datetime.DateUtils;
import eu.dnetlib.rmi.data.CollectorServiceException;
import org.apache.commons.lang3.math.NumberUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.http.HttpStatus;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.message.BasicHeader;
import org.dom4j.Document;
import org.dom4j.Node;
import org.dom4j.io.SAXReader;
import org.springframework.core.io.ClassPathResource;
import org.springframework.core.io.Resource;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseBody;

@Controller
public class OaiExplorerInternalController {

	private final static Resource oaiXslt = new ClassPathResource("/eu/dnetlib/functionality/modular/ui/oai/xslt/oai.xslt");

	private static final Log log = LogFactory.getLog(OaiExplorerInternalController.class);

	@RequestMapping("/ui/oai_verb")
	public
	@ResponseBody
	String oaiVerb(@RequestBody(required = true) OaiRequest req) throws Exception {
		return applyXslt(callOaiVerb(req));
	}

	@RequestMapping("/ui/test_oai_verb")
	public
	@ResponseBody
	ResponseDetails testOaiVerb(@RequestBody final OaiRequest req) throws Exception {
		final ResponseDetails response = new ResponseDetails();

		final Document doc = callOaiVerb(req, response);

		if (response.isValid() && doc != null) {
			final OaiRequest nextCall = new OaiRequest();
			nextCall.setBaseUrl(req.getBaseUrl());

			if ("Identify".equals(req.getVerb())) {
				nextCall.setVerb("ListSets");
			} else if ("ListSets".equals(req.getVerb())) {
				nextCall.setVerb("ListMetadataFormats");
			} else if ("ListMetadataFormats".equals(req.getVerb())) {
				nextCall.setVerb("ListRecords");
				if (doc.selectSingleNode("//*[local-name()='metadataPrefix' and text()='oai_dc']") != null) {
					nextCall.setMdf("oai_dc");
				} else {
					nextCall.setMdf(doc.selectSingleNode("//*[local-name()='metadataPrefix']").getText());
				}
			} else if ("ListRecords".equals(req.getVerb())) {
				nextCall.setVerb("ListIdentifiers");
				nextCall.setMdf(req.getMdf());
			} else if ("ListIdentifiers".equals(req.getVerb())) {
				nextCall.setVerb("GetRecord");
				nextCall.setMdf(req.getMdf());
				nextCall.setId(doc.selectSingleNode("//*[local-name()='identifier']").getText());
			} else if ("GetRecord".equals(req.getVerb())) {
				// NOTHING
			}
			response.setNextCall(nextCall);
		}
		return response;
	}

	@RequestMapping("/ui/test_harvesting")
	public
	@ResponseBody
	ResponseDetails testHarvesting(@RequestBody final OaiRequest req) throws Exception {
		final ResponseDetails response = new ResponseDetails();

		final Document doc = callOaiVerb(req, response);

		if (response.isValid() && doc != null) {

			final Node node = doc.selectSingleNode("//*[local-name() = 'resumptionToken']");

			if (node != null) {
				response.setSize(doc.selectNodes("//*[local-name()='" + req.getVerb() + "']/*[local-name() != 'resumptionToken']").size());
				response.setCursor(NumberUtils.toInt(node.valueOf("@cursor"), -1));
				response.setTotal(NumberUtils.toInt(node.valueOf("@completeListSize"), -1));

				final OaiRequest nextCall = new OaiRequest();
				nextCall.setBaseUrl(req.getBaseUrl());
				nextCall.setVerb(req.getVerb());
				nextCall.setToken(node.getText());
				response.setNextCall(nextCall);
			}
		}
		return response;
	}

	private CloseableHttpResponse initClient(final OaiRequest req) throws Exception{
		final RequestConfig requestConfig = RequestConfig.custom()
				.setConnectTimeout(360000)
				.setConnectionRequestTimeout(360000)
				.setSocketTimeout(360000)
				.build();

		HttpClientBuilder httpClientBuilder = HttpClients.custom().setDefaultRequestConfig(requestConfig);

		final CloseableHttpClient httpClient = httpClientBuilder.build();

		final URIBuilder builder = new URIBuilder(req.getBaseUrl());
		builder.addParameters(Lists.newArrayList(req.toQueryParams()));
		final HttpGet method = new HttpGet(builder.build());
		method.setHeader(new BasicHeader("Content-type", "text/xml; charset=UTF-8"));

		final CloseableHttpResponse response = httpClient.execute(method);


		return response;
	}

	private InputStream callOaiVerb(final OaiRequest req) throws Exception {
		try {


			 final CloseableHttpResponse response =initClient(req);
				int statusCode = response.getStatusLine().getStatusCode();

				if (HttpStatus.SC_OK != statusCode) {
					throw new RuntimeException("Error " + statusCode + " from  url: " + req.getBaseUrl());
				}

				return new BufferedInputStream(response.getEntity().getContent());

		} catch (IOException e) {
			throw new CollectorServiceException("Error requesting url: " + req.getBaseUrl());
		}

	}

	private Document callOaiVerb(final OaiRequest req, final ResponseDetails details) {
		final long start = DateUtils.now();
		Document doc = null;

		try {

			final CloseableHttpResponse response = initClient(req);
				int statusCode = response.getStatusLine().getStatusCode();

				if (HttpStatus.SC_OK != statusCode) {
					throw new Exception("Error " + statusCode + " from  url: " + req.getBaseUrl());
				}
				details.setHttpCode(statusCode);
				details.setValid(HttpStatus.SC_OK == statusCode);

				doc = new SAXReader().read(response.getEntity().getContent());

		} catch (Exception e) {
			details.setValid(false);
			details.setError(e.getMessage());

		}
		details.setTime(DateUtils.now() - start);
		details.setVerb(req.getVerb());
		return doc;
	}

	private String applyXslt(final InputStream input) throws Exception {
		final TransformerFactory tfactory = TransformerFactory.newInstance();

		final Transformer transformer = tfactory.newTransformer(new StreamSource(oaiXslt.getInputStream()));
		final StringWriter output = new StringWriter();
		transformer.transform(new StreamSource(input), new StreamResult(output));

		return output.toString();
	}
}
