package eu.dnetlib.functionality.modular.ui.oai;

import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.lang.reflect.Type;
import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Lists;
import com.google.common.collect.Multimap;
import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;
import eu.dnetlib.oai.PublisherField;
import eu.dnetlib.oai.conf.OAIConfigurationExistReader;
import eu.dnetlib.oai.conf.OAIConfigurationWriter;
import eu.dnetlib.oai.info.SetInfo;
import eu.dnetlib.rmi.enabling.ISLookUpException;
import eu.dnetlib.rmi.provision.MDFInfo;
import eu.dnetlib.utils.MetadataReference;
import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

@Controller
public class OAIInternalController {

	private static final Log log = LogFactory.getLog(OAIInternalController.class); // NOPMD by marko on 11/24/08 5:02 PM

	@Autowired
	private OAIConfigurationExistReader configuration;
	@Autowired
	private OAIConfigurationWriter configurationWriter;

	@RequestMapping(value = "/ui/getExportMDFs.do")
	public void getExportMDFs(final HttpServletResponse response) throws Exception {
		List<MDFInfo> exportMDFs = configuration.getMetadataFormatInfo();
		String json = new Gson().toJson(exportMDFs);
		IOUtils.copy(new StringReader(json), response.getOutputStream());
	}

	@RequestMapping(value = "/ui/getSourceMDFs.do")
	public
	@ResponseBody
	List<MetadataReference> getSourceMDFs(final HttpServletResponse response) throws Exception {

		return configuration.getMetadataFormatInfo().stream()
				.map(mdf -> mdf.getSourceMetadataReference())
				.distinct()
				.collect(Collectors.toList());

	}

	@RequestMapping(value = "/ui/getSetSpecs.do")
	public
	@ResponseBody
	List<String> getSetSpecs(final HttpServletResponse response) throws Exception {
		return configuration.getSetSpecs();
	}

	@RequestMapping(value = "/ui/getMetadataFormat.do")
	public
	@ResponseBody
	MDFInfo getMetadataFormat(final HttpServletResponse response, @RequestParam(value = "mdPrefix", required = true) final String mdPrefix)
			throws Exception {
		return this.configuration.getMetadataFormatInfo(mdPrefix);

	}

	@RequestMapping(value = "/ui/saveMetadataFormat.do", method = RequestMethod.POST)
	public void saveMetadataFormat(final HttpServletResponse response,
			final HttpServletRequest request,
			@RequestParam(value = "mdPrefix", required = true) final String mdPrefix) throws Exception {

		MDFInfo update = new Gson().fromJson(request.getReader(), MDFInfo.class);
		if (update.getBaseQuery() == null) {
			update.setBaseQuery("");
		}
		if (update.getTransformationRuleID() == null) {
			update.setTransformationRuleID("");
		}
		log.debug("Update to object" + update);
		boolean result = false;
		if (this.configuration.getMetadataFormatInfo(mdPrefix) != null) {
			log.debug("UPDATING mdPrefix = " + mdPrefix);
			result = this.configurationWriter.updateMetadataFormat(mdPrefix, update);
		} else {
			// ADD NEW mdf
			log.debug("CREATING new mdPrefix = " + mdPrefix);
			result = this.configurationWriter.addMetadataFormat(update);
		}
		response.getWriter().print(result);
	}

	@RequestMapping(value = "/ui/deleteMetadataFormat.do")
	public void deleteMetadataFormat(final HttpServletResponse response, @RequestParam(value = "mdPrefix", required = true) final String mdPrefix)
			throws Exception {
		boolean result = this.configurationWriter.deleteMetadataFormat(mdPrefix);
		response.getWriter().print(result);
	}

	@RequestMapping(value = "/ui/showSetDetails.do")
	public void showSetDetails(final HttpServletResponse response, @RequestParam(value = "setSpec", required = true) final String setSpec) throws Exception {
		SetInfo set = this.configuration.getSetInfo(setSpec);
		String json = new Gson().toJson(set);
		IOUtils.copy(new StringReader(json), response.getOutputStream());
	}

	@RequestMapping(value = "/ui/saveOAISet.do", method = RequestMethod.POST)
	public void saveOAISet(final HttpServletResponse response,
			final HttpServletRequest request,
			@RequestParam(value = "setSpec", required = true) final String setSpec) throws Exception {

		log.debug("setSpec = " + setSpec);
		SetInfo update = new Gson().fromJson(request.getReader(), SetInfo.class);
		log.debug("Update to object" + update);
		boolean result = false;
		if (this.configuration.getSetInfo(setSpec) != null) {
			log.debug("UPDATING set = " + setSpec);
			result = this.configurationWriter.updateOAISet(setSpec, update);
		} else {
			// ADD NEW mdf
			log.debug("CREATING new OAIset = " + setSpec);
			result = this.configurationWriter.addOAISet(update);
		}
		response.getWriter().print(result);
	}

	@RequestMapping(value = "/ui/deleteOAISet.do")
	public void deleteOAISet(final HttpServletResponse response, @RequestParam(value = "setSpec", required = true) final String setSpec) throws Exception {
		boolean result = this.configurationWriter.deleteOAISet(setSpec);
		response.getWriter().print(result);
	}

	@RequestMapping(value = "/ui/showIndices.do")
	public void showIndices(final HttpServletResponse response,
			@RequestParam(value = "format", required = true) final String format,
			@RequestParam(value = "layout", required = true) final String layout,
			@RequestParam(value = "interpretation", required = true) final String interpretation) throws Exception {

		List<PublisherField> fields = this.configuration.getFields(format, interpretation, layout);

		// Need to iterate over the values to build an object otherwise angularjs does not allow me to edit the content!
		// See http://stackoverflow.com/questions/13714884/difficulty-with-ng-model-ng-repeat-and-inputs
		List<OAIIndex> indices = Lists.newArrayList();
		for (PublisherField field : fields) {
			Collection<String> paths = field.getSources().get(format + "-" + layout + "-" + interpretation);
			List<IndexPath> pathList = Lists.newArrayList();
			for (String p : paths) {
				pathList.add(new IndexPath(p));
			}
			if(pathList.size()>0) {
				OAIIndex idx = new OAIIndex();
				idx.setRepeatable(field.isRepeatable());
				idx.setName(field.getFieldName());
				idx.setPaths(pathList);
				indices.add(idx);
			}
		}
		String json = new Gson().toJson(indices);
		log.debug("The map of indices: " + json);
		IOUtils.copy(new StringReader(json), response.getOutputStream());
	}

	@RequestMapping(value = "/ui/saveIndices.do", method = RequestMethod.POST)
	public void saveIndices(final HttpServletResponse response,
			final HttpServletRequest request,
			@RequestParam(value = "format", required = true) final String format,
			@RequestParam(value = "layout", required = true) final String layout,
			@RequestParam(value = "interpretation", required = true) final String interpretation) throws Exception {

		StringWriter sw = new StringWriter();
		IOUtils.copy(request.getInputStream(), sw);
		Type collectionType = new TypeToken<List<OAIIndex>>() {
		}.getType();
		List<OAIIndex> indexes = new Gson().fromJson(sw.toString(), collectionType);
		final String mdformatKey = format + "-" + layout + "-" + interpretation;
		List<PublisherField> updatedIndices = indexes.stream().map(oaiIndex -> {
			PublisherField f = new PublisherField();
			f.setFieldName(oaiIndex.getName());
			f.setRepeatable(oaiIndex.isRepeatable());
			Multimap<String, String> paths = ArrayListMultimap.create();
			for (IndexPath path : oaiIndex.getPaths()) {
				paths.put(mdformatKey, path.getPath());
			}
			f.setSources(paths);
			return f;
		}).collect(Collectors.toList());

		log.debug("Updating indices" + indexes);
		boolean result = this.configurationWriter.updateIndices(format, layout, interpretation, updatedIndices);
		response.getWriter().print(result);
	}

	@RequestMapping(value = "/ui/saveNewIndex.do")
	public void saveNewIndex(final HttpServletResponse response,
			@RequestParam(value = "format", required = true) final String format,
			@RequestParam(value = "layout", required = true) final String layout,
			@RequestParam(value = "interpretation", required = true) final String interpretation,
			@RequestParam(value = "indexName", required = true) final String indexName,
			@RequestParam(value = "repeatable", required = true) final boolean repeatable,
			@RequestParam(value = "paths", required = true) final String paths) throws Exception {

		String[] thePaths = paths.split(",");
		boolean result = this.configurationWriter.addNewIndex(format, layout, interpretation, indexName, repeatable, thePaths);
		response.getWriter().print(result);
	}

	@RequestMapping(value = "/ui/saveNewPathIndex.do")
	public void saveNewPathIndex(final HttpServletResponse response,
			@RequestParam(value = "format", required = true) final String format,
			@RequestParam(value = "layout", required = true) final String layout,
			@RequestParam(value = "interpretation", required = true) final String interpretation,
			@RequestParam(value = "indexName", required = true) final String indexName,
			@RequestParam(value = "newPath", required = true) final String newPath) throws Exception {

		List<PublisherField> fields = this.configurationWriter.getConfiguration().getFields(format, interpretation, layout).stream().map(field -> {
			if (field.getFieldName().equals(indexName)) {
				field.getSources().get(format + "-" + layout + "-" + interpretation).add(newPath);
			}
			return field;
		}).collect(Collectors.toList());
		boolean result = this.configurationWriter.updateIndices(format, layout, interpretation, fields);
		response.getWriter().print(result);
	}

	@RequestMapping(value = "/ui/getTransformationRules.do")
	public
	@ResponseBody
	List<TDSRule> getTransformationRules(final HttpServletResponse response) throws IOException, ISLookUpException {

		final String query = "for $x in collection('/db/DRIVER/TransformationRuleDSResources/TransformationRuleDSResourceType')"
				+ " return concat('id:-:',$x//RESOURCE_IDENTIFIER/@value/string(), ':-:title:-:', $x//CONFIGURATION//TITLE)";
		return this.configuration.getLookupClient().search(query, TDSRule.class, ":-:");
	}

}
