package eu.dnetlib.functionality.modular.ui.vocabularies.persistence;

import java.io.IOException;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import eu.dnetlib.functionality.modular.ui.vocabularies.model.Relation;
import eu.dnetlib.functionality.modular.ui.vocabularies.model.Synonym;
import eu.dnetlib.functionality.modular.ui.vocabularies.model.Term;
import eu.dnetlib.functionality.modular.ui.vocabularies.model.Vocabulary;
import eu.dnetlib.miscutils.datetime.DateUtils;
import eu.dnetlib.rmi.enabling.*;
import org.antlr.stringtemplate.StringTemplate;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringEscapeUtils;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;

public class RegistryServiceVocabularyDAO extends VocabularyDAO {

	@Override
	public List<Vocabulary> getVocabularies() throws VocabularyException {
		try {
			final String query =
					"for $x in collection('/db/DRIVER/VocabularyDSResources/VocabularyDSResourceType') order by $x//VOCABULARY_NAME "
							+ "return concat ($x//RESOURCE_IDENTIFIER/@value,' §§§ ',$x//VOCABULARY_NAME,' §§§ ',$x//VOCABULARY_DESCRIPTION,' §§§ ',$x//VOCABULARY_NAME/@code)";

			final List<Vocabulary> vocabularies = Lists.transform(this.serviceLocator.getService(ISLookUpService.class).quickSearchProfile(query),
					s -> {
						final String[] tokens = s.split("§§§");
						return new Vocabulary(tokens[0].trim(), tokens[1].trim(), tokens[2].trim(), tokens[3].trim());
					});
			return vocabularies;
		} catch (final ISLookUpException e) {
			throw new VocabularyException(e);
		}
	}

	@Override
	public List<Term> getTerms(final String vocabularyId) throws VocabularyException {
		try {
			final String profile = this.serviceLocator.getService(ISLookUpService.class).getResourceProfile(vocabularyId);
			final Document doc = new SAXReader().read(new StringReader(profile));
			final Map<String, Term> terms = Maps.newHashMap();

			for (final Object t : doc.selectNodes("//TERM")) {
				final Element termNode = (Element) t;
				final String code = termNode.valueOf("@code");

				if (!terms.containsKey(code)) {
					final Term term = new Term();
					term.setEnglishName(termNode.valueOf("@english_name"));
					term.setNativeName(termNode.valueOf("@native_name"));
					term.setEncoding(termNode.valueOf("@encoding"));
					term.setCode(code);
					term.setSynonyms(new ArrayList<Synonym>());
					term.setRelations(new ArrayList<Relation>());
					terms.put(code, term);
				}
				final Term term = terms.get(code);

				for (final Object s : termNode.selectNodes(".//SYNONYM")) {
					final Element synNode = (Element) s;
					final Synonym syn = new Synonym();
					syn.setTerm(synNode.valueOf("@term"));
					syn.setEncoding(synNode.valueOf("@encoding"));
					term.getSynonyms().add(syn);
				}

				for (final Object r : termNode.selectNodes(".//RELATION")) {
					final Element relNode = (Element) r;
					final Relation rel = new Relation();
					rel.setCode(relNode.valueOf("@code"));
					rel.setType(relNode.valueOf("@type"));
					term.getRelations().add(rel);
				}

				Collections.sort(term.getSynonyms());
				Collections.sort(term.getRelations());
			}

			final List<Term> list = Lists.newArrayList(terms.values());
			Collections.sort(list);

			return list;
		} catch (final ISLookUpDocumentNotFoundException e) {
			throw new VocabularyException(e);
		} catch (final ISLookUpException e) {
			throw new VocabularyException(e);
		} catch (final DocumentException e) {
			throw new VocabularyException(e);
		}
	}

	@Override
	public void commitTerms(final List<Term> terms, final String vocabularyId) throws VocabularyException {
		try {
			// prepare terms for XML
			for (final Term t : terms) {
				t.setCode(StringEscapeUtils.escapeXml11(t.getCode()));
				t.setEncoding(StringEscapeUtils.escapeXml11(t.getEncoding()));
				t.setEnglishName(StringEscapeUtils.escapeXml11(t.getEnglishName()));
				t.setNativeName(StringEscapeUtils.escapeXml11(t.getNativeName()));
				for (final Synonym s : t.getSynonyms()) {
					s.setEncoding(StringEscapeUtils.escapeXml11(s.getEncoding()));
					s.setTerm(StringEscapeUtils.escapeXml11(s.getTerm()));
				}
				for (final Relation r : t.getRelations()) {
					r.setType(StringEscapeUtils.escapeXml11(r.getType()));
					r.setCode(StringEscapeUtils.escapeXml11(r.getCode()));
				}
			}

			final StringTemplate st =
					new StringTemplate(IOUtils.toString(getClass().getResourceAsStream("/eu/dnetlib/functionality/modular/templates/terms.xml.st")));
			st.setAttribute("terms", terms);
			this.serviceLocator.getService(ISRegistryService.class).updateProfileNode(vocabularyId, "//TERMS", st.toString());
		} catch (final IOException e) {
			throw new VocabularyException(e);
		} catch (final ISRegistryException e) {
			throw new VocabularyException(e);
		}

	}

	@Override
	public void commitVocabularyInfo(final Vocabulary voc, final String vocabularyId) throws VocabularyException {
		try {
			String xml = "<VOCABULARY_DESCRIPTION>{desc}</VOCABULARY_DESCRIPTION>";
			xml = xml.replace("{desc}", StringEscapeUtils.escapeXml11(voc.getDescription()));
			this.serviceLocator.getService(ISRegistryService.class).updateProfileNode(vocabularyId, "//VOCABULARY_DESCRIPTION", xml);
		} catch (final ISRegistryException e) {
			throw new VocabularyException(e);
		}
	}

	@Override
	public String createVocabulary(final Vocabulary voc) throws VocabularyException {
		try {
			final StringTemplate st = new StringTemplate(
					IOUtils.toString(getClass().getResourceAsStream("/eu/dnetlib/functionality/modular/templates/vocabulary.xml.st")));
			st.setAttribute("name", voc.getName());
			st.setAttribute("description", voc.getDescription());
			st.setAttribute("code", voc.getCode());
			st.setAttribute("date", DateUtils.now_ISO8601());
			final String newVocabularyId = this.serviceLocator.getService(ISRegistryService.class).registerProfile(st.toString());

			return newVocabularyId;
		} catch (final IOException e) {
			throw new VocabularyException(e);
		} catch (final ISRegistryException e) {
			throw new VocabularyException(e);
		}
	}

	@Override
	public void dropVocabulary(final String vocabularyId) throws VocabularyException {
		try {
			this.serviceLocator.getService(ISRegistryService.class).deleteProfile(vocabularyId);
		} catch (final ISRegistryDocumentNotFoundException e) {
			throw new VocabularyException(e);
		} catch (final ISRegistryException e) {
			throw new VocabularyException(e);
		}
	}

}
