package eu.dnetlib.functionality.modular.ui.workflows.util;

import java.io.StringReader;
import java.util.*;
import java.util.stream.Collectors;

import com.google.common.base.Function;
import com.google.common.base.Splitter;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import eu.dnetlib.enabling.locators.UniqueServiceLocator;
import eu.dnetlib.functionality.modular.ui.repositories.objects.RepoHIWorkflow;
import eu.dnetlib.functionality.modular.ui.workflows.objects.WorkflowItem;
import eu.dnetlib.miscutils.collections.Pair;
import eu.dnetlib.rmi.enabling.ISLookUpDocumentNotFoundException;
import eu.dnetlib.rmi.enabling.ISLookUpException;
import eu.dnetlib.rmi.enabling.ISLookUpService;
import org.antlr.stringtemplate.StringTemplate;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Node;
import org.dom4j.io.SAXReader;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.ClassPathResource;

public class ISLookupClient {

	private static final Log log = LogFactory.getLog(ISLookupClient.class);

	@Autowired
	private UniqueServiceLocator serviceLocator;

	public List<String> listSimpleWorflowSections() {
		final String xquery = "distinct-values(//WORKFLOW_NAME/@menuSection/string())";
		try {
			return this.serviceLocator.getService(ISLookUpService.class).quickSearchProfile(xquery);
		} catch (final ISLookUpException e) {
			log.error("Error obtaining worflowSections", e);
			return Lists.newArrayList();
		}
	}

	public List<WorkflowItem> listWorflowsForSection(final String name) {
		return listWorflowsByCondition("//WORKFLOW_NAME/@menuSection='" + name + "'");
	}

	public List<WorkflowItem> listWorflowsForDatasource(final String dsId) {
		return listWorflowsByCondition("//DATASOURCE/@id='" + dsId + "'");
	}

	public List<WorkflowItem> listWorflowsForApi(final String dsId, final String ifaceId) {
		return listWorflowsByCondition("//DATASOURCE[@id='" + dsId + "' and @interface='" + ifaceId + "']");
	}

	private List<WorkflowItem> listWorflowsByCondition(final String cond) {
		final String query =
				"for $x in collection('/db/DRIVER/WorkflowDSResources/WorkflowDSResourceType') where $x" + cond
						+ " order by $x//WORKFLOW_NAME return concat($x//RESOURCE_IDENTIFIER/@value, ' ==@== ', $x//WORKFLOW_NAME, ' ==@== ', $x//WORKFLOW_DESCRIPTION, ' ==@== ', $x//DESTROY_WORKFLOW_TEMPLATE/@id)";

		try {
			return Lists.transform(this.serviceLocator.getService(ISLookUpService.class).quickSearchProfile(query), new Function<String, WorkflowItem>() {

				@Override
				public WorkflowItem apply(final String s) {
					final String[] arr = s.split("==@==");
					return new WorkflowItem(arr[0].trim(), arr[1].trim(), arr[2].trim(), arr[3].trim().length() > 0);
				}
			});
		} catch (final ISLookUpException e) {
			log.error("Error obtaining wfs using query: " + query, e);
			return Lists.newArrayList();
		}
	}

	public String getProfile(final String id) {
		try {
			return this.serviceLocator.getService(ISLookUpService.class).getResourceProfile(id);
		} catch (final ISLookUpException e) {
			log.error("Error finding profile: " + id, e);
			return null;
		}
	}

	public String getRepoProfile(final String id) throws ISLookUpException {
		try {
			return this.serviceLocator.getService(ISLookUpService.class).getResourceProfile(id);
		} catch (final ISLookUpDocumentNotFoundException e) {
			return this.serviceLocator.getService(ISLookUpService.class).getResourceProfileByQuery(
					"collection('/db/DRIVER/RepositoryServiceResources/RepositoryServiceResourceType')/*[.//DATASOURCE_ORIGINAL_ID='" + id + "']");
		}
	}

	public Set<String> getNotConfiguredNodes(final String id) {
		final String query = "for $x in (/*[.//RESOURCE_IDENTIFIER/@value='" + id + "']//NODE) "
				+ "where count($x//PARAM[@required='true' and string-length(normalize-space(.)) = 0]) > 0 " + "return $x/@name/string()";

		try {
			final List<String> list = this.serviceLocator.getService(ISLookUpService.class).quickSearchProfile(query);
			return Sets.newHashSet(list);
		} catch (final Exception e) {
			log.error("Error executing xquery: " + query, e);
			return Sets.newHashSet();
		}
	}

	public List<RepoHIWorkflow> listRepoHiWorkflows(final String compliance, final String type) {

		final List<RepoHIWorkflow> list = Lists.newArrayList();

		if (StringUtils.isNotBlank(compliance) && StringUtils.isNotBlank(type)) {

			try {
				final String query =
						IOUtils.toString(getClass().getResourceAsStream("/eu/dnetlib/functionality/modular/ui/workflows/xquery/find_repohi.xquery"));

				final SAXReader reader = new SAXReader();

				for (final String s : this.serviceLocator.getService(ISLookUpService.class).quickSearchProfile(query)) {
					final Document doc = reader.read(new StringReader(s));
					final Set<String> ifcTypes = Sets.newHashSet(Splitter.on(",").omitEmptyStrings().trimResults().split(doc.valueOf("//TYPES")));
					final Set<String> compliances = Sets.newHashSet(Splitter.on(",").omitEmptyStrings().trimResults().split(doc.valueOf("//COMPLIANCES")));

					if (isValidRepoHiTerm(compliance, compliances) && isValidRepoHiTerm(type, ifcTypes)) {

						final String id = doc.valueOf("//ID");
						final String name = doc.valueOf("//NAME");
						final String desc = doc.valueOf("//DESC");

						final List<Pair<String, String>> fields = new ArrayList<Pair<String, String>>();
						for (final Object o : doc.selectNodes(".//FIELD")) {
							final Node currentNode = (Node) o;
							final String key = currentNode.valueOf("@name");
							final String value = currentNode.getText();
							fields.add(new Pair<String, String>(key, value));
						}

						final RepoHIWorkflow r = new RepoHIWorkflow(id, name, desc, ifcTypes, compliances, fields);

						list.add(r);
					}
				}
				Collections.sort(list);
			} catch (final Exception e) {
				log.error("Error finding repo-hi wfs", e);
			}
		}
		return list;

	}

	private boolean isValidRepoHiTerm(final String term, final Set<String> valids) {
		if (StringUtils.isBlank(term)) {
			return false;
		} else if (valids.size() == 0) {
			return true;
		} else {
			boolean res = false;
			for (final String v : valids) {
				if (term.startsWith(v)) {
					res = true;
				}
			}
			return res;
		}
	}

	public List<String> listWfFamilies() {
		final String query = "distinct-values(for $x in collection('/db/DRIVER/WorkflowDSResources/WorkflowDSResourceType') "
				+ "where string-length($x//DATASOURCE/@id) > 0 return $x//WORKFLOW_TYPE/@text())";
		try {
			return this.serviceLocator.getService(ISLookUpService.class).quickSearchProfile(query);
		} catch (final ISLookUpException e) {
			log.error("Error executing xquery: " + query, e);
			return Lists.newArrayList();
		}
	}

	public String getDatasourceName(final String dsId) {
		final String query = "/*[.//RESOURCE_IDENTIFIER/@value='" + dsId + "']//OFFICIAL_NAME/text()";

		try {
			return this.serviceLocator.getService(ISLookUpService.class).getResourceProfileByQuery(query);
		} catch (final ISLookUpException e) {
			log.error("Error executing xquery: " + query, e);
			return "UNKNOWN";
		}
	}

	public List<Map<String, String>> obtainSubWorkflows(final String wfId) {

		try {
			final ClassPathResource resource = new ClassPathResource("/eu/dnetlib/functionality/modular/xquery/listSubWorkflows.xquery.st");
			final StringTemplate st = new StringTemplate(IOUtils.toString(resource.getInputStream()));
			st.setAttribute("wfId", wfId);

			final List<String> list = this.serviceLocator.getService(ISLookUpService.class).quickSearchProfile(st.toString());
			final SAXReader reader = new SAXReader();
			return list.stream().map(s -> {
				try {
					final Map<String, String> map = new HashMap<String, String>();
					for (final Object o : reader.read(new StringReader(s)).selectNodes("/res/*")) {
						map.put(((Node) o).getName(), ((Node) o).getText());
					}
					return map;
				} catch (final DocumentException e) {
					throw new RuntimeException(e);
				}
			}).collect(Collectors.toList());

		} catch (final Throwable e) {
			log.error("Error obtaining subWorkflows of " + wfId, e);
			return new ArrayList<>();
		}

	}
}
