package eu.dnetlib.functionality.modular.ui.workflows.controllers;

import java.io.IOException;
import java.io.StringReader;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Spliterator;
import java.util.Spliterators;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;
import javax.xml.transform.TransformerFactory;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.Document;
import org.dom4j.Element;
import org.dom4j.Node;
import org.dom4j.io.SAXReader;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.ClassPathResource;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import com.google.common.base.Splitter;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;

import eu.dnetlib.enabling.locators.UniqueServiceLocator;
import eu.dnetlib.functionality.modular.ui.AbstractAjaxController;
import eu.dnetlib.functionality.modular.ui.repositories.objects.RepoHIWorkflow;
import eu.dnetlib.functionality.modular.ui.repositories.objects.VocabularyEntry;
import eu.dnetlib.functionality.modular.ui.repositories.util.RepoUIUtils;
import eu.dnetlib.functionality.modular.ui.workflows.objects.EnvParam;
import eu.dnetlib.functionality.modular.ui.workflows.objects.GraphDetails;
import eu.dnetlib.functionality.modular.ui.workflows.objects.ProcessInfo;
import eu.dnetlib.functionality.modular.ui.workflows.objects.WorkflowItem;
import eu.dnetlib.functionality.modular.ui.workflows.objects.WorkflowNotificationInfo;
import eu.dnetlib.functionality.modular.ui.workflows.objects.WorkflowUpdateInfo;
import eu.dnetlib.functionality.modular.ui.workflows.objects.sections.WorkflowSectionGrouper;
import eu.dnetlib.functionality.modular.ui.workflows.util.ISLookupClient;
import eu.dnetlib.functionality.modular.ui.workflows.util.ISRegistryClient;
import eu.dnetlib.miscutils.functional.xml.ApplyXsltDom4j;
import eu.dnetlib.msro.logging.DnetLogger;
import eu.dnetlib.msro.logging.LogMessage;
import eu.dnetlib.msro.notification.NotificationCondition;
import eu.dnetlib.msro.workflows.graph.Graph;
import eu.dnetlib.msro.workflows.graph.GraphLoader;
import eu.dnetlib.msro.workflows.graph.GraphNode;
import eu.dnetlib.msro.workflows.procs.ProcessRegistry;
import eu.dnetlib.msro.workflows.procs.WorkflowExecutor;
import eu.dnetlib.msro.workflows.procs.WorkflowProcess;
import eu.dnetlib.msro.workflows.procs.WorkflowProcess.StartMode;
import eu.dnetlib.msro.workflows.util.ProcessCallback;
import eu.dnetlib.msro.workflows.util.WorkflowsConstants;
import eu.dnetlib.rmi.data.ProtocolParameterType;
import eu.dnetlib.rmi.datasource.DatasourceManagerService;
import eu.dnetlib.rmi.datasource.DatasourceManagerServiceException;
import eu.dnetlib.rmi.enabling.ISLookUpException;
import eu.dnetlib.rmi.enabling.ISRegistryException;
import eu.dnetlib.rmi.manager.MSROException;

// import com.google.common.collect.Lists;

/**
 * Web controller for the UI
 *
 * @author Michele Artini
 */

@Controller
public class WorkflowsController extends AbstractAjaxController {

	private static final Log log = LogFactory.getLog(WorkflowsController.class);

	@Autowired
	private ISLookupClient isLookupClient;

	@Autowired
	private ISRegistryClient isRegistryClient;

	@Autowired
	private RepoUIUtils repoUIUtils;

	@Autowired
	private GraphLoader graphLoader;

	@Autowired
	private ProcessRegistry graphProcessRegistry;

	@Autowired
	private WorkflowSectionGrouper workflowSectionGrouper;

	@Autowired
	private WorkflowExecutor workflowExecutor;

	@Resource(name = "msroWorkflowLogger")
	private DnetLogger dnetLogger;

	@Autowired
	private UniqueServiceLocator serviceLocator;

	@Value("${repo.ui.compatibilityLevels.vocabulary}")
	private String compatibilityLevelsVocabulary;

	@Autowired
	private TransformerFactory saxonTransformerFactory;

	@RequestMapping("/ui/wf/list_workflows.json")
	public @ResponseBody List<WorkflowItem> listWorflowsForSection(@RequestParam(value = "section", required = false) final String sectionName,
			@RequestParam(value = "dsId", required = false) final String dsId)
			throws ISLookUpException, IOException {
		if (sectionName != null) {
			return workflowSectionGrouper.listWorflowsForSection(sectionName);
		} else if (dsId != null) {
			return workflowSectionGrouper.listWorflowsForDatasource(dsId);
		} else {
			return new ArrayList<>();
		}
	}

	@RequestMapping("/ui/wf/workflow.html")
	public void getWorkflowHtml(final HttpServletResponse res, @RequestParam(value = "id", required = true) final String id) throws Exception {
		final SAXReader reader = new SAXReader();
		final String profile = isLookupClient.getProfile(id);
		final Document doc = reader.read(new StringReader(profile));
		final Element dsNode = (Element) doc.selectSingleNode("//DATASOURCE");
		if (dsNode != null) {
			final String dsId = dsNode.valueOf("@id");
			final String ifaceId = dsNode.valueOf("@interface");
			final String dsProfile = isLookupClient.getProfile(dsId);
			final Document doc2 = reader.read(new StringReader(dsProfile));
			dsNode.addAttribute("name", doc2.valueOf("//OFFICIAL_NAME"));
			dsNode.addAttribute("protocol", doc2.valueOf("//INTERFACE[@id = '" + ifaceId + "']/ACCESS_PROTOCOL"));
			final Node ifcNode = doc2.selectSingleNode("//INTERFACE[@id = '" + ifaceId + "']");
			final Element extraFields = dsNode.addElement("extraFields");
			for (final Object o : ifcNode.selectNodes("./INTERFACE_EXTRA_FIELD")) {
				final Element f = extraFields.addElement("field");
				f.addAttribute("name", ((Node) o).valueOf("@name"));
				f.setText(((Node) o).getText());
			}
		}

		final ApplyXsltDom4j xslt = new ApplyXsltDom4j(new ClassPathResource("/eu/dnetlib/functionality/modular/ui/workflows/xslt/wf_profile2html.xslt"), saxonTransformerFactory);

		res.setContentType("text/html");
		IOUtils.copy(new StringReader(xslt.apply(doc)), res.getOutputStream());
	}

	@RequestMapping("/ui/wf/getGraph.do")
	public @ResponseBody GraphDetails getWorkflowGraphHtml(@RequestParam(value = "wfId", required = true) final String id) throws Exception {
		final String profile = isLookupClient.getProfile(id);
		final Document doc = new SAXReader().read(new StringReader(profile));

		final Graph graph = graphLoader.loadGraph(doc, new HashMap<>());

		return GraphDetails.from(graph);
	}

	@RequestMapping("/ui/wf/formProtocolParameters.find")
	public @ResponseBody List<Map<String, String>> getFormProtocolParameters(@RequestParam(value = "ds", required = true) final String repoId,
			@RequestParam(value = "iface", required = true) final String ifaceId)
			throws Exception {

		final String profile = isLookupClient.getRepoProfile(repoId);

		final SAXReader reader = new SAXReader();
		final Document doc = reader.read(new StringReader(profile));

		final Node ifcNode = doc.selectSingleNode("//INTERFACE[@id = '" + ifaceId + "']");
		final String protocol = ifcNode.valueOf("./ACCESS_PROTOCOL");

		final List<Map<String, String>> list = new ArrayList<>();
		final Map<String, String> baseUrlParam = new HashMap<>();

		baseUrlParam.put("name", "baseUrl");
		baseUrlParam.put("type", ProtocolParameterType.TEXT.toString());
		baseUrlParam.put("value", ifcNode.valueOf("./BASE_URL"));
		baseUrlParam.put("regex", "^(http|https|ftp|ftps|sftp|file):\\/\\/");
		list.add(baseUrlParam);

		repoUIUtils.listParametersForProtocol(protocol).stream().map(pp -> {
			final Map<String, String> res = new HashMap<>();
			res.put("name", pp.getName());
			res.put("type", pp.getType().toString());
			res.put("value", ifcNode.valueOf("./ACCESS_PROTOCOL/@" + pp.getName()));

			if (StringUtils.isNotBlank(pp.getRegex())) {
				res.put("regex", pp.getRegex());
			}
			if (pp.isFunctionPopulated()) {
				res.put("functionPopulated", "true");
			}
			if (pp.isOptional()) {
				res.put("optional", "true");
			}

			return res;
		}).forEach(list::add);;

		return list;
	}

	@RequestMapping("/ui/wf/repoApi.html")
	public void getWorkflowHtml(final HttpServletResponse res,
			@RequestParam(value = "repoId", required = true) final String repoId,
			@RequestParam(value = "ifaceId", required = true) final String ifaceId) throws Exception {

		final String profile = isLookupClient.getRepoProfile(repoId);

		final SAXReader reader = new SAXReader();
		final Document doc = reader.read(new StringReader(profile));

		final Node ifcNode = doc.selectSingleNode("//INTERFACE[@id = '" + ifaceId + "']");

		final String profileId = doc.valueOf("//RESOURCE_IDENTIFIER/@value");

		final String protocol = ifcNode.valueOf("./ACCESS_PROTOCOL");
		final Element extra = doc.getRootElement().addElement("extra_info");

		final Element compLevels = extra.addElement("compatibilityLevels");
		for (final VocabularyEntry e : repoUIUtils.fetchVocabularyTerms(compatibilityLevelsVocabulary)) {
			final Element l = compLevels.addElement("level");
			l.setText(e.getName());
		}

		final Element parameters = extra.addElement("parameters");
		for (final Object o : ifcNode.selectNodes("./ACCESS_PROTOCOL/@*")) {
			final Element p = parameters.addElement("param");
			p.addAttribute("name", ((Node) o).getName());
			p.setText(((Node) o).getText());
		}

		final Element extraFields = extra.addElement("extraFields");
		for (final Object o : ifcNode.selectNodes("./INTERFACE_EXTRA_FIELD")) {
			final Element f = extraFields.addElement("field");
			f.addAttribute("name", ((Node) o).valueOf("@name"));
			f.setText(((Node) o).getText());
		}

		final Element wfs = extra.addElement("workflows");
		for (final WorkflowItem item : isLookupClient.listWorflowsForApi(profileId, ifaceId)) {
			final Element wf = wfs.addElement("workflow");
			wf.addAttribute("id", item.getWfId());
			wf.addAttribute("name", item.getName());
			wf.addAttribute("description", item.getDesc());
			if (item.isDestroy()) {
				wf.addAttribute("destroy", "1");
			}
		}
		final Map<String, String> params = new HashMap<>();
		params.put("profileId", doc.valueOf("//RESOURCE_IDENTIFIER/@value"));
		params.put("ifaceId", ifaceId);
		params.put("protocol", protocol);
		params.put("baseUrl", ifcNode.valueOf("./BASE_URL"));
		params.put("prefix", doc.valueOf(".//FIELD[./key = 'NamespacePrefix']/value"));
		params.put("typology", ifcNode.valueOf("@typology"));
		params.put("compliance", ifcNode.valueOf("@compliance"));
		params.put("removable", ifcNode.valueOf("@removable"));
		params.put("overrideCompliance", ifcNode.valueOf("./INTERFACE_EXTRA_FIELD[@name='overriding_compliance']"));

		doc.selectSingleNode("/RESOURCE_PROFILE/HEADER").detach();
		doc.selectSingleNode("/RESOURCE_PROFILE/BODY/CONFIGURATION/INTERFACES").detach();
		doc.selectSingleNode("/RESOURCE_PROFILE/BODY/CONFIGURATION/EXTRA_FIELDS").detach();
		doc.selectSingleNode("/RESOURCE_PROFILE/BODY/QOS").detach();
		doc.selectSingleNode("/RESOURCE_PROFILE/BODY/STATUS").detach();
		doc.selectSingleNode("/RESOURCE_PROFILE/BODY/SECURITY_PARAMETERS").detach();
		doc.selectSingleNode("/RESOURCE_PROFILE/BODY/BLACKBOARD").detach();

		final ApplyXsltDom4j xslt = new ApplyXsltDom4j(new ClassPathResource("/eu/dnetlib/functionality/modular/ui/workflows/xslt/repoApi.xslt"), params, saxonTransformerFactory);

		res.setContentType("text/html");

		IOUtils.copy(new StringReader(xslt.apply(doc)), res.getOutputStream());
	}

	@RequestMapping("/ui/wf/wf.start")
	public @ResponseBody ProcessInfo startWorkflow(@RequestParam(value = "id", required = true) final String id) throws Exception {
		final String procId = workflowExecutor.startWorkflow(id, null, null);
		final WorkflowProcess process = graphProcessRegistry.findProcess(procId);
		return process != null ? new ProcessInfo(process) : new ProcessInfo(procId);
	}

	@RequestMapping("/ui/wf/wfTemplate.start")
	public @ResponseBody ProcessInfo startWorkflowTemplate(
			@RequestParam(value = "node", required = true) final String nodeName,
			@RequestParam(value = "parentWf", required = true) final String parentWf) throws Exception {

		final String profile = isLookupClient.getProfile(parentWf);
		final Document doc = (new SAXReader()).read(new StringReader(profile));
		final String family = doc.valueOf("//WORKFLOW_FAMILY");
		final int priority = NumberUtils.toInt(doc.valueOf("//WORKFLOW_PRIORITY"), WorkflowsConstants.DEFAULT_WF_PRIORITY);
		final String dsId = doc.valueOf("//DATASOURCE/@id");
		final String iface = doc.valueOf("//DATASOURCE/@interface");

		final Map<String, String> globalParams = new HashMap<>();
		for (final Object o : doc.selectNodes("//CONFIGURATION/PARAMETERS/PARAM")) {
			final Element p = (Element) o;
			globalParams.put(p.valueOf("@name"), p.getTextTrim());
		}

		final Node node = doc.selectSingleNode("//NODE[@name='" + nodeName + "']");
		final Map<String, Object> params =
				GraphNode.newNode(nodeName, node.valueOf("@type"),
						graphLoader.calculateParamsForNode(node, globalParams))
						.resolveParamsWithNoEnv();

		if (!params.containsKey("wfTemplateId") || !(params.get("wfTemplateId") instanceof String)
				|| StringUtils.isBlank((String) params.get("wfTemplateId"))) {
			log.error("wfTemplateId is invalid or missing in profile " + parentWf);
			throw new MSROException("wfTemplateId is invalid or missing in profile " + parentWf);
		}

		if (params.containsKey("wfTemplateParams") && !(params.get("wfTemplateParams") instanceof Map)) {
			log.error("wfTemplateParams is invalid in profile " + parentWf);
			throw new MSROException("wfTemplateParams is invalid in profile " + parentWf);
		}

		final String wfTtemplateId = (String) params.get("wfTemplateId");
		@SuppressWarnings("unchecked")
		final Map<String, String> wfTtemplateParams =
				params.containsKey("wfTemplateParams") ? (Map<String, String>) params.get("wfTemplateParams") : new HashMap<>();

		final String procId = workflowExecutor.startWorkflowTemplate(wfTtemplateId, nodeName, family, priority, dsId, iface, wfTtemplateParams, null, parentWf);

		final WorkflowProcess process = graphProcessRegistry.findProcess(procId);

		return process != null ? new ProcessInfo(process) : new ProcessInfo(procId);
	}

	@RequestMapping("/ui/wf/update_workflow.get")
	public @ResponseBody WorkflowUpdateInfo getWorkflowUpdateInfo(@RequestParam(value = "wfId", required = true) final String wfId) throws Exception {

		final String xml = isLookupClient.getProfile(wfId);

		final Document doc = (new SAXReader()).read(new StringReader(xml));

		final WorkflowUpdateInfo info = new WorkflowUpdateInfo();
		info.setWfId(wfId);
		info.setMode(StartMode.valueOf(doc.valueOf("//CONFIGURATION/@start")));
		info.setPriority(NumberUtils.toInt(doc.valueOf("//WORKFLOW_PRIORITY"), WorkflowsConstants.DEFAULT_WF_PRIORITY));

		info.setScheduled("true".equalsIgnoreCase(doc.valueOf("//SCHEDULING/@enabled")));
		info.setCron(doc.valueOf("//SCHEDULING/CRON"));
		info.setInterval(NumberUtils.toInt(doc.valueOf("//SCHEDULING/MININTERVAL"), 120)); // Default: 120 minutes

		for (final Object o : doc.selectNodes("//NOTIFICATIONS/EMAIL")) {
			info.getNotifications().add(new WorkflowNotificationInfo(
					((Element) o).valueOf("@address"),
					((Element) o).valueOf("@messageProfileId"),
					NotificationCondition.valueOf(((Element) o).valueOf("@condition"))));
		}

		return info;
	}

	@RequestMapping("/ui/wf/update_workflow.do")
	public @ResponseBody boolean updateWorkflow(@RequestParam(value = "json", required = true) final String json) throws Exception {

		final WorkflowUpdateInfo info = (new Gson()).fromJson(json, WorkflowUpdateInfo.class);

		log.info("Updating workflow " + info.getWfId());

		final String xml = isLookupClient.getProfile(info.getWfId());
		final boolean res = isRegistryClient.updateWorkflowProfile(info.getWfId(), xml, info);

		return res;
	}

	@RequestMapping("/ui/wf/proc.kill")
	public @ResponseBody boolean killProcessWorkflow(@RequestParam(value = "id", required = true) final String id) throws Exception {
		graphProcessRegistry.findProcess(id).kill();
		return true;
	}

	@RequestMapping("/ui/wf/journal.range")
	public @ResponseBody Collection<ProcessInfo> rangeWfJournal(@RequestParam(value = "start", required = true) final String start,
			@RequestParam(value = "end", required = true) final String end) throws Exception {

		final DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd");
		final LocalDateTime startDate = LocalDate.parse(start, formatter).atTime(0, 0);
		final LocalDateTime endDate = LocalDate.parse(end, formatter).atTime(23, 59);

		final Iterator<Map<String, String>> iterator = dnetLogger.range(startDate, endDate);
		final LocalDateTime now = LocalDateTime.now();

		final Map<String, ProcessInfo> res = StreamSupport.stream(Spliterators.spliteratorUnknownSize(iterator, Spliterator.ORDERED), false)
				.map(this::toProcessInfo)
				.collect(Collectors.toMap(e -> e.getProcId(), e -> e, (e1, e2) -> e1));

		if (startDate.isBefore(now) && endDate.isAfter(now)) {
			for (final WorkflowProcess proc : graphProcessRegistry.listProcesses()) {
				res.put(proc.getId(), new ProcessInfo(proc));
			}
		}

		return res.values();
	}

	@RequestMapping("/ui/wf/journal.find")
	public @ResponseBody List<ProcessInfo> findWfJournal(@RequestParam(value = "wfs", required = true) final String wfs) {
		final Map<String, ProcessInfo> map = Maps.newHashMap();

		final Set<String> wfFilter = Sets.newHashSet(Splitter.on(",").omitEmptyStrings().trimResults().split(wfs));

		for (final String wfId : wfFilter) {
			dnetLogger.find(WorkflowsConstants.LOG_WF_PROFILE_ID, wfId).forEachRemaining(log -> {
				map.put(log.get(WorkflowsConstants.LOG_WF_PROCESS_ID), toProcessInfo(log));
			});
			dnetLogger.find(WorkflowsConstants.LOG_WF_PARENT, wfId).forEachRemaining(log -> {
				map.put(log.get(WorkflowsConstants.LOG_WF_PROCESS_ID), toProcessInfo(log));
			});
		}

		for (final WorkflowProcess proc : graphProcessRegistry.listProcesses()) {
			if (wfFilter.contains(proc.getProfileId()) || wfFilter.contains(proc.getParentProfileId())) {
				map.put(proc.getId(), new ProcessInfo(proc));
			}
		}

		return map.values().stream().sorted((p1, p2) -> {
			if (p1.getDate() == 0) {
				return -1;
			} else if (p2.getDate() == 0) {
				return 1;
			} else {
				return Long.compare(p2.getDate(), p1.getDate());
			}
		}).collect(Collectors.toList());
	}

	@RequestMapping("/ui/wf/journal_byFamily.find")
	public @ResponseBody Collection<ProcessInfo> findWfJournalByFamily(@RequestParam(value = "family", required = true) final String family)
			throws IOException {

		final Iterator<Map<String, String>> iterator = dnetLogger.find(WorkflowsConstants.LOG_WF_FAMILY, family);

		return StreamSupport.stream(Spliterators.spliteratorUnknownSize(iterator, Spliterator.ORDERED), false)
				.map(this::toProcessInfo)
				.collect(Collectors.toList());

	}

	@RequestMapping("/ui/wf/journal.get")
	public @ResponseBody ProcessInfo getWfJournalLog(@RequestParam(value = "id", required = true) final String procId) throws Exception {
		final WorkflowProcess process = graphProcessRegistry.findProcess(procId);
		final Map<String, String> logs = dnetLogger.findOne("system:processId", procId);

		final ProcessInfo info = (process != null) ? new ProcessInfo(process) : new ProcessInfo(logs);

		if ((logs != null) && !logs.isEmpty()) {
			logs.keySet().stream().sorted().forEach(k -> info.getOutputParams().add(new EnvParam(k, logs.get(k))));
		}

		return info;
	}

	@RequestMapping(value = "/ui/wf/saveParams.do")
	public @ResponseBody boolean saveWorkflowParams(@RequestParam(value = "wf", required = true) final String wfId,
			@RequestParam(value = "params", required = true) final String jsonParams) throws Exception {

		final String xml = isLookupClient.getProfile(wfId);

		final Map<String, Object> params = new Gson().fromJson(jsonParams, new TypeToken<Map<String, Object>>() {}.getType());

		return isRegistryClient.updateWorkflowProfile(wfId, xml, params);
	}

	@RequestMapping(value = "/ui/wf/obtainSubWorkflows.do")
	public @ResponseBody List<Map<String, String>> obtainSubWorkflows(@RequestParam(value = "id", required = true) final String wfId) {

		final List<Map<String, String>> subWfs = isLookupClient.obtainSubWorkflows(wfId);

		for (final Map<String, String> map : subWfs) {
			if ("true".equalsIgnoreCase(map.get("isTemplate"))) {
				final Map<String, Object> query = new HashMap<>();
				query.put(WorkflowsConstants.LOG_WF_PARENT, wfId);
				query.put(WorkflowsConstants.LOG_WF_PROFILE_TEMPLATE_ID, map.get("id"));
				final Map<String, String> log = dnetLogger.findOne(query);
				if ((log != null) && !log.isEmpty()) {
					map.put("lastDate", log.get(WorkflowsConstants.LOG_WF_PROCESS_END_DATE));
					map.put("lastStatus", log.get(WorkflowsConstants.LOG_WF_PROCESS_STATUS));
				}
			}
		}

		return subWfs;
	}

	@RequestMapping(value = "/ui/wf/repohi_wfs.find")
	public @ResponseBody List<RepoHIWorkflow> listRepoHIWorkflows(@RequestParam(value = "compliance", required = true) final String compliance,
			@RequestParam(value = "type", required = true) final String type) throws ISLookUpException {

		return isLookupClient.listRepoHiWorkflows(compliance, type);
	}

	@RequestMapping("/ui/wf/repohi.start")
	public @ResponseBody List<String> newRepoWorkflow(@RequestParam(value = "id", required = true) final String repoId,
			@RequestParam(value = "iface", required = true) final String ifaceId,
			@RequestParam(value = "wf", required = true) final String wfId) throws Exception {

		final String procId = workflowExecutor.startRepoHiWorkflow(wfId, repoId, ifaceId, new ProcessCallback() {

			@Override
			public void onSuccess() {
				try {
					serviceLocator.getService(DatasourceManagerService.class).updateActivationStatus(repoId, ifaceId, true);
				} catch (final DatasourceManagerServiceException e) {
					log.error("Error updating activation status of " + repoId, e);
				}
			}

			@Override
			public void onFail() {}
		}, null);

		return Arrays.asList(procId);
	}

	@RequestMapping("/ui/wf/repobye.start")
	public @ResponseBody List<String> destroyRepoWorkflow(@RequestParam(value = "wf", required = true) final String wf)
			throws Exception {

		final String profile = isLookupClient.getProfile(wf);
		final Document doc = (new SAXReader()).read(new StringReader(profile));
		final String dsId = doc.valueOf("//DATASOURCE/@id");
		final String iface = doc.valueOf("//DATASOURCE/@interface");

		final Map<String, String> globalParams = new HashMap<>();
		for (final Object o : doc.selectNodes("//CONFIGURATION/PARAMETERS/PARAM")) {
			final Element p = (Element) o;
			globalParams.put(p.valueOf("@name"), p.getTextTrim());
		}

		final Node node = doc.selectSingleNode("//DESTROY_WORKFLOW_TEMPLATE");
		if (node == null) {
			log.error("DESTROY WF is invalid or missing in profile " + wf);
			throw new MSROException("DESTROY WF is invalid or missing in profile " + wf);
		}

		final String wfTtemplateId = node.valueOf("@id");

		final Map<String, String> params = GraphNode.newNode("REPO_BYE", "REPO_BYE",
				graphLoader.calculateParamsForNode(node, globalParams))
				.resolveParamsWithNoEnv().entrySet()
				.stream()
				.filter(e -> e.getValue() instanceof String)
				.collect(Collectors.toMap(e -> e.getKey(), e -> (String) e.getValue(), (e1, e2) -> e1));

		final String procId =
				workflowExecutor.startWorkflowTemplate(wfTtemplateId, "REPO_BYE", "REPO_BYE", WorkflowsConstants.DEFAULT_WF_PRIORITY, dsId, iface,
						params, new ProcessCallback() {

							@Override
							public void onSuccess() {
								try {
									isRegistryClient.deleteProfile(wf);
								} catch (final ISRegistryException e) {
									log.error("Error deleting workflow: " + wf);
								}
							}

							@Override
							public void onFail() {}
						}, wf);

		return Arrays.asList(procId);
	}

	private ProcessInfo toProcessInfo(final Map<String, String> input) {
		final ProcessInfo info = new ProcessInfo();

		info.setProcId(input.get(WorkflowsConstants.LOG_WF_PROCESS_ID));
		info.setWfId(input.get(WorkflowsConstants.LOG_WF_PROFILE_ID));
		info.setName(input.get(WorkflowsConstants.LOG_WF_NAME));
		info.setFamily(input.get(WorkflowsConstants.LOG_WF_FAMILY));
		info.setDatasource(input.containsKey(WorkflowsConstants.LOG_DATASOURCE_NAME) ? input.get(WorkflowsConstants.LOG_DATASOURCE_NAME) : "");
		info.setStatus(input.get(WorkflowsConstants.LOG_WF_PROCESS_STATUS));
		info.setDate(NumberUtils.toLong(input.get(LogMessage.LOG_DATE_FIELD), 0));
		info.setStartDate(info.getStartDate());
		info.setEndDate(info.getEndDate());

		return info;

	}

}
