package eu.dnetlib.msro.workflows.graph;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import eu.dnetlib.rmi.manager.MSRORuntimeException;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.Document;
import org.dom4j.Element;
import org.dom4j.Node;
import org.springframework.beans.factory.annotation.Required;

import com.google.common.collect.Sets;

import eu.dnetlib.conf.PropertyFetcher;
import eu.dnetlib.msro.workflows.util.NodeHelper;
import eu.dnetlib.rmi.manager.MSROException;

/**
 * Created by michele on 19/11/15.
 */
public class GraphLoader {

	private static final Log log = LogFactory.getLog(GraphLoader.class);

	private NodeHelper nodeHelper;

	@Resource(name = "propertyFetcher")
	private PropertyFetcher propertyFetcher;

	public Graph loadGraph(final Document doc, final Map<String, String> globalParams) throws MSROException {
		final Graph graph = new Graph();

		for (final Object o : doc.selectNodes("//CONFIGURATION/WORKFLOW/NODE")) {
			final Element n = (Element) o;
			final String nodeName = n.valueOf("@name");
			final String nodeType = n.valueOf("@type");
			final boolean isStart = StringUtils.equalsIgnoreCase(n.valueOf("@isStart"), "true");
			final boolean isJoin = StringUtils.equalsIgnoreCase(n.valueOf("@isJoin"), "true");

			final Map<String, GraphNodeParameter> params = calculateParamsForNode(n, globalParams);

			if (isStart) {
				graph.addNode(GraphNode.newStartNode(nodeName, nodeType, params));
			} else if (isJoin) {
				graph.addNode(GraphNode.newJoinNode(nodeName, nodeType, params));
			} else {
				graph.addNode(GraphNode.newNode(nodeName, nodeType, params));
			}

			for (final Object o1 : n.selectNodes(".//ARC")) {
				final Element a = (Element) o1;
				final String arcName = a.valueOf("@name");
				final String to = a.valueOf("@to");
				graph.addArc(new Arc(StringUtils.isNotBlank(arcName) ? arcName : Arc.DEFAULT_ARC, nodeName, to));
			}

			graph.addNode(GraphNode.newSuccessNode());
		}

		checkValidity(graph);

		return graph;
	}

	public Map<String, GraphNodeParameter> calculateParamsForNode(final Node node, final Map<String, String> globalParams) {

		final Map<String, GraphNodeParameter> params = new HashMap<>();

		if (node != null) {
			for (final Object o : node.selectNodes(".//PARAM")) {
				final Element p = (Element) o;

				final String pName = p.valueOf("@name");
				final GraphNodeParameter pValue = calculateSimpleValue((Element) o, globalParams);

				if (pValue != null) {
					params.put(pName, pValue);
				} else if (p.selectSingleNode("./MAP") != null) {

					@SuppressWarnings("unchecked")
					final Map<String, GraphNodeParameter> map = ((List<Element>) p.selectNodes("./MAP/ENTRY"))
							.stream()
							.collect(Collectors.toMap(
											e -> e.valueOf("@key"),
											e -> {
												final GraphNodeParameter gnp = calculateSimpleValue(e, globalParams);
												if (gnp == null) {
													final String msg = String.format("missing value for param: \"%s\"", e.valueOf("@key"));
													log.debug(msg);
													return GraphNodeParameter.newNullParam();
												}
												return gnp;
											}));

					params.put(pName, GraphNodeParameter.newMapParam(map));

				} else if (p.selectSingleNode("./LIST") != null) {
					@SuppressWarnings("unchecked")
					final List<GraphNodeParameter> list = ((List<Element>) p.selectNodes("./LIST/ITEM"))
							.stream()
							.map(e -> calculateSimpleValue(e, globalParams))
							.collect(Collectors.toList());
					params.put(pName, GraphNodeParameter.newListParam(list));
				}
			}
		}

		return params;
	}

	private GraphNodeParameter calculateSimpleValue(final Element elem, final Map<String, String> globalParams) {
		final String value = elem.valueOf("@value");
		final String ref = elem.valueOf("@ref");
		final String prop = elem.valueOf("@property");
		final String envRef = elem.valueOf("@env");

		if (StringUtils.isNotBlank(ref) && StringUtils.isNotBlank(globalParams.get(ref))) {
			return GraphNodeParameter.newSimpleParam(globalParams.get(ref));
		} else if (StringUtils.isNotBlank(envRef)) {
			return GraphNodeParameter.newEnvParam(envRef);
		} else if (StringUtils.isNotBlank(value)) {
			return GraphNodeParameter.newSimpleParam(value);
		} else if (StringUtils.isNotBlank(prop)) {
			return GraphNodeParameter.newSimpleParam(this.propertyFetcher.getProperty(prop));
		} else {
			return null;
		}
	}

	private void checkValidity(final Graph graph) throws MSROException {

		final Set<String> nodesFromArcs = new HashSet<String>();

		boolean foundSuccess = false;
		boolean foundStart = false;

		for (final Arc arc : graph.getArcs()) {
			if (StringUtils.isBlank(arc.getFrom()) || StringUtils.isBlank(arc.getFrom())) { throw new MSROException("Invalid arc: missing from e/o to"); }
			if (StringUtils.equals(arc.getTo(), GraphNode.SUCCESS_NODE)) {
				foundSuccess = true;
			}
			nodesFromArcs.add(arc.getFrom());
			nodesFromArcs.add(arc.getTo());
		}

		if (!foundSuccess) { throw new MSROException("Arc to success not found"); }

		final Set<String> diff = Sets.symmetricDifference(graph.nodeNames(), nodesFromArcs);
		if (!diff.isEmpty()) { throw new MSROException("Missing or invalid nodes in arcs: " + diff); }

		for (final GraphNode n : graph.nodes()) {
			if (StringUtils.isBlank(n.getName())) { throw new MSROException("Invalid node: missing name"); }
			if (n.isStart()) {
				foundStart = true;
			}
			if (!this.nodeHelper.isValidType(n.getType())) { throw new MSROException("Invalid node type: " + n.getType()); }
		}
		if (!foundStart) { throw new MSROException("Start node not found"); }
	}

	public NodeHelper getNodeHelper() {
		return this.nodeHelper;
	}

	@Required
	public void setNodeHelper(final NodeHelper nodeHelper) {
		this.nodeHelper = nodeHelper;
	}

}
