package eu.dnetlib.msro.workflows.nodes.transform;

import java.io.IOException;
import java.util.Map;

import com.google.common.collect.Maps;
import eu.dnetlib.enabling.locators.UniqueServiceLocator;
import eu.dnetlib.enabling.resultset.factory.ResultSetFactory;
import eu.dnetlib.msro.workflows.graph.Arc;
import eu.dnetlib.msro.workflows.nodes.SimpleJobNode;
import eu.dnetlib.msro.workflows.procs.Env;
import eu.dnetlib.rmi.common.ResultSet;
import eu.dnetlib.rmi.enabling.ISLookUpService;
import eu.dnetlib.rmi.manager.MSROException;
import groovy.lang.GroovyShell;
import groovy.util.GroovyScriptEngine;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;

public class GroovyJobNode extends SimpleJobNode {

	private static final Log log = LogFactory.getLog(GroovyJobNode.class);

	/**
	 * used to transform the records using Groovy.
	 */
	@Autowired
	private ResultSetFactory resultSetFactory;

	private String inputEprParam;
	private String outputEprParam;
	private String transformationRuleId;
	private Map<String, String> groovyParams;

	@Autowired
	private UniqueServiceLocator serviceLocator;

	private Map<String, String> retrieveGroovyParameter() {
		final Map<String, String> out = Maps.newHashMap();

		final String query = "for $x in collection('/db/DRIVER/GroovyProcessingDSResource/GroovyProcessingDSResourceType')"
				+ "where $x[.//RESOURCE_IDENTIFIER/@value='" + this.transformationRuleId + "']"
				+ "return concat($x//GROOVY_CLASSPATH/text(),':::',$x//GROOVY_DNETCLASS/text())";
		try {
			final String result = this.serviceLocator.getService(ISLookUpService.class).quickSearchProfile(query).get(0);
			if (result == null) { return null; }
			final String[] data = result.trim().split(":::");
			if (data.length == 2) {
				out.put("classpath", data[0]);
				out.put("mainClass", data[1]);
			}

			return out;
		} catch (final Exception e) {
			log.error(e);
			return null;
		}
	}

	@Override
	protected String execute(final Env env) throws Exception {
		final ResultSet<?> rsIn = env.getAttribute(this.inputEprParam, ResultSet.class);
		if (rsIn == null) { throw new MSROException("InputEprParam (" + this.inputEprParam + ") not found in ENV"); }

		String groovyClasspath, groovyDnetClass;
		final Map<String, String> prop = retrieveGroovyParameter();
		groovyClasspath = prop.get("classpath");
		groovyDnetClass = prop.get("mainClass");

		final ResultSet<?> rsOut = transformGroovy(rsIn, groovyClasspath, groovyDnetClass, this.groovyParams);
		env.setAttribute(this.outputEprParam, rsOut);

		return Arc.DEFAULT_ARC;
	}

	private ResultSet<?> transformGroovy(final ResultSet<?> source,
			final String groovyClasspath,
			final String groovyDnetClass,
			final Map<String, String> params) throws ClassNotFoundException, IOException {

		final GroovyScriptEngine gse = new GroovyScriptEngine(groovyClasspath);
		gse.getGroovyClassLoader().loadClass(groovyDnetClass);
		log.info("***********************************************");
		log.info("Loaded Groovy classes:");
		for (final Class<?> c : gse.getGroovyClassLoader().getLoadedClasses()) {
			log.info(c.getCanonicalName());
		}
		log.info("***********************************************");
		final GroovyShell groovyShell = new GroovyShell(gse.getGroovyClassLoader());

		final Object go = groovyShell.evaluate("new " + groovyDnetClass + "()");
		if (go instanceof GroovyUnaryFunction) {
			final GroovyUnaryFunction groovyUnaryFunction = (GroovyUnaryFunction) go;
			if (params != null) {
				groovyUnaryFunction.setParams(params);
			}
			return this.resultSetFactory.map(source, String.class, groovyUnaryFunction);
		} else {
			throw new RuntimeException("Groovy object " + go + " is not supported");
		}
	}

	public String getInputEprParam() {
		return this.inputEprParam;
	}

	public void setInputEprParam(final String inputEprParam) {
		this.inputEprParam = inputEprParam;
	}

	public String getOutputEprParam() {
		return this.outputEprParam;
	}

	public void setOutputEprParam(final String outputEprParam) {
		this.outputEprParam = outputEprParam;
	}

	public String getTransformationRuleId() {
		return this.transformationRuleId;
	}

	public void setTransformationRuleId(final String transformationRuleId) {
		this.transformationRuleId = transformationRuleId;
	}

	public Map<String, String> getGroovyParams() {
		return this.groovyParams;
	}

	public void setGroovyParams(final Map<String, String> groovyParams) {
		this.groovyParams = groovyParams;
	}

}
