package eu.dnetlib.msro.workflows.nodes.xmlvalidation;

import java.io.StringReader;
import javax.xml.transform.sax.SAXSource;

import eu.dnetlib.enabling.resultset.client.ResultSetClient;
import eu.dnetlib.msro.workflows.graph.Arc;
import eu.dnetlib.msro.workflows.nodes.SimpleJobNode;
import eu.dnetlib.msro.workflows.procs.Env;
import eu.dnetlib.rmi.common.ResultSet;
import eu.dnetlib.rmi.manager.MSROException;
import net.sf.saxon.s9api.*;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.xml.sax.InputSource;

/**
 * This class checks if the records declare an expected XML schema URL
 * Created by alessia on 07/03/17.
 *
 * Implemented for Parthenos but not used: eu.dnetlib.msro.workflows.nodes.xmlvalidation.XMLSchemaValidatorJobNode is used instead for a full validation of records.
 * TODO: delete this class if not used by 07/06/17 as it means we do not need it
 */
public class SchemaVerifierJobNode extends SimpleJobNode {

	private static final Log log = LogFactory.getLog(SchemaVerifierJobNode.class);
	private String expectedSchemaURL;
	private boolean enableVerification;
	private String eprParam;
	private final static String XSI_NS = "http://www.w3.org/2001/XMLSchema-instance";
	private final static String XSD_LOCATION = "xsi:schemaLocation";
	private final static String XSD_NO_NS_LOCATION = "xsi:noNamespaceSchemaLocation";

	@Autowired
	private ResultSetClient resultSetClient;

	private Processor xmlProcessor = new Processor(false);

	private XPathSelector xpathSelectorSchema;
	private XPathSelector xpathSelectorSchemaNoNS;
	private XPathSelector xpathSelectorId;

	@Override
	protected String execute(final Env env) throws Exception {
		if (enableVerification) {
			final ResultSet<?> rsIn = env.getAttribute(this.eprParam, ResultSet.class);
			if ((rsIn == null)) { throw new MSROException("InputEprParam (" + this.eprParam + ") not found in ENV"); }

			prepareXpath();
			Iterable<String> records = resultSetClient.iter(rsIn, String.class);

			for (String record : records) {
				verifySchema(record);
			}
		}
		return Arc.DEFAULT_ARC;
	}

	private void prepareXpath() throws SaxonApiException {
		XPathCompiler compiler = xmlProcessor.newXPathCompiler();
		compiler.declareNamespace("xsi", XSI_NS);
		XPathExecutable xpathSchema = compiler.compile("@" + XSD_LOCATION);
		xpathSelectorSchema = xpathSchema.load();

		XPathExecutable xpathSchemaNoNS = compiler.compile("@" + XSD_NO_NS_LOCATION);
		xpathSelectorSchemaNoNS = xpathSchemaNoNS.load();

		XPathExecutable xpathID = compiler.compile("//*[local-name()='objIdentifier']");
		xpathSelectorId = xpathID.load();

	}

	private void verifySchema(String record) throws SaxonApiException {

		SAXSource source = new SAXSource(new InputSource(new StringReader(record)));
		DocumentBuilder docBuilder = xmlProcessor.newDocumentBuilder();
		XdmNode xdmNode = docBuilder.build(source);
		xpathSelectorId.setContextItem(xdmNode);
		XdmItem objidItem = xpathSelectorId.evaluateSingle();
		if (objidItem == null) {
			throw new RuntimeException("Record with no objIdentifier");
		}
		String objid = objidItem.getStringValue();
		xpathSelectorSchema.setContextItem(xdmNode);
		XdmItem res = xpathSelectorSchema.evaluateSingle();
		if (res == null) {
			xpathSelectorSchemaNoNS.setContextItem(xdmNode);
			res = xpathSelectorSchemaNoNS.evaluateSingle();
			if (res == null) {
				//TODO: add to report: no schemaLocation / no noNamespaceSchemaLocation
				log.warn("Record " + objid + " has no " + XSD_LOCATION + " nor " + XSD_NO_NS_LOCATION);
			}
		}
		if (res != null) {
			String schema = res.getStringValue();
			if (!schema.equalsIgnoreCase(expectedSchemaURL)) {
				//TODO: add to report
				log.warn("Record " + objid + " has XSD: " + schema + ". Expected schema: " + expectedSchemaURL);
			}
		}

	}

	public String getExpectedSchemaURL() {
		return expectedSchemaURL;
	}

	public void setExpectedSchemaURL(final String expectedSchemaURL) {
		this.expectedSchemaURL = expectedSchemaURL;
	}

	public boolean isEnableVerification() {
		return enableVerification;
	}

	public void setEnableVerification(final boolean enableVerification) {
		this.enableVerification = enableVerification;
	}

	public String getEprParam() {
		return eprParam;
	}

	public void setEprParam(final String eprParam) {
		this.eprParam = eprParam;
	}
}
