package eu.dnetlib.msro.workflows.nodes.transform;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.function.Function;

import eu.delving.x3ml.X3MLEngineFactory;
import eu.delving.x3ml.X3MLEngineFactory.OutputFormat;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * Created by alessia on 13/03/17.
 */
public class ApplyX3Mapping implements Function<String, String> {

	private static final Log log = LogFactory.getLog(ApplyX3Mapping.class);
	private String[] mappings;
	private String generatorPolicy;
	private boolean verboseLogging;

	public ApplyX3Mapping(final String[] mappings, final String generatorPolicy, final boolean verboseLogging) {
		this.mappings = mappings;
		this.generatorPolicy = generatorPolicy;
		this.verboseLogging = verboseLogging;
	}

	@Override
	public String apply(final String metadata) {
		InputStream[] mappingStreams = new InputStream[mappings.length];
		try {
			for (int i = 0; i < mappings.length; i++) {
				mappingStreams[i] = getStream(mappings[i]);
			}
		}catch(IOException e){
			log.error("Can't create mappingStreams for mappings");
			closeStreams(mappingStreams);
			throw new RuntimeException(e);
		}
		try (   InputStream policyStream = getStream(generatorPolicy);
				InputStream metadataStream = getStream(metadata);
				final ByteArrayOutputStream os = new ByteArrayOutputStream()){
			X3MLEngineFactory x3mEngineFactory = getConfiguredX3MEngineFactory(mappingStreams, policyStream);
			x3mEngineFactory.withInput(metadataStream).withOutput(os, OutputFormat.RDF_XML_PLAIN);
			x3mEngineFactory.execute();
			return new String(os.toByteArray());
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
		finally {
			closeStreams(mappingStreams);
		}
	}

	private void closeStreams(InputStream[] streams){
		for(InputStream s: streams){
			try {
				if(s != null) s.close();
			} catch (IOException e) {
				log.error("Can't close stream"+e.getMessage());
			}
		}
	}

	private X3MLEngineFactory getConfiguredX3MEngineFactory(InputStream[] mappings, InputStream policy) {
		X3MLEngineFactory x3mEngineFactory = X3MLEngineFactory.create().withMappings(mappings);
		if (policy != null) {
			x3mEngineFactory.withGeneratorPolicy(policy);
		}
		if (verboseLogging)
			x3mEngineFactory.withVerboseLogging();
		return x3mEngineFactory;
	}

	private InputStream getStream(String s) throws IOException {
		if (StringUtils.isNotBlank(s)) {
			log.debug("Getting stream from "+s);
			return IOUtils.toInputStream(s, "UTF-8");
		}
		return null;
	}
}
