package eu.dnetlib.msro.workflows.nodes.objectStore;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import eu.dnetlib.enabling.locators.UniqueServiceLocator;
import eu.dnetlib.msro.workflows.graph.Arc;
import eu.dnetlib.msro.workflows.nodes.SimpleJobNode;
import eu.dnetlib.msro.workflows.procs.Env;
import eu.dnetlib.rmi.enabling.ISLookUpException;
import eu.dnetlib.rmi.enabling.ISLookUpService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Required;

/**
 * The Class RetrieveMdStoreId is a job node used to retrieve the correct MDStore from which extract the url of the file to download.
 * metadata format and interpretation are injected as properties
 */
public class RetrieveMdStoreId extends SimpleJobNode {

	/**
	 * The metadata format.
	 */
	private String metadataFormat;

	/**
	 * The interpretation.
	 */
	private String interpretation;

	/**
	 * The provider id.
	 */
	private String providerId;

	/**
	 * The service locator.
	 */
	@Autowired
	private UniqueServiceLocator serviceLocator;

	@Override
	protected String execute(final Env env) throws Exception {

		final String workflowQuery =
				"distinct-values(for $x in collection('/db/DRIVER/MetaWorkflowDSResources/MetaWorkflowDSResourceType') where($x//DATAPROVIDER/@id='%s') return  $x//WORKFLOW/@id/string() )";

		final List<String> result = this.serviceLocator.getService(ISLookUpService.class).quickSearchProfile(String.format(workflowQuery, this.providerId));
		if (result.size() == 0) { throw new RuntimeException("there is no mdStore Associated to the provider " + env.getAttribute(getProviderId())); }
		final Set<String> workflowIds = Sets.newHashSet(result);

		final Set<String> metadataIds = getMdStores(workflowIds);
		final Gson g = new GsonBuilder().disableHtmlEscaping().create();
		env.setAttribute("mdId", g.toJson(metadataIds));

		env.setAttribute("mdFormat", getMetadataFormat());
		return Arc.DEFAULT_ARC;
	}

	private Set<String> getMdStores(final Set<String> workflowsId) {
		try {

			final String query = "distinct-values( for $x in collection('/db/DRIVER/WorkflowDSResources/WorkflowDSResourceType') "
					+ "where $x//RESOURCE_IDENTIFIER/@value='%s' "
					+ "return $x//PARAM[./@category='MDSTORE_ID']/text() )";

			final Set<String> mdStores = new HashSet<>();

			if (workflowsId == null) { return null; }

			for (final String workflowId : workflowsId) {
				final List<String> result = this.serviceLocator.getService(ISLookUpService.class).quickSearchProfile(String.format(query, workflowId));
				final Set<String> metadataIds = Sets.newHashSet(result);
				mdStores.addAll(getRightMetadataId(Lists.newArrayList(metadataIds)));
			}
			return mdStores;

		} catch (final ISLookUpException e) {

			return null;
		}
	}

	/**
	 * Gets the right metadata id whith the format metadataFormat and interpretation interpretation
	 *
	 * @return the right metadata id
	 * @throws ISLookUpException
	 */
	private Set<String> getRightMetadataId(final Iterable<String> ids) throws ISLookUpException {
		final String query =

				" for $x in collection('/db/DRIVER/MDStoreDSResources/MDStoreDSResourceType') "
						+ "where $x//RESOURCE_IDENTIFIER/@value='%s' "
						+ "return concat($x//METADATA_FORMAT/text(), '::<<>>::', $x//METADATA_FORMAT_INTERPRETATION/text())";


		final Set<String> result = Sets.newHashSet();

		for (final String id : ids) {

			final List<String> results = this.serviceLocator.getService(ISLookUpService.class).quickSearchProfile(String.format(query, id));
			if (results.size() > 0) {
				final String[] values = results.get(0).split("::<<>>::");
				if (this.metadataFormat.equals(values[0]) && this.interpretation.equals(values[1])) {
					result.add(id);
				}
			}
		}
		return result;

	}

	/**
	 * Gets the interpretation.
	 *
	 * @return the interpretation
	 */
	public String getInterpretation() {
		return this.interpretation;
	}

	/**
	 * Sets the interpretation.
	 *
	 * @param interpretation
	 *            the interpretation to set
	 */
	@Required
	public void setInterpretation(final String interpretation) {
		this.interpretation = interpretation;
	}

	/**
	 * Gets the metadata format.
	 *
	 * @return the metadataFormat
	 */
	public String getMetadataFormat() {
		return this.metadataFormat;
	}

	/**
	 * Sets the metadata format.
	 *
	 * @param metadataFormat
	 *            the metadataFormat to set
	 */
	@Required
	public void setMetadataFormat(final String metadataFormat) {
		this.metadataFormat = metadataFormat;
	}

	/**
	 * Gets the provider id.
	 *
	 * @return the providerId
	 */
	public String getProviderId() {
		return this.providerId;
	}

	/**
	 * Sets the provider id.
	 *
	 * @param providerId
	 *            the providerId to set
	 */
	public void setProviderId(final String providerId) {
		this.providerId = providerId;
	}

}
