package eu.dnetlib.msro.workflows.procs;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.PriorityBlockingQueue;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Required;

import com.google.common.collect.BiMap;
import com.google.common.collect.HashBiMap;

import eu.dnetlib.miscutils.datetime.DateUtils;
import eu.dnetlib.msro.workflows.util.WorkflowsConstants;
import eu.dnetlib.rmi.manager.MSROException;

/**
 * Created by michele on 20/11/15.
 */
public class ProcessRegistry {

	private static final Log log = LogFactory.getLog(ProcessRegistry.class);
	private final BiMap<String, WorkflowProcess> procs = HashBiMap.create();
	private final Map<String, Collection<WorkflowProcess>> byOtherId = new HashMap<String, Collection<WorkflowProcess>>();

	private final PriorityBlockingQueue<WorkflowProcess> pendingProcs = new PriorityBlockingQueue<WorkflowProcess>();

	private int maxSize;

	synchronized public int countRunningWfs() {
		int count = 0;
		for (final Map.Entry<String, WorkflowProcess> e : this.procs.entrySet()) {
			final WorkflowProcess proc = e.getValue();
			if (!proc.isTerminated()) {
				count++;
			}
		}
		return count;
	}

	public WorkflowProcess findProcess(final String procId) {
		return this.procs.get(procId);
	}

	public Set<WorkflowProcess> listProcesses() {
		return this.procs.values();
	}

	public Collection<WorkflowProcess> findProcsByOtherId(final String id) {
		synchronized (this) {
			final Collection<WorkflowProcess> res = this.byOtherId.get(id);
			return res != null ? res : new ArrayList<WorkflowProcess>();
		}
	}

	public String registerProcess(final WorkflowProcess process, final String... ids) throws MSROException {
		if (this.procs.containsValue(process) || this.procs.containsKey(process.getId())) {
			log.error("Already registered process: " + process);
			throw new MSROException("Already registered process: " + process);
		}

		if (this.procs.size() >= this.maxSize) {
			removeOldestProcess();
		}

		this.procs.put(process.getId(), process);
		for (final String id : ids) {
			synchronized (this) {
				if (!this.byOtherId.containsKey(id)) {
					this.byOtherId.put(id, new ArrayList<WorkflowProcess>());
				}
				this.byOtherId.get(id).add(process);
			}
		}

		synchronized (this.pendingProcs) {
			if (this.pendingProcs.size() > WorkflowsConstants.MAX_PENDING_PROCS_SIZE) {
				log.warn("Wf [" + process.getName() + "] not launched, Max number of pending procs reached: " + WorkflowsConstants.MAX_PENDING_PROCS_SIZE);
				throw new MSROException("Max number of pending procs reached: " + WorkflowsConstants.MAX_PENDING_PROCS_SIZE);
			}
			this.pendingProcs.put(process);

			log.info("WorkflowProcess [" + process + "] in queue, priority=" + process.getPriority());
		}

		return process.getId();
	}

	private void removeOldestProcess() {
		long oldDate = DateUtils.now();
		String oldId = null;

		for (final Map.Entry<String, WorkflowProcess> e : this.procs.entrySet()) {
			final WorkflowProcess proc = e.getValue();

			if (proc.isTerminated()) {
				final long date = proc.getLastActivityDate();
				if (date < oldDate) {
					oldDate = date;
					oldId = e.getKey();
				}
			}
		}

		if (oldId != null) {
			unregisterProcess(oldId);
		}

	}

	public void unregisterProcess(final String procId) {
		synchronized (this) {
			final WorkflowProcess process = this.procs.remove(procId);
			if (process != null) {
				for (final Collection<WorkflowProcess> processes : this.byOtherId.values()) {
					processes.remove(process);
				}
			}
		}
	}

	public WorkflowProcess nextProcessToStart() {
		synchronized (this.pendingProcs) {
			return this.pendingProcs.poll();
		}
	}

	public int getMaxSize() {
		return this.maxSize;
	}

	@Required
	public void setMaxSize(final int maxSize) {
		this.maxSize = maxSize;
	}

}
