package eu.dnetlib.msro.workflows.nodes.index;

import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.Map;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMResult;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;

import eu.dnetlib.enabling.locators.UniqueServiceLocator;
import eu.dnetlib.enabling.resultset.client.ResultSetClient;
import eu.dnetlib.enabling.resultset.factory.ResultSetFactory;
import eu.dnetlib.enabling.tools.blackboard.BlackboardJob;
import eu.dnetlib.msro.workflows.nodes.BlackboardJobNode;
import eu.dnetlib.msro.workflows.nodes.blackboard.BlackboardWorkflowJobListener;
import eu.dnetlib.msro.workflows.procs.Env;
import eu.dnetlib.msro.workflows.procs.Token;
import eu.dnetlib.msro.workflows.util.ResultsetProgressProvider;
import eu.dnetlib.msro.workflows.util.WorkflowsConstants;
import eu.dnetlib.rmi.common.ResultSet;
import eu.dnetlib.rmi.common.ResultSetException;
import eu.dnetlib.rmi.enabling.ISLookUpDocumentNotFoundException;
import eu.dnetlib.rmi.enabling.ISLookUpException;
import eu.dnetlib.rmi.enabling.ISLookUpService;
import eu.dnetlib.rmi.provision.IndexService;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.Resource;

public class UpdateIndexJobNode extends BlackboardJobNode {

	private static final Log log = LogFactory.getLog(UpdateIndexJobNode.class);

	private String eprParam;
	private String indexId;
	private String format;
	@Value("${service.index.default.interpretation}")
	private String interpretation;
	private String layout;
	private String feedingType;
	private String defaultIndexId;

	@Autowired
	private ResultSetFactory resultSetFactory;

	@Autowired
	private ResultSetClient resultSetClient;

	@Autowired
	private TransformerFactory saxonTransformerFactory;

	/**
	 * Stylesheet which transforms a layout to another stylesheet which converts a input record to a index record.
	 */
	@Value("${service.index.layoutToRecordStylesheet}")
	private Resource layoutToRecordStylesheet;

	/**
	 * service locator.
	 */
	@Autowired
	private UniqueServiceLocator serviceLocator;

	@Override
	protected String obtainServiceId(final Env env) {
		return getServiceLocator().getServiceId(IndexService.class);
	}

	@Override
	protected void prepareJob(final BlackboardJob job, final Token token) throws ResultSetException, ISLookUpException, IOException, TransformerException {
		log.info("preparing blackboard job update index: " + getIndexId());

		final ResultSet<?> rs = token.getEnv().getAttribute(getEprParam(), ResultSet.class);

		final ResultSet<?> mappedRs = prepareForIndexing(rs, getFormat(), getInterpretation(), getLayout());

		token.setProgressProvider(new ResultsetProgressProvider(mappedRs, this.resultSetClient));

		job.setAction("FEED");
		job.getParameters().put("resultset_epr", mappedRs.toJson());
		job.getParameters().put("id", getIndexId());
		job.getParameters().put("feeding_type", getFeedingType());
		job.getParameters().put("backend_Id", this.defaultIndexId);
	}

	/**
	 * Transforms each mdstore record into a index record.
	 *
	 * @param resultSet
	 *            mdstore resulsetset
	 * @param layout
	 *            layout
	 * @param format
	 *            format
	 * @return resultset with transformed records
	 * @throws ISLookUpException
	 *             could happen
	 * @throws IOException
	 *             could happen
	 * @throws TransformerException
	 *             could happen
	 */
	protected ResultSet<String> prepareForIndexing(final ResultSet<?> resultSet, final String format, final String interpretation, final String layout)
			throws ISLookUpException, IOException, TransformerException {

		log.info("Using layout to record stylesheet: "+getLayoutToRecordStylesheet().getURI());
		final Transformer layoutTransformer = saxonTransformerFactory.newTransformer(new StreamSource(getLayoutToRecordStylesheet().getInputStream()));

		final DOMResult layoutToXsltXslt = new DOMResult();
		layoutTransformer.setParameter("format", format);
		layoutTransformer.transform(new StreamSource(new StringReader(getLayoutSource(format, interpretation, layout))), layoutToXsltXslt);

		dumpXslt(layoutToXsltXslt);

		return this.resultSetFactory.xsltMap(resultSet, new DOMSource(layoutToXsltXslt.getNode()));
	}

	private String getLayoutSource(final String format, final String interpretation, final String layout) throws ISLookUpDocumentNotFoundException, ISLookUpException {
		return this.serviceLocator.getService(ISLookUpService.class).getResourceProfileByQuery(
				"collection('')//RESOURCE_PROFILE[.//RESOURCE_TYPE/@value = 'MDFormatDSResourceType' and .//NAME='" + format + "' and .//INTERPRETATION='"+interpretation+"']//LAYOUT[@name='" + layout
						+ "']");
	}

	private void dumpXslt(final DOMResult layoutToXsltXslt) throws TransformerConfigurationException, TransformerException {
		if (log.isDebugEnabled()) {
			final StringWriter buffer = new StringWriter();
			saxonTransformerFactory.newTransformer().transform(new DOMSource(layoutToXsltXslt.getNode()), new StreamResult(buffer));
			log.debug(buffer.toString());
		}
	}

	@Override
	protected BlackboardWorkflowJobListener generateBlackboardListener(final Token token) {
		return new BlackboardWorkflowJobListener(token) {

			@Override
			protected void responseToEnv(final Env env, final Map<String, String> responseParams) {
				env.setAttribute(WorkflowsConstants.MAIN_LOG_PREFIX + "indexed", responseParams.get("added"));
				env.setAttribute(WorkflowsConstants.MAIN_LOG_PREFIX + "skipped", responseParams.get("skipped"));
				env.setAttribute(WorkflowsConstants.MAIN_LOG_PREFIX + "marked", responseParams.get("marked"));
				env.setAttribute(WorkflowsConstants.MAIN_LOG_PREFIX + "time", responseParams.get("time"));
			}
		};
	}

	// setters and getters

	public String getIndexId() {
		return this.indexId;
	}

	public void setIndexId(final String indexId) {
		this.indexId = indexId;
	}

	public String getEprParam() {
		return this.eprParam;
	}

	public void setEprParam(final String eprParam) {
		this.eprParam = eprParam;
	}

	public String getFeedingType() {
		return this.feedingType;
	}

	public void setFeedingType(final String feedingType) {
		this.feedingType = feedingType;
	}

	public org.springframework.core.io.Resource getLayoutToRecordStylesheet() {
		return this.layoutToRecordStylesheet;
	}

	public void setLayoutToRecordStylesheet(final Resource layoutToRecordStylesheet) {
		this.layoutToRecordStylesheet = layoutToRecordStylesheet;
	}

	public String getFormat() {
		return this.format;
	}

	public void setFormat(final String format) {
		this.format = format;
	}

	public String getInterpretation() {
		return interpretation;
	}

	public void setInterpretation(final String interpretation) {
		this.interpretation = interpretation;
	}

	public String getLayout() {
		return this.layout;
	}

	public void setLayout(final String layout) {
		this.layout = layout;
	}

	/**
	 * @return the defaultIndexId
	 */
	public String getDefaultIndexId() {
		return this.defaultIndexId;
	}

	/**
	 * @param defaultIndexId
	 *            the defaultIndexId to set
	 */
	@Required
    public void setDefaultIndexId(final String defaultIndexId) {
        this.defaultIndexId = defaultIndexId;
    }

}
