package eu.dnetlib.msro.workflows.nodes.xmlvalidation;

import java.io.IOException;
import java.io.StringReader;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Map;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;
import javax.xml.validation.SchemaFactory;

import com.google.common.collect.Iterables;
import com.google.common.collect.Maps;
import eu.dnetlib.enabling.resultset.client.ResultSetClient;
import eu.dnetlib.enabling.resultset.factory.ResultSetFactory;
import eu.dnetlib.miscutils.functional.xml.SaxonHelper;
import eu.dnetlib.msro.workflows.graph.Arc;
import eu.dnetlib.msro.workflows.nodes.AsyncJobNode;
import eu.dnetlib.msro.workflows.procs.Env;
import eu.dnetlib.rmi.common.ResultSet;
import eu.dnetlib.rmi.manager.MSROException;
import net.sf.saxon.s9api.SaxonApiException;
import net.sf.saxon.s9api.Serializer.Property;
import net.sf.saxon.s9api.XPathSelector;
import net.sf.saxon.s9api.XdmItem;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;

/**
 * Created by alessia on 16/03/17.
 */
public class XMLSchemaValidatorJobNode extends AsyncJobNode {

	private static final Log log = LogFactory.getLog(XMLSchemaValidatorJobNode.class);

	@Autowired
	private ResultSetClient resultSetClient;
	@Autowired
	private ResultSetFactory resultSetFactory;
	@Autowired
	private SaxonHelper saxonHelper;

	private String eprParam;
	private String validEprParam;
	private boolean enableValidation;
	private boolean useDeclaredSchemaUrl;
	private String xmlSchemaURL;

	private Map<String, String> report = Maps.newHashMap();
	private Map<String,String> namespaces = Maps.newHashMap();
	private XPathSelector originalIdSelector;
	private XPathSelector metadataSelector;

	private XMLErrorHandler errorHandler = new XMLErrorHandler();

	@Override
	protected String execute(final Env env) throws Exception {
		final ResultSet<?> rsIn = env.getAttribute(this.eprParam, ResultSet.class);
		if ((rsIn == null)) { throw new MSROException("InputEprParam (" + this.eprParam + ") not found in ENV"); }
		if (!enableValidation) {
			env.setAttribute(this.validEprParam, rsIn);
			return Arc.DEFAULT_ARC;
		}

		XMLReader reader = prepareReader();
		Iterable<String> records = resultSetClient.iter(rsIn, String.class);
		addNamespaces();
		originalIdSelector = saxonHelper.help().prepareXPathSelector("//dri:recordIdentifier/text()", namespaces);
		metadataSelector = saxonHelper.help().prepareXPathSelector("//oai:metadata/*", namespaces);

			final ResultSet<String> rsOut = this.resultSetFactory.createResultSet(
					Iterables.filter(records, s -> {
						try {
						XdmItem metadataItem = saxonHelper.help().evaluateSingle(s, metadataSelector);
						return validate(saxonHelper.help().setSerializerProperty(Property.OMIT_XML_DECLARATION, "yes").evaluateSingleAsString(s, originalIdSelector), metadataItem.toString(), reader);
						} catch (SaxonApiException e) {
							throw new RuntimeException(e);
						}
					})
			);
		env.setAttribute(this.validEprParam, rsOut);
		env.setAttribute("validationReport", errorHandler.getReport() );
		return Arc.DEFAULT_ARC;
	}


	private void addNamespaces(){
		namespaces.put("dri", "http://www.driver-repository.eu/namespace/dri");
		namespaces.put("oai", "http://www.openarchives.org/OAI/2.0/");
	}

	private XMLReader prepareReader() throws ParserConfigurationException, SAXException, MSROException {
		SAXParserFactory factory = SAXParserFactory.newInstance();
		factory.setValidating(true);
		factory.setNamespaceAware(true);
		SAXParser parser = factory.newSAXParser();
		parser.setProperty("http://java.sun.com/xml/jaxp/properties/schemaLanguage",
				"http://www.w3.org/2001/XMLSchema");

		if (!useDeclaredSchemaUrl) {
			if (StringUtils.isBlank(xmlSchemaURL))
				throw new MSROException("Cannot create XMLReader for blank xml schema");
			try {
				URL url = new URL(xmlSchemaURL);
				SchemaFactory schemaFactory =
						SchemaFactory.newInstance("http://www.w3.org/2001/XMLSchema");
				factory.setSchema(schemaFactory.newSchema(url));
			} catch (MalformedURLException e) {
				throw new MSROException("XML schema url " + xmlSchemaURL + " is malformed", e);
			}
		}
		XMLReader reader = parser.getXMLReader();
		reader.setErrorHandler(errorHandler);
		return reader;
	}

	private boolean validate(final String id, final String record, final XMLReader reader) {
		InputSource inputSource = new InputSource( new StringReader( record ) );
		try {
			XMLErrorHandler err = (XMLErrorHandler) reader.getErrorHandler();
			err.setValid(true);
			err.setXmlID(id);
			reader.parse(inputSource);
			return err.isValid();
		} catch (SAXException | IOException e) {
			log.error("The record could not be validated because of " + e.getMessage());
			return false;
		}
	}


	public ResultSetClient getResultSetClient() {
		return resultSetClient;
	}

	public void setResultSetClient(final ResultSetClient resultSetClient) {
		this.resultSetClient = resultSetClient;
	}

	public String getEprParam() {
		return eprParam;
	}

	public void setEprParam(final String eprParam) {
		this.eprParam = eprParam;
	}

	public boolean isEnableValidation() {
		return enableValidation;
	}

	public void setEnableValidation(final boolean enableValidation) {
		this.enableValidation = enableValidation;
	}

	public boolean isUseDeclaredSchemaUrl() {
		return useDeclaredSchemaUrl;
	}

	public void setUseDeclaredSchemaUrl(final boolean useDeclaredSchemaUrl) {
		this.useDeclaredSchemaUrl = useDeclaredSchemaUrl;
	}

	public String getXmlSchemaURL() {
		return xmlSchemaURL;
	}

	public void setXmlSchemaURL(final String xmlSchemaURL) {
		this.xmlSchemaURL = xmlSchemaURL;
	}

	public ResultSetFactory getResultSetFactory() {
		return resultSetFactory;
	}

	public void setResultSetFactory(final ResultSetFactory resultSetFactory) {
		this.resultSetFactory = resultSetFactory;
	}

	public String getValidEprParam() {
		return validEprParam;
	}

	public void setValidEprParam(final String validEprParam) {
		this.validEprParam = validEprParam;
	}

	public Map<String, String> getReport() {
		return report;
	}

	public void setReport(final Map<String, String> report) {
		this.report = report;
	}
}
