package eu.dnetlib.msro.workflows.graph;

import java.io.StringWriter;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;

import eu.dnetlib.msro.workflows.procs.Env;

/**
 * Created by michele on 19/11/15.
 */
public class GraphNode {

	public static final String SUCCESS_NODE = "success";

	private final String name;
	private final String type;
	private final boolean isStart;
	private final boolean isJoin;
	private final boolean isSucessNode;
	private final Map<String, GraphNodeParameter> nodeParams;

	private GraphNode(final String name,
			final String type,
			final boolean isStart,
			final boolean isJoin,
			final boolean isSuccessNode,
			final Map<String, GraphNodeParameter> nodeParams) {
		this.name = name;
		this.type = type;
		this.isStart = isStart;
		this.isJoin = isJoin;
		this.isSucessNode = isSuccessNode;
		this.nodeParams = nodeParams;
	}

	public static GraphNode newNode(final String name,
			final String type,
			final Map<String, GraphNodeParameter> nodeParams) {
		return new GraphNode(name, type, false, false, false, nodeParams);
	}

	public static GraphNode newStartNode(final String name,
			final String type,
			final Map<String, GraphNodeParameter> nodeParams) {
		return new GraphNode(name, type, true, false, false, nodeParams);
	}

	public static GraphNode newJoinNode(final String name,
			final String type,
			final Map<String, GraphNodeParameter> nodeParams) {
		return new GraphNode(name, type, false, true, false, nodeParams);
	}

	public static GraphNode newSuccessNode() {
		return new GraphNode(SUCCESS_NODE, null, false, true, true, null);
	}

	public String getName() {
		return this.name;
	}

	public String getType() {
		return this.type;
	}

	public boolean isStart() {
		return this.isStart;
	}

	public boolean isJoin() {
		return this.isJoin;
	}

	public boolean isSucessNode() {
		return this.isSucessNode;
	}

	@Override
	public String toString() {
		final StringWriter sw = new StringWriter();
		sw.append("[ name: ");
		sw.append(this.name);
		if (StringUtils.isNotBlank(this.type)) {
			sw.append(", type: ");
			sw.append(this.type);
		}
		if (isStart()) {
			sw.append(" - isStart");
		}
		if (isJoin()) {
			sw.append(" - isJoin");
		}
		sw.append(" ]");
		return sw.toString();
	}

	public Map<String, GraphNodeParameter> getNodeParams() {
		return this.nodeParams;
	}

	public Map<String, Object> resolveParamsWithNoEnv() {
		return resolveParams(null);
	}

	@SuppressWarnings("unchecked")
	public Map<String, Object> resolveParams(final Env env) {
		final Map<String, Object> params = new HashMap<>();

		if (this.nodeParams != null) {

			for (final Map.Entry<String, GraphNodeParameter> e : this.nodeParams.entrySet()) {
				final String pName = e.getKey();
				final GraphNodeParameter param = e.getValue();

				if (param.isEnvParam()) {
					params.put(pName, resolveFromEnv(param, env));
				} else if (param.isMap()) {

					final Map<String, Object> map = new HashMap<>();

					for (final Map.Entry<String, GraphNodeParameter> e1 : ((Map<String, GraphNodeParameter>) param.getValue()).entrySet()) {
						map.put(e1.getKey(), e1.getValue().isEnvParam() ? resolveFromEnv(e1.getValue(), env) : e1.getValue().getValue());
					}

					params.put(pName, map);

				} else if (param.isList()) {
					params.put(pName,
							((List<GraphNodeParameter>) param.getValue())
									.stream()
									.map(p -> p.isEnvParam() ? resolveFromEnv(p, env) : p.getValue())
									.collect(Collectors.toList()));

				} else {
					params.put(pName, param.getValue());
				}

			}
		}

		return params;
	}

	private Object resolveFromEnv(final GraphNodeParameter param, final Env env) {
		return env != null ? env.getAttribute(param.getEnvVariable()) : "[this value will be resolved using the runtime ENV]";
	}

}
