package eu.dnetlib.msro.workflows.nodes.dedup;

import java.util.Queue;

import eu.dnetlib.msro.workflows.graph.Arc;
import eu.dnetlib.msro.workflows.nodes.SimpleJobNode;

import eu.dnetlib.msro.workflows.nodes.dedup.utils.DedupConfigurationOrchestration;
import eu.dnetlib.msro.workflows.nodes.dedup.utils.DedupConfigurationOrchestrationLoader;
import eu.dnetlib.msro.workflows.procs.Env;
import eu.dnetlib.rmi.manager.MSROException;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;

import com.google.common.base.Splitter;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

public class DedupCheckEntitySequenceJobNode extends SimpleJobNode {

	private static final Log log = LogFactory.getLog(DedupCheckEntitySequenceJobNode.class);

	public static final String DEDUP_GROUPER_LOOPER = "dedup.grouper.looper";

	@Autowired
	private DedupConfigurationOrchestrationLoader dedupOrchestrationLoader;

	private String dedupConfigSequenceParam;

	private String entitySequence;

	private String entityTypeNameParam;

	private String entityTypeIdParam;

	private String actionSetParam;

	@Override
	protected String execute(final Env env) throws Exception {

		if (StringUtils.isBlank(getEntitySequence())) throw new MSROException("missing entity sequence, e.g. a csv: organization,person,result");

		if (env.hasAttribute(DEDUP_GROUPER_LOOPER)) {
			log.info("reset env variable: " + DEDUP_GROUPER_LOOPER + " to zero");
			env.setAttribute(DEDUP_GROUPER_LOOPER, 0);
		}

		if (!env.hasAttribute("entitySequence")) {

			log.info("parsing config sequence: " + getEntitySequence());

			env.setAttribute("entitySequence", getEntitySequence());

			final Iterable<String> sequence = Splitter.on(",").omitEmptyStrings().split(getEntitySequence());
			final Queue<DedupConfigurationOrchestration> q =
					Lists.newLinkedList(Iterables.transform(sequence, entityName -> {
						try {
							final DedupConfigurationOrchestration dco = Iterables.getFirst(dedupOrchestrationLoader.loadByEntityName(entityName), null);
							if (dco == null) throw new RuntimeException("unable to find DedupOrchestration profile for entity type: " + entityName);
							return dco;
						} catch (final Throwable e) {
							throw new RuntimeException("", e);
						}
					}));

			log.info("built sequence of dedup orchestration profiles, size: " + q.size());
			final DedupConfigurationOrchestration dco = q.remove();
			log.info("closing mesh for entity: " + dco.getEntity().getName());
			setDedupConfParams(env, dco);
			env.setAttribute("entitySequenceQueue", q);

			return Arc.DEFAULT_ARC;
		}

		final Queue<DedupConfigurationOrchestration> q = (Queue<DedupConfigurationOrchestration>) env.getAttribute("entitySequenceQueue");

		if (!q.isEmpty()) {
			log.info("remaining dedup orchestration profiles: " + q.size());
			final DedupConfigurationOrchestration dco = q.remove();
			log.info("closing mesh for entity: " + dco.getEntity().getName());

			setDedupConfParams(env, dco);
			return Arc.DEFAULT_ARC;
		}

		log.info("completed closing mesh for entities: " + getEntitySequence());
		return "done";

	}

	private void setDedupConfParams(final Env env, final DedupConfigurationOrchestration dco) {
		env.setAttribute(getEntityTypeNameParam(), dco.getEntity().getName());
		env.setAttribute(getEntityTypeIdParam(), dco.getEntity().getCode());
		env.setAttribute(getActionSetParam(), dco.getActionSetId());
		env.setAttribute(getDedupConfigSequenceParam(), dco);
	}

	public String getEntitySequence() {
		return entitySequence;
	}

	public void setEntitySequence(final String entitySequence) {
		this.entitySequence = entitySequence;
	}

	public String getDedupConfigSequenceParam() {
		return dedupConfigSequenceParam;
	}

	public void setDedupConfigSequenceParam(final String dedupConfigSequenceParam) {
		this.dedupConfigSequenceParam = dedupConfigSequenceParam;
	}

	public String getEntityTypeNameParam() {
		return entityTypeNameParam;
	}

	public void setEntityTypeNameParam(final String entityTypeNameParam) {
		this.entityTypeNameParam = entityTypeNameParam;
	}

	public String getEntityTypeIdParam() {
		return entityTypeIdParam;
	}

	public void setEntityTypeIdParam(final String entityTypeIdParam) {
		this.entityTypeIdParam = entityTypeIdParam;
	}

	public String getActionSetParam() {
		return actionSetParam;
	}

	public void setActionSetParam(final String actionSetParam) {
		this.actionSetParam = actionSetParam;
	}
}
