package eu.dnetlib.msro.workflows.nodes.hadoop;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import eu.dnetlib.msro.workflows.graph.Arc;
import eu.dnetlib.msro.workflows.nodes.SimpleJobNode;
import eu.dnetlib.msro.workflows.procs.Env;
import eu.dnetlib.msro.workflows.util.WorkflowsConstants;
import eu.dnetlib.rmi.manager.MSROException;

public class CheckHDFSCountJobNode extends SimpleJobNode {

	private static final Log log = LogFactory.getLog(CheckHDFSCountJobNode.class); // NOPMD by marko on 11/24/08 5:02 PM

	/** Name of the env param where to find the number to check. **/
	private String numberToVerifyParamName;
	/** Name of the enc param where to find the number of records stored to hdfs. **/
	private String hdfsCounterParamName = "mainlog:storeHdfsRecords:count";

	/**
	 * {@inheritDoc}
	 *
	 * @throws MSROException
	 *
	 * @see com.googlecode.sarasvati.mem.MemNode#execute(com.googlecode.sarasvati.Engine, com.googlecode.sarasvati.NodeToken)
	 */
	@Override
	public String execute(final Env env) throws MSROException {
		int hdfsCount = 0;
		int hbaseCount = 0;
		final String numberToVerifyInEnvParam = WorkflowsConstants.BLACKBOARD_PARAM_PREFIX + this.numberToVerifyParamName;
		hbaseCount = getValueFromEnv(env, numberToVerifyInEnvParam);
		hdfsCount = getValueFromEnv(env, this.hdfsCounterParamName);

		if (hdfsCount != hbaseCount) {
			log.warn("Numbers are not the same. Number to verify (hbase count): " + hbaseCount + ". From hdfs: " + hdfsCount);
			log.warn((hdfsCount - hbaseCount) + " records come without a 'metadata' field");
			env.setAttribute("noMetadataRecords", hdfsCount - hbaseCount);
		}

		return Arc.DEFAULT_ARC;
	}

	private int getValueFromEnv(final Env env, final String paramName) throws MSROException {
		final String valueString = env.getAttribute(paramName, String.class);
		int number = 0;
		try {
			number = Integer.parseInt(valueString);
		} catch (final NumberFormatException e) {
			log.error("Env param name: " + paramName + " is " + valueString + " and cannot be parsed as integer");
			throw new MSROException(e);
		}
		return number;
	}

	public String getNumberToVerifyParamName() {
		return this.numberToVerifyParamName;
	}

	public void setNumberToVerifyParamName(final String numberToVerifyParamName) {
		this.numberToVerifyParamName = numberToVerifyParamName;
	}

	public String getHdfsCounterParamName() {
		return this.hdfsCounterParamName;
	}

	public void setHdfsCounterParamName(final String hdfsCounterParamName) {
		this.hdfsCounterParamName = hdfsCounterParamName;
	}

}
