package eu.dnetlib.msro.workflows.nodes.dedup;

import java.util.List;
import java.util.Map;
import javax.annotation.Resource;

import com.google.common.collect.Iterables;
import com.google.gson.Gson;
import eu.dnetlib.enabling.locators.UniqueServiceLocator;
import eu.dnetlib.miscutils.datetime.DateUtils;
import eu.dnetlib.msro.workflows.graph.Arc;
import eu.dnetlib.msro.workflows.nodes.SimpleJobNode;
import eu.dnetlib.msro.workflows.procs.Env;
import eu.dnetlib.rmi.enabling.ISLookUpException;
import eu.dnetlib.rmi.enabling.ISLookUpService;
import eu.dnetlib.rmi.manager.MSROException;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class PrepareDedupIndexJobNode extends SimpleJobNode {

	/**
	 * logger.
	 */
	private static final Log log = LogFactory.getLog(PrepareDedupIndexJobNode.class); // NOPMD by marko on 11/24/08 5:02 PM

	public static final String SEPARATOR = "_";

	private String mdFormat;

	private String layout;

	private String interpretation;

	private String rottenRecordsPathParam;

	private String hbaseTable;

	@Resource
	private UniqueServiceLocator serviceLocator;

	@Override
	protected String execute(final Env env) throws Exception {

		log.info("start preparing job");

		final String fields = getFields(getMdFormat(), getLayout());
		env.setAttribute("index.fields", fields);

		if (StringUtils.isBlank(getRottenRecordsPathParam())) {
			throw new MSROException("missing rotten records path param");
		}
		env.setAttribute(getRottenRecordsPathParam(), "/tmp" + getFileName("rottenrecords"));


		env.setAttribute("index.solr.url", getIndexSolrUrlZk());
		env.setAttribute("index.solr.collection", getCollectionName());

		env.setAttribute("index.shutdown.wait.time", getIndexSolrShutdownWait());
		env.setAttribute("index.buffer.flush.threshold", getIndexBufferFlushTreshold());
		env.setAttribute("index.solr.sim.mode", isFeedingSimulationMode());

		env.setAttribute("index.feed.timestamp", DateUtils.now_ISO8601());

		env.setAttribute("entityType", env.getAttribute("entityType"));
		env.setAttribute("entityTypeId", env.getAttribute("entityTypeId"));

		return Arc.DEFAULT_ARC;
	}

	private String getFields(final String format, final String layout) throws ISLookUpException {
		return isLookup(String
				.format("<FIELDS>{for $x in collection('')//RESOURCE_PROFILE[.//RESOURCE_TYPE/@value = 'MDFormatDSResourceType' and .//NAME='%s']//LAYOUT[@name='%s']/FIELDS/FIELD return $x[string(@path)]}</FIELDS>",
						format, layout));
	}

	public String getIndexSolrUrlZk() throws ISLookUpException {
		return isLookup("for $x in /RESOURCE_PROFILE[.//RESOURCE_TYPE/@value='IndexServiceResourceType'] return $x//PROTOCOL[./@name='solr']/@address/string()");
	}

	public String getIndexSolrShutdownWait() throws ISLookUpException {
		return queryForServiceProperty("solr:feedingShutdownTolerance");
	}

	public String getIndexBufferFlushTreshold() throws ISLookUpException {
		return queryForServiceProperty("solr:feedingBufferFlushThreshold");
	}

	public String isFeedingSimulationMode() throws ISLookUpException {
		return queryForServiceProperty("solr:feedingSimulationMode");
	}

	private String queryForServiceProperty(final String key) throws ISLookUpException {
		return isLookup("for $x in /RESOURCE_PROFILE[.//RESOURCE_TYPE/@value='IndexServiceResourceType'] return $x//SERVICE_PROPERTIES/PROPERTY[./@ key='"
				+ key + "']/@value/string()");
	}

	private String isLookup(final String xquery) throws ISLookUpException {
		log.debug("quering for service property: " + xquery);
		final String res = serviceLocator.getService(ISLookUpService.class).getResourceProfileByQuery(xquery);
		if (StringUtils.isBlank(res)) throw new IllegalStateException("unable to find unique service property, xquery: " + xquery);
		return res;
	}

	private String getFileName(final String fileNamePrefix) {
		return "/" + fileNamePrefix + "_" + getHbaseTable() + "_" + getMdFormat() + ".seq";
	}

	private String getCollectionName() {
		return getMdFormat() + SEPARATOR + getLayout() + SEPARATOR + getInterpretation();
	}

	public String getHbaseTable() {
		return hbaseTable;
	}

	public void setHbaseTable(final String hbaseTable) {
		this.hbaseTable = hbaseTable;
	}

	public String getRottenRecordsPathParam() {
		return rottenRecordsPathParam;
	}

	public void setRottenRecordsPathParam(final String rottenRecordsPathParam) {
		this.rottenRecordsPathParam = rottenRecordsPathParam;
	}

	public String getMdFormat() {
		return mdFormat;
	}

	public void setMdFormat(final String mdFormat) {
		this.mdFormat = mdFormat;
	}

	public String getLayout() {
		return layout;
	}

	public void setLayout(final String layout) {
		this.layout = layout;
	}

	public String getInterpretation() {
		return interpretation;
	}

	public void setInterpretation(final String interpretation) {
		this.interpretation = interpretation;
	}

}
