package eu.dnetlib.msro.workflows.nodes.dedup.utils;

import java.io.StringReader;
import java.util.List;
import java.util.Queue;

import javax.annotation.Resource;

import eu.dnetlib.rmi.enabling.ISLookUpDocumentNotFoundException;
import eu.dnetlib.rmi.enabling.ISLookUpException;
import eu.dnetlib.rmi.enabling.ISLookUpService;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;

import com.google.common.collect.Lists;

import eu.dnetlib.enabling.locators.UniqueServiceLocator;
import eu.dnetlib.pace.config.DedupConfig;

/**
 * The Class DedupConfigurationOrchestrationLoader.
 */
public class DedupConfigurationOrchestrationLoader {

	/** The Constant log. */
	private static final Log log = LogFactory.getLog(DedupConfigurationOrchestrationLoader.class);

	/** The service locator. */
	@Resource
	private UniqueServiceLocator serviceLocator;

	/**
	 * Load the dedup orchestration profile from the IS.
	 *
	 * @param id
	 *            the id
	 * @return the dedup configuration orchestration
	 * @throws ISLookUpDocumentNotFoundException
	 *             the IS look up document not found exception
	 * @throws ISLookUpException
	 *             the IS look up exception
	 * @throws DocumentException
	 *             the document exception
	 */
	public DedupConfigurationOrchestration loadByActionSetId(final String id) throws Exception {

		final ISLookUpService isLookUpService = serviceLocator.getService(ISLookUpService.class);

		final String xquery = String.format("/RESOURCE_PROFILE[.//DEDUPLICATION/ACTION_SET/@id = '%s']", id);
		log.info("loading dedup orchestration: " + xquery);

		return parseOrchestrationProfile(isLookUpService, isLookUpService.getResourceProfileByQuery(xquery));
	}

	public List<DedupConfigurationOrchestration> loadByEntityName(final String entityName) throws Exception {

		final ISLookUpService isLookUpService = serviceLocator.getService(ISLookUpService.class);

		final String xquery = String.format("/RESOURCE_PROFILE[.//DEDUPLICATION/ENTITY/@name = '%s']", entityName);
		log.info("loading dedup orchestration: " + xquery);

		final List<DedupConfigurationOrchestration> res = Lists.newArrayList();

		for (final String profile : isLookUpService.quickSearchProfile(xquery)) {
			res.add(parseOrchestrationProfile(isLookUpService, profile));
		}

		return res;
	}

	private DedupConfigurationOrchestration parseOrchestrationProfile(final ISLookUpService isLookUpService, final String dedupOrchestation)
			throws DocumentException, ISLookUpException {

		final Document doc = new SAXReader().read(new StringReader(dedupOrchestation));

		final Element e = (Element) doc.selectSingleNode("//DEDUPLICATION/ENTITY");
		final Entity entity = new Entity(e.attributeValue("name"), e.attributeValue("code"), e.attributeValue("label"));

		final String actionSetId = doc.valueOf("//DEDUPLICATION/ACTION_SET/@id");
		final Queue<DedupConfig> configurations = Lists.newLinkedList();

		for (final Object o : doc.selectNodes("//SCAN_SEQUENCE/SCAN")) {
			configurations.add(loadConfig(isLookUpService, actionSetId, o));
		}

		final DedupConfigurationOrchestration dco = new DedupConfigurationOrchestration(entity, actionSetId, configurations);

		log.debug("loaded dedup configuration orchestration: " + dco.toString());
		log.info("loaded dedup configuration orchestration, size: " + dco.getConfigurations().size());
		return dco;
	}

	private DedupConfig loadConfig(final ISLookUpService isLookUpService, final String actionSetId, final Object o) throws ISLookUpException {
		final Element s = (Element) o;
		final String configProfileId = s.attributeValue("id");
		final String conf =
				isLookUpService.getResourceProfileByQuery(String.format(
						"for $x in /RESOURCE_PROFILE[.//RESOURCE_IDENTIFIER/@value = '%s'] return $x//DEDUPLICATION/text()",
						configProfileId));
		log.debug("loaded dedup configuration from IS profile: " + conf);
		if (StringUtils.isBlank(conf)) {
			throw new IllegalStateException(String.format("got empty dedup configuration profile for '%s'", configProfileId));
		}
		final DedupConfig dedupConfig = DedupConfig.load(conf);
		dedupConfig.getWf().setConfigurationId(actionSetId);
		return dedupConfig;
	}
}
