package eu.dnetlib.msro.workflows.nodes;

import java.io.StringReader;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.Document;
import org.dom4j.io.SAXReader;
import org.springframework.beans.factory.annotation.Autowired;

import eu.dnetlib.enabling.locators.UniqueServiceLocator;
import eu.dnetlib.enabling.resultset.client.ResultSetClient;
import eu.dnetlib.msro.workflows.graph.Arc;
import eu.dnetlib.msro.workflows.procs.Env;
import eu.dnetlib.msro.workflows.procs.Token;
import eu.dnetlib.msro.workflows.util.ResultsetProgressProvider;
import eu.dnetlib.msro.workflows.util.WorkflowsConstants;
import eu.dnetlib.rmi.common.ResultSet;
import eu.dnetlib.rmi.enabling.ISRegistryService;

public class UpdateRepositoryProfilesJobNode extends SimpleJobNode {

	private static final String REPOSITORY_SERVICE_RESOURCE_TYPE = "RepositoryServiceResourceType";

	@Autowired
	private ResultSetClient resultSetClient;

	@Autowired
	private UniqueServiceLocator serviceLocator;

	private String eprParam = "repoEpr";
	private String existingReposParam = "existingRepos";

	private static final Log log = LogFactory.getLog(UpdateRepositoryProfilesJobNode.class); // NOPMD by marko on 11/24/08 5:02 PM

	@Override
	protected void beforeStart(final Token token) {
		final ResultSet<?> epr = token.getEnv().getAttribute(getEprParam(), ResultSet.class);
		token.setProgressProvider(new ResultsetProgressProvider(epr, this.resultSetClient));
	}

	@Override
	protected String execute(final Env env) throws Exception {
		// datasourceID -> profileID
		@SuppressWarnings("unchecked")
		final Map<String, String> existingRepos = env.getAttribute(getExistingReposParam(), Map.class);
		final ResultSet<?> epr = env.getAttribute(getEprParam(), ResultSet.class);

		final SAXReader reader = new SAXReader();

		int countUpdated = 0;
		int countInserted = 0;
		int countProfileErrors = 0;

		log.info("Adding/updating repository profiles...");

		final ISRegistryService registry = this.serviceLocator.getService(ISRegistryService.class);

		for (final String profile : this.resultSetClient.iter(epr, String.class)) {
			try {
				final Document doc = reader.read(new StringReader(profile));
				final String dsId = doc.valueOf("//EXTRA_FIELDS/FIELD[./key='OpenAireDataSourceId']/value");

				log.debug("Registering/updating profile:\n " + profile + "\n");

				if (existingRepos.containsKey(dsId)) {
					final String profId = existingRepos.get(dsId);
					doc.selectSingleNode("//RESOURCE_IDENTIFIER/@value").setText(profId);
					registry.updateProfile(profId, doc.asXML(), REPOSITORY_SERVICE_RESOURCE_TYPE);
					log.debug("Profile " + profId + " UPDATED for ds " + dsId);
					countUpdated++;
				} else {
					final String profId = registry.registerProfile(doc.asXML());
					log.debug("Valid Profile " + profId + " REGISTERED for ds " + dsId);
					countInserted++;
				}
			} catch (final Exception e) {
				log.error("INVALID PROFILE: " + profile, e);
				countProfileErrors++;
			}
		}

		log.info("   - updated_profiles: " + countUpdated);
		log.info("   - inserted_profiles: " + countInserted);
		log.info("   - profiles_with_errors: " + countProfileErrors);

		env.setAttribute(WorkflowsConstants.MAIN_LOG_PREFIX + "updated_profiles", countUpdated);
		env.setAttribute(WorkflowsConstants.MAIN_LOG_PREFIX + "inserted_profiles", countInserted);
		env.setAttribute(WorkflowsConstants.MAIN_LOG_PREFIX + "profiles_with_errors", countProfileErrors);

		return Arc.DEFAULT_ARC;
	}

	public String getEprParam() {
		return this.eprParam;
	}

	public void setEprParam(final String eprParam) {
		this.eprParam = eprParam;
	}

	public String getExistingReposParam() {
		return this.existingReposParam;
	}

	public void setExistingReposParam(final String existingReposParam) {
		this.existingReposParam = existingReposParam;
	}

}
