package eu.dnetlib.msro.workflows.nodes.index;

import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;
import java.io.StringWriter;

import javax.annotation.Resource;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.ClassPathResource;

import eu.dnetlib.enabling.locators.UniqueServiceLocator;
import eu.dnetlib.miscutils.datetime.DateUtils;
import eu.dnetlib.miscutils.functional.hash.Hashing;
import eu.dnetlib.msro.workflows.graph.Arc;
import eu.dnetlib.msro.workflows.nodes.SimpleJobNode;
import eu.dnetlib.msro.workflows.procs.Env;
import eu.dnetlib.rmi.enabling.ISLookUpDocumentNotFoundException;
import eu.dnetlib.rmi.enabling.ISLookUpException;
import eu.dnetlib.rmi.enabling.ISLookUpService;

public class PrepareIndexDataJobNode extends SimpleJobNode {

	/**
	 * logger.
	 */
	private static final Log log = LogFactory.getLog(PrepareIndexDataJobNode.class); // NOPMD by marko on 11/24/08 5:02 PM

	public static final String SEPARATOR = "_";

	@Resource
	private UniqueServiceLocator serviceLocator;

	/**
	 * Stylesheet which transforms a layout to another stylesheet which converts a input record to a index record.
	 */
	private String layoutToRecordStylesheet;

	private String outputRecordsPathParam;

	private String rottenRecordsPathParam;

	private String hbaseTable;

	private String oafSchemaLocation;

	@Override
	protected String execute(final Env env) throws Exception {

		log.info("start preparing job");

		final String xslt = prepareXslt(env("format", env), env("layout", env));

		env.setAttribute("index.xslt", xslt);

		if (!StringUtils.isBlank(getOutputRecordsPathParam())) {
			env.setAttribute(getOutputRecordsPathParam(), "/tmp" + getFileName(env, "indexrecords"));
		}
		if (!StringUtils.isBlank(getRottenRecordsPathParam())) {
			env.setAttribute(getRottenRecordsPathParam(), "/tmp" + getFileName(env, "rottenrecords"));
		}

		env.setAttribute("index.solr.url", getIndexSolrUrlZk());
		env.setAttribute("index.solr.collection", getCollectionName(env));

		env.setAttribute("index.shutdown.wait.time", getIndexSolrShutdownWait());
		env.setAttribute("index.buffer.flush.threshold", getIndexBufferFlushTreshold());
		env.setAttribute("index.solr.sim.mode", isFeedingSimulationMode());

		env.setAttribute("index.feed.timestamp", DateUtils.now_ISO8601());

		env.setAttribute("", this.oafSchemaLocation); // TODO : inserire il nome della property

		return Arc.DEFAULT_ARC;
	}

	public String getIndexSolrUrlZk() throws ISLookUpException {
		return getServiceConfigValue(
				"for $x in /RESOURCE_PROFILE[.//RESOURCE_TYPE/@value='IndexServiceResourceType'] return $x//PROTOCOL[./@name='solr']/@address/string()");
	}

	public String getIndexSolrShutdownWait() throws ISLookUpException {
		return queryForServiceProperty("solr:feedingShutdownTolerance");
	}

	public String getIndexBufferFlushTreshold() throws ISLookUpException {
		return queryForServiceProperty("solr:feedingBufferFlushThreshold");
	}

	public String isFeedingSimulationMode() throws ISLookUpException {
		return queryForServiceProperty("solr:feedingSimulationMode");
	}

	private String queryForServiceProperty(final String key) throws ISLookUpException {
		return getServiceConfigValue(
				"for $x in /RESOURCE_PROFILE[.//RESOURCE_TYPE/@value='IndexServiceResourceType'] return $x//SERVICE_PROPERTIES/PROPERTY[./@ key='"
						+ key + "']/@value/string()");
	}

	private String getServiceConfigValue(final String xquery) throws ISLookUpException {
		log.debug("quering for service property: " + xquery);
		final String res = this.serviceLocator.getService(ISLookUpService.class).getResourceProfileByQuery(xquery);
		if (StringUtils.isBlank(res)) { throw new IllegalStateException("unable to find unique service property, xquery: " + xquery); }
		return res;
	}

	private String getFileName(final Env env, final String fileNamePrefix) {
		return "/" + fileNamePrefix + "_" + getHbaseTable() + "_" + env.getAttribute("format") + ".seq";
	}

	private String getCollectionName(final Env env) {
		return env("format", env) + SEPARATOR + env("layout", env) + SEPARATOR + env("interpretation", env);
	}

	private String env(final String s, final Env env) {
		return env.getAttribute(s, String.class);
	}

	protected String prepareXslt(final String format, final String layout) throws ISLookUpException, IOException, TransformerException {

		final TransformerFactory factory = TransformerFactory.newInstance();
		final Transformer layoutTransformer = factory.newTransformer(new StreamSource(new StringReader(readXslt(getLayoutToRecordStylesheet()))));

		final StreamResult layoutToXsltXslt = new StreamResult(new StringWriter());

		layoutTransformer.setParameter("format", format);
		layoutTransformer.transform(new StreamSource(new StringReader(getLayoutSource(format, layout))), layoutToXsltXslt);

		return new String(Hashing.encodeBase64(layoutToXsltXslt.getWriter().toString()));
	}

	private String readXslt(final String s) throws IOException {
		final ClassPathResource resource = new ClassPathResource(s);
		final InputStream inputStream = resource.getInputStream();
		return IOUtils.toString(inputStream);
	}

	private String getLayoutSource(final String format, final String layout) throws ISLookUpDocumentNotFoundException, ISLookUpException {
		return this.serviceLocator.getService(ISLookUpService.class).getResourceProfileByQuery(
				"collection('')//RESOURCE_PROFILE[.//RESOURCE_TYPE/@value = 'MDFormatDSResourceType' and .//NAME='" + format + "']//LAYOUT[@name='" + layout
						+ "']");
	}

	public String getLayoutToRecordStylesheet() {
		return this.layoutToRecordStylesheet;
	}

	public void setLayoutToRecordStylesheet(final String layoutToRecordStylesheet) {
		this.layoutToRecordStylesheet = layoutToRecordStylesheet;
	}

	public String getHbaseTable() {
		return this.hbaseTable;
	}

	@Required
	public void setHbaseTable(final String hbaseTable) {
		this.hbaseTable = hbaseTable;
	}

	public String getOutputRecordsPathParam() {
		return this.outputRecordsPathParam;
	}

	public void setOutputRecordsPathParam(final String outputRecordsPathParam) {
		this.outputRecordsPathParam = outputRecordsPathParam;
	}

	public String getRottenRecordsPathParam() {
		return this.rottenRecordsPathParam;
	}

	public void setRottenRecordsPathParam(final String rottenRecordsPathParam) {
		this.rottenRecordsPathParam = rottenRecordsPathParam;
	}

	public void setOafSchemaLocation(final String oafSchemaLocation) {
		this.oafSchemaLocation = oafSchemaLocation;
	}
}
