package eu.dnetlib.pace.distance.algo;

import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import com.google.common.base.Function;
import com.google.common.base.Predicates;
import com.google.common.collect.Iterables;
import com.google.common.collect.Sets;
import com.google.common.collect.Sets.SetView;
import eu.dnetlib.pace.distance.ConfigurableDistanceAlgo;
import eu.dnetlib.pace.distance.DistanceAlgo;
import eu.dnetlib.pace.model.Field;
import eu.dnetlib.pace.model.FieldListImpl;
import eu.dnetlib.pace.model.gt.CoAuthor;
import eu.dnetlib.pace.model.gt.CoAuthors;
import eu.dnetlib.pace.model.gt.GTAuthor;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class PersonDistance extends ConfigurableDistanceAlgo implements DistanceAlgo {

	/**
	 * logger.
	 */
	private static final Log log = LogFactory.getLog(PersonDistance.class); // NOPMD by marko on 11/24/08 5:02 PM

	private Integer commonAnchors = null;

	private Integer commonSurnames = null;

	public PersonDistance(final Map<String, String> params, final double weight) {
		super(params, weight);

		final String commonAnchors = getParams().get("common.anchors");
		this.commonAnchors = isEnabled(commonAnchors) ? Integer.parseInt(commonAnchors) : null;
		log.debug("min common anchors: " + commonAnchors);

		final String commonSurnames = getParams().get("common.surnames");
		this.commonSurnames = isEnabled(commonSurnames) ? Integer.parseInt(commonSurnames) : null;
		log.debug("min common surnames: " + commonSurnames);
	}

	@Override
	public double distance(final Field a, final Field b) {

		final GTAuthor ga = Iterables.getLast(GTAuthor.fromDNGFJson(((FieldListImpl) a).stringList()));
		final GTAuthor gb = Iterables.getLast(GTAuthor.fromDNGFJson(((FieldListImpl) b).stringList()));

		if (commonAnchors != null && (anchorsInCommon(ga, gb).size() >= commonAnchors)) // log.info("matched coauthor intersection");
			return 1.0;

		if (commonSurnames != null && (surnamesInCommon(ga, gb).size() >= commonSurnames))
			return 1.0;

		return 0;
	}

	private boolean isEnabled(final String property) {
		return (property != null) && (Integer.parseInt(property) >= 0);
	}

	private Set<String> anchorsInCommon(final GTAuthor a, final GTAuthor b) {
		final SetView<String> set = Sets.intersection(getAnchorIds(a.getCoAuthors()), getAnchorIds(b.getCoAuthors()));
		if (log.isDebugEnabled()) {
			log.debug(
					String.format("anchors intersection '%s' - '%s': size: %s, %s", a.getAuthor().getFullname(), b.getAuthor().getFullname(), set.size(), set));
		}
		return set;
	}

	private Set<String> getAnchorIds(final CoAuthors ca) {
		if ((ca == null) || ca.isEmpty()) return new HashSet<String>();
		final Iterable<String> anchorIds = Iterables.transform(ca, new Function<CoAuthor, String>() {

			@Override
			public String apply(final CoAuthor c) {
				return c.getAnchorId();
			}
		});
		final Iterable<String> filtered = Iterables.filter(anchorIds, Predicates.notNull());
		return Sets.newHashSet(filtered);
	}

	private Set<String> surnamesInCommon(final GTAuthor a, final GTAuthor b) {
		final SetView<String> set = Sets.intersection(getSurnames(a.getCoAuthors()), getSurnames(b.getCoAuthors()));
		if (log.isDebugEnabled()) {
			log.debug(String.format("surnames intersection '%s' - '%s' size: %s, %s", a.getAuthor().getFullname(), b.getAuthor().getFullname(),
					set.size(), set));
		}
		return set;
	}

	private Set<String> getSurnames(final CoAuthors ca) {
		if ((ca == null) || ca.isEmpty()) return new HashSet<String>();
		return Sets.newHashSet(Iterables.filter(Iterables.transform(ca, new Function<CoAuthor, String>() {

			@Override
			public String apply(final CoAuthor c) {
				return c.getSecondnames();
			}
		}), Predicates.notNull()));
	}

	@Override
	public double getWeight() {
		return getWeigth();
	}

}
