package eu.dnetlib.parthenos.virtuoso;

import java.util.Map;

import com.google.common.collect.Maps;
import eu.dnetlib.miscutils.functional.xml.SaxonHelper;
import net.sf.saxon.s9api.SaxonApiException;
import net.sf.saxon.s9api.Serializer.Property;
import net.sf.saxon.s9api.XPathSelector;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.jena.rdf.model.Model;
import org.apache.jena.rdf.model.Resource;
import org.apache.jena.rdf.model.ResourceFactory;
import org.apache.jena.rdf.model.Statement;
import virtuoso.jena.driver.VirtModel;

/**
 * Created by Alessia Bardi on 12/07/2017.
 *
 * @author Alessia Bardi
 */
public class VirtuosoClient {

	private static final Log log = LogFactory.getLog(VirtuosoClient.class);

	private static final String OAI_NAMESPACE_URI = "http://www.openarchives.org/OAI/2.0/";
	private static final String DRI_NAMESPACE_URI = "http://www.driver-repository.eu/namespace/dri";
	private static final String RDF_NAMESPACE_URI = "http://www.w3.org/1999/02/22-rdf-syntax-ns#";

	private SaxonHelper saxonHelper;
	private XPathSelector xpathSelectorObjIdentifier;
	private XPathSelector xpathSelectorCollectionDate;
	private XPathSelector xpathSelectorTransformationDate;
	private XPathSelector xpathSelectorDatasourceName;
	private XPathSelector xpathSelectorRDF;

	private String connectionString;
	private String username;
	private String password;
	private String defaultBaseURI;

	protected VirtuosoClient(final String connectionString,
			final String username,
			final String password,
			final SaxonHelper saxonHelper,
			final String defaultBaseURI)
			throws SaxonApiException {
		this.connectionString = connectionString;
		this.username = username;
		this.password = password;
		this.saxonHelper = saxonHelper;
		this.defaultBaseURI = defaultBaseURI;
		prepareXpathSelectors();
	}

	public long feed(final String record) {
		if (StringUtils.isBlank(record)) {
			log.warn("Got empty record");
			return 0;
		}
		String objIdentifier = extractFromRecord(record, xpathSelectorObjIdentifier);
		if (StringUtils.isBlank(objIdentifier)) {
			log.warn("Got record with no objIdentifier -- skipping");
			return 0;
		}
		String rdfBlock = extractFromRecord(record, xpathSelectorRDF);
		if (StringUtils.isBlank(rdfBlock)) {
			log.warn("Missing rdf:RDF in record with objIdentifier " + objIdentifier + " all triples in that named graph will be deleted");
		}
		String collectionDate = extractFromRecord(record, xpathSelectorCollectionDate);
		String transformationDate = extractFromRecord(record, xpathSelectorTransformationDate);
		String datasource = extractFromRecord(record, xpathSelectorDatasourceName);

		Model md = VirtModel.openDatabaseModel(objIdentifier, getConnectionString(), getUsername(), getPassword());
		log.debug("Opened virtuoso model for graph " + objIdentifier);
		md.removeAll();
		log.debug("Removed all triples from graph " + objIdentifier);
		md.read(IOUtils.toInputStream(rdfBlock), getDefaultBaseURI());
		long size = md.size();
		log.info("Graph " + objIdentifier + " now has " + size + " triples");

		long ntriples = feedProvenance(objIdentifier, collectionDate, transformationDate, datasource);
		log.debug("provenance graph updated with " + ntriples + " triples");

		return size;
	}

	long feedProvenance(final String objIdentifier, final String collectionDate, final String transformationDate, final String datasource) {
		Model md = VirtModel.openDatabaseModel("provenance", getConnectionString(), getUsername(), getPassword());

		//TODO: use prov-o instead: https://www.w3.org/TR/prov-o/#description
		Resource r = ResourceFactory.createResource(getRecordDefaultURI(objIdentifier, datasource));
		Statement stmCollFrom =
				ResourceFactory.createStatement(r, ResourceFactory.createProperty("dnet", "collectedFrom"), ResourceFactory.createPlainLiteral(datasource));
		//TODO: how to set the type of dates correctly?
		Statement stmCollDate = ResourceFactory
				.createStatement(r, ResourceFactory.createProperty("dnet", "collectedInDate"), ResourceFactory.createPlainLiteral(collectionDate));
		Statement stmTransDate = ResourceFactory
				.createStatement(r, ResourceFactory.createProperty("dnet", "transformedInDate"), ResourceFactory.createPlainLiteral(transformationDate));

		//let's remove previous provenance statements for this resource:
		md.removeAll(r, null, null);
		//and add the new ones
		md.add(stmCollDate).add(stmCollFrom).add(stmTransDate);
		return 3;
	}

	public long feed(final Iterable<String> records) {
		//TODO: can we do it in parallel? if all records have different objIdentifier it is safe, and this must be the case anyway, because the source of records is a D-Net mdstore.
		long count = 0;
		for (String r : records) count += this.feed(r);
		return count;
	}

	private String getRecordDefaultURI(final String objIdentifier, final String datasource) {
		return defaultBaseURI + datasource + "/" + objIdentifier;
	}

	private void prepareXpathSelectors() throws SaxonApiException {
		Map<String, String> namespaces = Maps.newHashMap();
		namespaces.put("oai", OAI_NAMESPACE_URI);
		namespaces.put("dri", DRI_NAMESPACE_URI);
		namespaces.put("rdf", RDF_NAMESPACE_URI);
		xpathSelectorObjIdentifier = this.saxonHelper.help().prepareXPathSelector("//oai:header/dri:objIdentifier/text()", namespaces);
		xpathSelectorCollectionDate = this.saxonHelper.help().prepareXPathSelector("//oai:header/dri:dateOfCollection/text()", namespaces);
		xpathSelectorTransformationDate = this.saxonHelper.help().prepareXPathSelector("//oai:header/dri:dateOfTransformation/text()", namespaces);
		xpathSelectorDatasourceName = this.saxonHelper.help().prepareXPathSelector("//oai:header/dri:datasourcename/text()", namespaces);
		xpathSelectorRDF = this.saxonHelper.help().prepareXPathSelector("//oai:metadata/rdf:RDF", namespaces);
	}

	private String extractFromRecord(final String record, final XPathSelector xPathSelector) {
		try {
			return this.saxonHelper.help().setSerializerProperty(Property.OMIT_XML_DECLARATION, "yes").evaluateSingleAsString(record, xPathSelector);
		} catch (SaxonApiException e) {
			throw new RuntimeException("Cannot extract content ", e);
		}
	}

	public String getConnectionString() {
		return connectionString;
	}

	public String getUsername() {
		return username;
	}

	public String getPassword() {
		return password;
	}

	public SaxonHelper getSaxonHelper() {
		return saxonHelper;
	}

	public String getDefaultBaseURI() {
		return defaultBaseURI;
	}
}
