package eu.dnetlib.parthenos.virtuoso;

import java.io.IOException;
import java.io.OutputStream;
import java.io.StringWriter;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import com.google.common.collect.Iterators;
import com.google.common.collect.Lists;
import eu.dnetlib.parthenos.CRM;
import eu.dnetlib.parthenos.publisher.ParthenosPublisherException;
import freemarker.template.Configuration;
import freemarker.template.Template;
import freemarker.template.TemplateException;
import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.http.HttpEntity;
import org.apache.http.HttpHeaders;
import org.apache.http.NameValuePair;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.util.EntityUtils;
import org.apache.jena.query.Query;
import org.apache.jena.query.QueryExecutionFactory;
import org.apache.jena.query.QueryFactory;
import org.apache.jena.query.ResultSet;
import org.apache.jena.sparql.engine.http.QueryEngineHTTP;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpStatus;
import org.springframework.web.bind.annotation.*;

/**
 * Created by Alessia Bardi on 31/01/2018.
 * Read-only API for virtuoso.
 *
 * //TODO: error handling (http://www.springboottutorial.com/spring-boot-exception-handling-for-rest-services)
 * //TODO: pagination
 * //TODO swagger documentation?
 *
 * @author Alessia Bardi
 */
@RestController
public class VirtuosoReadAPI {

	private static final Log log = LogFactory.getLog(VirtuosoReadAPI.class);

	@Value("${virtuoso.sparqlurl}")
	private String sparqlUrl;
	@Value("${virtuoso.pwd}")
	private String username;
	@Value("${virtuoso.pwd}")
	private String password;
	@Value("${virtuoso.uri.base.default}")
	private String defaultBaseURI;
	@Value("${virtuoso.connectionstring")
	private String virtuosoConnectionString;

	@Autowired
	private Configuration freemarkerConfig;

	@RequestMapping(value = "/virtuoso/apiSubjectsWithType", produces = { "application/json" }, method = RequestMethod.GET)
	public List<String> getSubjectsForApiWithType(@RequestParam final String api, @RequestParam final String typeNamespace, @RequestParam final String typeName, @RequestParam final int limit, @RequestParam final int offset){
		String fullTypeName = typeNamespace + typeName;
		log.debug(String.format("Getting subjects of type %s for API %s limit %d offset %d", fullTypeName, api, limit, offset));
		//if I add the ORDER BY the query is too slow: let's hope we are not getting the same subjects over and over again
		String queryForSubjectsTemplate = "DEFINE input:inference 'parthenos_rules' SELECT DISTINCT ?s WHERE { GRAPH ?g {?s a <%s> .} . GRAPH <provenance> {?g <dnetcollectedFrom> <%s> .}} LIMIT %d OFFSET %d";
		String q = String.format(queryForSubjectsTemplate, fullTypeName, defaultBaseURI+api, limit, offset);
		final QueryEngineHTTP serviceRequest = new QueryEngineHTTP(sparqlUrl, q);
		ResultSet subjects = serviceRequest.execSelect();
		Iterator<String> s = Iterators.transform(subjects, qs -> qs.getResource("s").getURI());
		List<String> res = Lists.newArrayList(s);
		serviceRequest.close();
		return res;
	}

	@RequestMapping(value = "/virtuoso/apiSubjects", produces = { "application/json" }, method = RequestMethod.GET)
	public List<String> getSubjectsForApi(@RequestParam final String api, @RequestParam final int limit, @RequestParam final int offset){
		log.debug(String.format("Getting subjects for API %s, limit %d offset %d", api, limit, offset));
		//if I add the ORDER BY the query is too slow: let's hope we are not getting the same subjects over and over again
		String queryForSubjectsTemplate = "DEFINE input:inference 'parthenos_rules' SELECT DISTINCT ?s WHERE { GRAPH ?g {?s a ?t . FILTER (?t != <%s> && ?t != <%s>)} . GRAPH <provenance> {?g <dnetcollectedFrom> <%s> .}} LIMIT %d OFFSET %d ";
		String q = String.format(queryForSubjectsTemplate, CRM.E55_Type.getURI(), CRM.E41_Appellation.getURI(), defaultBaseURI+api, limit, offset);
		final QueryEngineHTTP serviceRequest = new QueryEngineHTTP(sparqlUrl, q);
		ResultSet subjects = serviceRequest.execSelect();
		Iterator<String> s = Iterators.transform(subjects, qs -> qs.getResource("s").getURI());
		List<String> res = Lists.newArrayList(s);
		serviceRequest.close();
		return res;
	}



	@Deprecated
	@RequestMapping(value = "/virtuoso/graphs", produces = { "application/json" }, method = RequestMethod.GET)
	public List<String> getGraphURLs(@RequestParam final String api) {
		String queryForGraphs =
				"SELECT DISTINCT ?g WHERE { ?g <dnetcollectedFrom> <"+defaultBaseURI+api+"> }";
		Query query = QueryFactory.create(queryForGraphs);
		final QueryEngineHTTP serviceRequest = QueryExecutionFactory.createServiceRequest(sparqlUrl, query);
		ResultSet graphs = serviceRequest.execSelect();
		Iterator<String> s = Iterators.transform(graphs, qs -> qs.getResource("g").getURI());
		List<String> res = Lists.newArrayList(s);
		serviceRequest.close();
		return res;
	}

	@Deprecated
	@RequestMapping(value = "/virtuoso/subjects", produces = { "application/json" }, method = RequestMethod.GET)
	public List<String> getSubjectsForGraph(@RequestParam final String graph) {
		// Based on FORTH observation, resources without a URL should not be interesting by themselves.
		// Here we filter out those + resources whose type is not in the parthenos namespace.

		//old query selecting any subjects in the given graph
		//String queryForSubjectsTemplate = "SELECT DISTINCT ?s WHERE {GRAPH <%s> {?s ?p ?o .}}";
		String queryForSubjectsTemplate = "SELECT DISTINCT ?s WHERE {GRAPH <%s> {?s ?p ?o . ?s a ?t. FILTER (isURI(?s) && ?t != <%s> && ?t != <%s>) }}";
		String q = String.format(queryForSubjectsTemplate, graph, CRM.E55_Type.getURI(), CRM.E41_Appellation.getURI());
		final QueryEngineHTTP serviceRequest =
				QueryExecutionFactory.createServiceRequest(sparqlUrl, QueryFactory.create(q));
		ResultSet subjects = serviceRequest.execSelect();
		Iterator<String> s = Iterators.transform(subjects, qs -> qs.getResource("s").getURI());
		List<String> res = Lists.newArrayList(s);
		serviceRequest.close();
		return res;
	}

	@Deprecated
	@RequestMapping(value = "/virtuoso/subjectsWithType", produces = { "application/json" }, method = RequestMethod.GET)
	public List<String> getSubjectsForGraphWithType(@RequestParam final String graph, @RequestParam final String typeNamespace, @RequestParam final String typeName) {
		String fullTypeName = typeNamespace + typeName;
		log.debug(String.format("Getting subjects of type %s for graph %s", fullTypeName, graph));
		String queryForSubjectsTemplate = "DEFINE input:inference 'parthenos_rules' SELECT ?s WHERE {GRAPH <%s> {?s a <%s>. FILTER (isURI(?s)) }}";
		String q = String.format(queryForSubjectsTemplate, graph, fullTypeName);
		final QueryEngineHTTP serviceRequest = new QueryEngineHTTP(sparqlUrl, q);
		ResultSet subjects = serviceRequest.execSelect();
		Iterator<String> s = Iterators.transform(subjects, qs -> qs.getResource("s").getURI());
		List<String> res = Lists.newArrayList(s);
		serviceRequest.close();
		return res;
	}

	@RequestMapping(value = "/virtuoso/subject", produces = { "application/rdf+xml", "application/xml" }, method = RequestMethod.GET)
	@ResponseStatus(value = HttpStatus.OK)
	public void getSubject(@RequestParam final String subjectURL, final OutputStream responseStream) throws IOException, ParthenosPublisherException {
		//String describeQuery = "DESCRIBE <" + subjectURL + ">";
		//log.debug(describeQuery);
		//Can't use DESCRIBE query: server decides what to return (i.e. not all triples) and cannot find out if and how Virtuoso can be configured

		String q = "DEFINE input:inference 'parthenos_rules' CONSTRUCT {<%s> ?p ?o . } WHERE { <%s> ?p ?o .}";
		String query = String.format(q, subjectURL, subjectURL);
		sendConstructResponse(query, responseStream);
	}

	@RequestMapping(value = "/virtuoso/subjectForJRR", produces = { "application/rdf+xml", "application/xml" }, method = RequestMethod.GET)
	@ResponseStatus(value = HttpStatus.OK)
	public void getSubjectForJRR(@RequestParam final String subjectURL, @RequestParam final String typeName, final OutputStream responseStream)
			throws IOException, TemplateException, ParthenosPublisherException {
		String templateName = typeName+".sparql";
		Template temp = freemarkerConfig.getTemplate(templateName);
		Map<String, String> values = new HashMap<>();
		values.put("subjectURL", subjectURL);

		StringWriter sw = new StringWriter();
		temp.process(values, sw);
		String q = sw.toString();
		log.debug("Querying for "+subjectURL+" with query "+templateName);
		sendConstructResponse(q, responseStream);
	}


	protected void sendConstructResponse(final String query, final OutputStream responseStream) throws IOException, ParthenosPublisherException {
		String res = executeSparqlPost(query);
		IOUtils.write(res, responseStream);
	}

	protected String executeSparqlPost(final String query) throws IOException, ParthenosPublisherException {
		try (CloseableHttpClient httpclient = HttpClients.createDefault()) {
			HttpPost httpPost = new HttpPost(sparqlUrl);
			httpPost.setHeader(HttpHeaders.ACCEPT, "application/rdf+xml");
			List<NameValuePair> nvps = Lists.newArrayList();
			nvps.add(new BasicNameValuePair("query", query));
			httpPost.setEntity(new UrlEncodedFormEntity(nvps));

			try (CloseableHttpResponse response2 = httpclient.execute(httpPost)) {
				HttpEntity entity2 = response2.getEntity();
				String res = IOUtils.toString(entity2.getContent());
				EntityUtils.consume(entity2);
				int statusCode = response2.getStatusLine().getStatusCode();
				switch(statusCode){
				case 200:
					return res;
				case 504:
					String msg = String.format("ERROR 504 on query %s", query);
					throw new ParthenosPublisherException(msg);
				default:
					String errorMessage = String.format("ERROR HTTP STATUS CODE %d, REASON PHRASE: %s\n ERROR BODY: %s", statusCode, response2.getStatusLine(), res);
					log.error(errorMessage);
					throw new ParthenosPublisherException(errorMessage);
				}
			}
		}


	}
	




}
