package eu.dnetlib.parthenos.registry;

import java.io.IOException;

import com.fasterxml.jackson.core.JsonGenerator;
import com.google.common.base.Joiner;
import eu.dnetlib.parthenos.CRM;
import eu.dnetlib.parthenos.CRMpe;
import eu.dnetlib.parthenos.rdf.ResourceReader;
import org.apache.commons.lang3.StringUtils;
import org.apache.jena.rdf.model.RDFNode;
import org.apache.jena.rdf.model.Resource;
import org.apache.jena.rdf.model.StmtIterator;
import org.apache.jena.vocabulary.RDF;
import org.gcube.informationsystem.model.entity.facet.IdentifierFacet.IdentificationType;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

/**
 * Created by Alessia Bardi on 02/10/2017.
 *
 * @author Alessia Bardi
 */
@Component
public class FacetWriter {

	@Autowired
	private ResourceReader resourceReader;

	//final IdentificationType type
	public void writeP1Facet(final JsonGenerator jg, final String value) throws IOException {
		jg.writeStartObject();
		jg.writeStringField("@class", "P1_is_identified_by");
		jg.writeObjectFieldStart("target");
		jg.writeStringField("@class", "SimpleFacet");
		jg.writeStringField("value", value);
		jg.writeEndObject();
		jg.writeEndObject();
	}

	public void writeIdentifierFacet(final JsonGenerator jg, final String identifier) throws IOException {
		jg.writeStartObject();
		jg.writeStringField("@class", "isIdentifiedBy");
		jg.writeObjectFieldStart("target");
		jg.writeStringField("@class", "IdentifierFacet");
		jg.writeStringField("value", identifier);
		jg.writeStringField("type", IdentificationType.URI.name());
		jg.writeEndObject();
		jg.writeEndObject();
	}

	public void writeP1Facets(final JsonGenerator jg, final Resource res) throws IOException {
		if(res.hasProperty(CRM.P1_is_identified_by)){
			StmtIterator idRelsIt = res.listProperties(CRM.P1_is_identified_by);
			while(idRelsIt.hasNext()){
				RDFNode obj = idRelsIt.nextStatement().getObject();
				String value = "";
				if(obj.isLiteral()){
					value = obj.asLiteral().getLexicalForm();
				}
				else if(obj.isResource()) {
					value = resourceReader.getLabel(obj.asResource());
				}
				if(StringUtils.isNotBlank(value)) writeP1Facet(jg, value);
			}
		}
	}

	public void writeInfoFacet(final JsonGenerator jg, final Resource res) throws IOException {
		jg.writeStartObject();
		jg.writeStringField("@class", "consistsOf");
		jg.writeObjectFieldStart("target");
		jg.writeStringField("@class", "PE_Info_Facet");
		jg.writeStringField("title", resourceReader.getTitle(res));
		jg.writeStringField("description", Joiner.on(",").join(resourceReader.getDescriptions(res)));
		//new object comp + field value + schema="noSchema"
		String competence = Joiner.on(',').join(resourceReader.getCompetences(res));
		if(StringUtils.isNotBlank(competence)) {
			writeValueSchema(jg, "competence", competence, "noSchema");
		}
		//TODO: uncomment this when George adds the rel to the model
		//String availability = getAvailabilityFromRDFResource(res);
//		if(StringUtils.isNotBlank(availability)) {
//			writeValueSchema(jg, "availability", availability, "noSchema");
//		}
		jg.writeEndObject();
		jg.writeEndObject();
	}

	protected void writeValueSchema(final JsonGenerator jg, final String fieldName, final String value, final String schema) throws IOException {
		jg.writeObjectFieldStart(fieldName);
		jg.writeStringField("value", value);
		jg.writeStringField("schema", schema);
		jg.writeEndObject();
	}

	public void writeEventFacet(final JsonGenerator jg) {
		//TODO: implement me. get begin/end of operation from PP42_has_declarative_time
	}

	public void writeRightsFacet(final JsonGenerator jg, final Resource res) throws IOException {
		//TODO: implement me. E30_Right facet extends from licenseFacet but it is not correct (textUrl is mandatory, we can't use it in Parthenos)
	}

	public void writeContactReferenceFacet(final JsonGenerator jg, final Resource actor) throws IOException {
		if (actor != null) {
			String appellation = resourceReader.getTitle(actor);
			String description = Joiner.on(',').join(resourceReader.getDescriptions(actor));
			String legalAddress = "";
			String email = "";
			String website = "";
			String address = "";
			String phoneNumber = "";
			//more contact point per provider
			StmtIterator contactPointsStm = actor.listProperties(CRM.P76_has_contact_point);
			while (contactPointsStm.hasNext()) {
				Resource cp = contactPointsStm.nextStatement().getResource();
				//contact points can bean E45_Address or E51_Contact_Point
				if ((cp.hasProperty(RDF.type, CRM.E45_Address) || cp.hasProperty(RDF.type, CRM.E51_Contact_Point)) && cp.hasProperty(CRM.P2_has_type)) {
					Resource addressType = cp.getPropertyResourceValue(CRM.P2_has_type);
					String info = resourceReader.getLabel(cp);
					String addressTypeLabel = resourceReader.getLabel(addressType);
					switch (addressTypeLabel.toLowerCase()) {
					case "email":
						email = info;
						break;
					case "legal address":
						legalAddress = info;
						break;
					case "address":
						address = info;
						break;
					case "phone":
						phoneNumber = info;
						break;
					case "web":
						website = info;
						break;
					}
				}
			}
			jg.writeStartObject();
			jg.writeStringField("@class", "consistsOf");
			jg.writeObjectFieldStart("target");
			jg.writeStringField("@class", "PE_Contact_Reference_Facet");
			jg.writeStringField("appellation", appellation);
			jg.writeStringField("description", description);
			jg.writeStringField("legalAddress", legalAddress);
			if(StringUtils.isNotBlank(email)) jg.writeStringField("email", email);
			jg.writeStringField("website", website);
			jg.writeStringField("address", address);
			jg.writeStringField("phoneNumber", phoneNumber);
			jg.writeEndObject();
			jg.writeEndObject();
		}
	}

	public void writeDesignatedAccessPointFacet(final JsonGenerator jg, final Resource resource) throws IOException {
		//PP28_has_designated_access_point
		StmtIterator apStms = resource.listProperties(CRMpe.PP28_has_designated_access_point);
		while (apStms.hasNext()) {
			String entryName = resourceReader.getLabel(resource);
			//(mandatory)
			String endpoint = "";
			String protocol = "";
			String description = Joiner.on(',').join(resourceReader.getDescriptions(resource));
			//TODO: authorization is a ValueSchema, I do not understand how to use it and how to map it
			String authorization = "";
			Resource ap = apStms.next().getResource();
			endpoint = ap.getURI();
			//TODO: where to get protocol and authorization?
			jg.writeStartObject();
			jg.writeStringField("@class", "consistsOf");
			jg.writeObjectFieldStart("target");
			jg.writeStringField("@class", "PE29_Access_Point");
			jg.writeStringField("entryName", entryName);
			jg.writeStringField("description", description);
			jg.writeStringField("endpoint", endpoint);
			jg.writeStringField("protocol", protocol);
			//TODO: authorization is a ValueSchema, I do not understand how to use it and how to map it
			//jg.writeStringField("authorization", authorization);
			jg.writeEndObject();
			jg.writeEndObject();
		}

	}

	public void writeTemporalCoverageFacet(final JsonGenerator jg, final Resource resource) {
		//TODO: CoverageFacet wants a value and a mandatory schema. Can't be applied to Parthenos.
	}

	public ResourceReader getResourceReader() {
		return resourceReader;
	}

	public void setResourceReader(final ResourceReader resourceReader) {
		this.resourceReader = resourceReader;
	}
}
