package eu.dnetlib.parthenos.registry;

import java.io.BufferedOutputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import javax.annotation.PostConstruct;

import com.fasterxml.jackson.core.JsonEncoding;
import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonGenerator;
import com.google.common.collect.Lists;
import eu.dnetlib.parthenos.CRM;
import eu.dnetlib.parthenos.CRMdig;
import eu.dnetlib.parthenos.CRMpe;
import eu.dnetlib.parthenos.jrr.ParthenosRegistryRel;
import eu.dnetlib.parthenos.jrr.ParthenosRegistryResource;
import eu.dnetlib.parthenos.publisher.ParthenosPublisherException;
import eu.dnetlib.parthenos.rdf.ResourceReader;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.jena.ext.com.google.common.collect.Maps;
import org.apache.jena.rdf.model.Property;
import org.apache.jena.rdf.model.Resource;
import org.apache.jena.rdf.model.StmtIterator;
import org.apache.jena.vocabulary.RDF;
import org.gcube.common.authorization.client.Constants;
import org.gcube.common.authorization.library.AuthorizationEntry;
import org.gcube.common.authorization.library.provider.SecurityTokenProvider;
import org.gcube.common.scope.api.ScopeProvider;
import org.gcube.informationsystem.model.entity.facet.IdentifierFacet;
import org.gcube.informationsystem.model.relation.IsIdentifiedBy;
import org.gcube.informationsystem.resourceregistry.api.exceptions.ResourceRegistryException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.query.InvalidQueryException;
import org.gcube.informationsystem.resourceregistry.client.ParthenosRegistryClientSetter;
import org.gcube.informationsystem.resourceregistry.client.ResourceRegistryClient;
import org.gcube.informationsystem.resourceregistry.client.ResourceRegistryClientFactory;
import org.gcube.informationsystem.resourceregistry.publisher.ParthenosRegistryPublisherSetter;
import org.gcube.informationsystem.resourceregistry.publisher.ResourceRegistryPublisher;
import org.gcube.informationsystem.resourceregistry.publisher.ResourceRegistryPublisherFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

/**
 * Created by Alessia Bardi on 26/09/2017.
 *
 * TODO: when the mappings have a definitive URI generation policy then we need to be able to "upsert" (Luca will prepare new method for me).
 * When generating relationships, it could be the case that the linked entity was generated from another file, so we have to ask
 * the registry if it already exists a resource with a given URI. We cannot do it now because different entities with the same fake UUID are
 * generated by the mappings, now. See also https://support.d4science.org/issues/9820#change-56673
 *
 * @author Alessia Bardi
 */
@Component
public class GCubeResourceRegistrator {

	private static final Log log = LogFactory.getLog(GCubeResourceRegistrator.class);

	@Value("${gcube.registry.baseurl}")
	private String registryBaseURL;
	@Value("${gcube.registry.application.token}")
	private String applicationToken;

	@Autowired
	private FacetWriter facetWriter;
	@Autowired
	private RelWriter relWriter;
	@Autowired
	private ResourceReader resourceReader;

	private ResourceRegistryPublisher resourceRegistryPublisher;
	private ResourceRegistryClient resourceRegistryClient;


	@PostConstruct
	public void init() throws Exception {
		ParthenosRegistryClientSetter.forceToURL(getRegistryBaseURL());
		ParthenosRegistryPublisherSetter.forceToURL(getRegistryBaseURL());
		log.info("Registry URL forced to " + getRegistryBaseURL());
		if (StringUtils.isNotBlank(getApplicationToken())) {
			GCubeResourceRegistrator.setContext(getApplicationToken());
			log.info("GCube Context set");
		} else {
			log.fatal("GCube context cannot be configured without a value in gcube.registry.application.token");
		}
		this.resourceRegistryPublisher = ResourceRegistryPublisherFactory.create();
		this.resourceRegistryClient = ResourceRegistryClientFactory.create();
	}

	protected static String getCurrentScope(String token) throws Exception {
		AuthorizationEntry authorizationEntry = Constants.authorizationService().get(token);
		String context = authorizationEntry.getContext();
		log.info("Context of token " + token + " is: " + context);
		return context;
	}

	protected static void setContext(String token) throws Exception {
		SecurityTokenProvider.instance.set(token);
		ScopeProvider.instance.set(getCurrentScope(token));
	}

	public int unregister(final String datasourceApi) {
		//TODO: implement me
		return -1;
	}

	protected boolean deleteResourcebyUUID(final Resource resourceType, final UUID uuid) throws ResourceRegistryException {
		return resourceRegistryPublisher.deleteResource(StringUtils.remove(resourceType.getLocalName(), '-'), uuid);
	}

	/**
	 *
	 * @param rdfResource
	 * @param uuid
	 * @param type
	 * @throws IOException
	 * @throws ResourceRegistryException
	 * @throws ParthenosPublisherException
	 *
	 */
	public void register(final Resource rdfResource, final String uuid, final Resource type) throws IOException, ResourceRegistryException, ParthenosPublisherException {
		String resURI = rdfResource.getURI();
		log.debug("Processing resource " + resURI + " for registration on JRR");
		JsonFactory jsonFactory = new JsonFactory();
		ParthenosRegistryResource prr = null;
		List<ParthenosRegistryRel> rels;
		switch(type.getLocalName()){
		case "E7_Activity":
			if(rdfResource.hasProperty(RDF.type, CRMpe.PE1_Service)) {
				prr = processService(rdfResource, jsonFactory, uuid);
			}
			else{
				if(rdfResource.hasProperty(RDF.type, CRMpe.PE35_Project)){
					prr = processProject(rdfResource, jsonFactory, uuid);
				}
			}
			break;
		case "E55_Type":
			if(rdfResource.hasProperty(RDF.type, CRMpe.PE36_Competency_Type)) {
				prr = processBasicResource(rdfResource, CRMpe.PE36_Competency_Type, jsonFactory, uuid);
			}
			else{
				if(rdfResource.hasProperty(RDF.type, CRMpe.PE37_Protocol_Type))
					prr = processBasicResource(rdfResource, CRMpe.PE37_Protocol_Type, jsonFactory, uuid);
			}
			//TODO: need other E55_Types?
			break;
		case "E70_Thing":
			if(rdfResource.hasProperty(RDF.type, CRMdig.D1_Digital_Object)) {
				prr = processDigitalObject(rdfResource, jsonFactory, uuid);
			}
			//TODO: include PE32_Curated_Thing and E19_PhysicalObject?
			break;
		case "E39_Actor":
			prr = processActor(rdfResource, jsonFactory, uuid);
			break;
		case "E29_Design_or_Procedure":
			prr = processBasicResource(rdfResource, CRM.E29_Design_or_Procedure, jsonFactory, uuid);
			break;
		default:
			throw new IllegalArgumentException(String.format("Type "+type.getLocalName()+" not supported"));
		}
		if(prr != null) {
			this.resourceRegistryPublisher.createResource(prr.getRegistryType(), prr.getJson());
			rels = getResourceRels(rdfResource, prr);
			int registeredRels = 0;
			for(ParthenosRegistryRel r : rels){
				//TODO: handle exception
				String res = this.resourceRegistryPublisher.createIsRelatedTo(r.getRelName(), r.getJson());
				//TODO: handle negative response
				if(StringUtils.isNotBlank(res)) registeredRels++;
			}
			log.debug(String.format("Registered %d/%d relationships for uri %s, uuid %s", registeredRels, rels.size(), resURI, uuid));
		}
		else{
			log.warn("Could not register on registry: "+resURI);
		}

	}


	protected ParthenosRegistryResource processService(final Resource res, final JsonFactory jsonFactory, final String uuid) throws IOException {
		//ensure we are not registering generic PE1_Service because the class is abstract and the registry complaints.
		//We only want specific types of services.
		if (res.hasProperty(RDF.type, CRMpe.PE8_E_Service)
				|| res.hasProperty(RDF.type, CRMpe.PE2_Hosting_Service)
				|| res.hasProperty(RDF.type, CRMpe.PE3_Curating_Service)) {
			final ByteArrayOutputStream out = new ByteArrayOutputStream();
			BufferedOutputStream bos = new BufferedOutputStream(out);
			JsonGenerator jg = jsonFactory.createGenerator(bos, JsonEncoding.UTF8);
			jg.writeStartObject();
			//it should never happen to get PE1_Service as a specific type -- it happened in Parthenos top level entities
			String specificType = resourceReader.findSpecificType(res, CRMpe.PE1_Service).getLocalName();
			writeCommon(res, specificType, jg, uuid);
			//******THE FACETS *******//
			jg.writeArrayFieldStart("consistsOf");
			//list of facets
			writeCommonFacets(res, jg);
			facetWriter.writeEventFacet(jg);
			facetWriter.writeRightsFacet(jg, res);
			if (res.hasProperty(CRMpe.PP2_provided_by)) {
				//TODO: shouldn't this be a rel to an actor?
				Resource provider = res.getPropertyResourceValue(CRMpe.PP2_provided_by);
				facetWriter.writeContactReferenceFacet(jg, provider);
			}
			facetWriter.writeDesignatedAccessPointFacet(jg, res);
			jg.writeEndArray();
			jg.writeEndObject();
			jg.close();
			String json = out.toString("UTF-8");
			log.debug(json);
			return new ParthenosRegistryResource().setJson(json).setType(specificType).setUuid(uuid);

		}
		else{
			log.warn(String.format("Skipping resource: %s It is not an instance of a specific service"));
			return null;
		}
	}

	protected ParthenosRegistryResource processProject(final Resource res, final JsonFactory jsonFactory, final String uuid) throws IOException {
		String specificType = resourceReader.findSpecificType(res, CRMpe.PE35_Project).getLocalName();
		final ByteArrayOutputStream out = new ByteArrayOutputStream();
		BufferedOutputStream bos = new BufferedOutputStream(out);
		JsonGenerator jg = jsonFactory.createGenerator(bos, JsonEncoding.UTF8);
		jg.writeStartObject();

		writeCommon(res, specificType, jg, uuid);
		jg.writeArrayFieldStart("consistsOf");
		writeCommonFacets(res, jg);

		jg.writeEndArray();
		jg.writeEndObject();
		jg.close();
		String json = out.toString("UTF-8");
		log.debug(json);
		return new ParthenosRegistryResource().setJson(json).setType(specificType).setUuid(uuid);
	}

	protected ParthenosRegistryResource processActor(final Resource res, final JsonFactory jsonFactory, final String uuid) throws IOException {
		String specificType = resourceReader.findSpecificType(res, CRM.E39_Actor).getLocalName();
		final ByteArrayOutputStream out = new ByteArrayOutputStream();
		BufferedOutputStream bos = new BufferedOutputStream(out);
		JsonGenerator jg = jsonFactory.createGenerator(bos, JsonEncoding.UTF8);
		jg.writeStartObject();

		writeCommon(res, specificType, jg, uuid);
		//******THE FACETS *******//
		jg.writeArrayFieldStart("consistsOf");
		//list of facets
		writeCommonFacets(res, jg);
		facetWriter.writeContactReferenceFacet(jg, res);

		jg.writeEndArray();
		jg.writeEndObject();
		jg.close();
		String json = out.toString("UTF-8");
		log.debug(json);
		return new ParthenosRegistryResource().setJson(json).setType(specificType).setUuid(uuid);
	}

	protected ParthenosRegistryResource processDigitalObject(final Resource res, final JsonFactory jsonFactory, final String uuid) throws IOException {
		String specificType = resourceReader.findSpecificType(res, CRMdig.D1_Digital_Object).getLocalName();
		final ByteArrayOutputStream out = new ByteArrayOutputStream();
		BufferedOutputStream bos = new BufferedOutputStream(out);
		JsonGenerator jg = jsonFactory.createGenerator(bos, JsonEncoding.UTF8);
		jg.writeStartObject();

		writeCommon(res, specificType, jg, uuid);
		//******THE FACETS *******//
		jg.writeArrayFieldStart("consistsOf");
		//list of facets
		writeCommonFacets(res, jg);
		facetWriter.writeTemporalCoverageFacet(jg, res);
		facetWriter.writeRightsFacet(jg, res);

		jg.writeEndArray();
		jg.writeEndObject();
		jg.close();
		String json = out.toString("UTF-8");
		log.debug(json);
		return new ParthenosRegistryResource().setUuid(uuid).setType(specificType).setJson(json);
	}

	protected ParthenosRegistryResource processBasicResource(final Resource res, final Resource basicType, final JsonFactory jsonFactory, final String uuid) throws IOException {
		final ByteArrayOutputStream out = new ByteArrayOutputStream();
		BufferedOutputStream bos = new BufferedOutputStream(out);
		JsonGenerator jg = jsonFactory.createGenerator(bos, JsonEncoding.UTF8);
		jg.writeStartObject();
		String specificType = resourceReader.findSpecificType(res, basicType).getLocalName();
		writeCommon(res, specificType, jg, uuid);
		//******THE FACETS *******//
		jg.writeArrayFieldStart("consistsOf");
		writeCommonFacets(res, jg);
		jg.writeEndArray();
		jg.writeEndObject();
		jg.close();
		String json = out.toString("UTF-8");
		log.debug(json);
		return new ParthenosRegistryResource().setJson(json).setType(specificType).setUuid(uuid);
	}

	protected List<ParthenosRegistryRel> getResourceRels(final Resource resource, final ParthenosRegistryResource prr) throws IOException, ResourceRegistryException, InvalidQueryException, ParthenosPublisherException {
		JsonFactory jsonFactory = new JsonFactory();
		List<ParthenosRegistryRel> jsonRels = Lists.newArrayList();

		if(resource.hasProperty(RDF.type, CRMpe.PE1_Service)) {
			jsonRels.addAll(getRels(resource, prr, CRM.P129_is_about, jsonFactory));
			jsonRels.addAll(getRels(resource, prr, CRMpe.PP2_provided_by, jsonFactory));
			//TODO: IsRelatedTo the contact person of the provider. Interpret '"Follow path of service ‘Provided by’ and switch E39 for E21: E21- >p76->E51"
			jsonRels.addAll(getRels(resource, prr, CRMpe.PP45_has_competency, jsonFactory));
			jsonRels.addAll(getRels(resource, prr, CRMpe.PP4_hosts_object, jsonFactory));
			jsonRels.addAll(getRels(resource, prr, CRMpe.PP31_uses_curation_plan, jsonFactory));
			jsonRels.addAll(getRels(resource, prr, CRMpe.PP32_curates, jsonFactory));
			jsonRels.addAll(getRels(resource, prr, CRMpe.PP6_hosts_digital_object, jsonFactory));
			jsonRels.addAll(getRels(resource, prr, CRMpe.PP7_hosts_software_object, jsonFactory));
			jsonRels.addAll(getRels(resource, prr, CRMpe.PP8_hosts_dataset, jsonFactory));
			jsonRels.addAll(getRels(resource, prr, CRMpe.PP29_uses_access_protocol, jsonFactory));
			jsonRels.addAll(getRels(resource, prr, CRMpe.PP47_has_protocol_type, jsonFactory));
			jsonRels.addAll(getRels(resource, prr, CRMpe.PP46_brokers_access_to, jsonFactory));
			jsonRels.addAll(getRels(resource, prr, CRMpe.PP11_curates_volatile_digital_object, jsonFactory));
			jsonRels.addAll(getRels(resource, prr, CRMpe.PP12_curates_volatile_software, jsonFactory));
			jsonRels.addAll(getRels(resource, prr, CRMpe.PP13_curates_volatile_dataset, jsonFactory));
			jsonRels.addAll(getRels(resource, prr, CRMpe.PP15_delivers_on_request, jsonFactory));
			jsonRels.addAll(getRels(resource, prr, CRMpe.PP45_has_competency, jsonFactory));
			jsonRels.addAll(getInverseRels(resource, prr, CRMpe.PP1i_is_currently_offered_by, CRMpe.PP1_currently_offers, jsonFactory));
			jsonRels.addAll(getInverseRels(resource, prr, CRMpe.PP46i_has_access_brokered_by, CRMpe.PP46_brokers_access_to, jsonFactory));
		}
		if(resource.hasProperty(RDF.type, CRMdig.D1_Digital_Object)){
			jsonRels.addAll(getInverseRels(resource, prr, CRMpe.PP6i_is_digital_object_hosted_by, CRMpe.PP6_hosts_digital_object, jsonFactory));
			jsonRels.addAll(getInverseRels(resource, prr, CRMpe.PP18i_is_digital_object_part_of, CRMpe.PP18_has_digital_object_part, jsonFactory));
			jsonRels.addAll(getInverseRels(resource, prr, CRMpe.PP39i_has_metadata, CRMpe.PP39_is_metadata_for, jsonFactory));
			jsonRels.addAll(getInverseRels(resource, prr, CRMpe.PP41i_is_indexed_by, CRMpe.PP41_is_index_of, jsonFactory));
		}
		if(resource.hasProperty(RDF.type, CRMpe.PE18_Dataset)) {
			jsonRels.addAll(getInverseRels(resource, prr, CRMpe.PP23i_is_dataset_part_of, CRMpe.PP23_has_dataset_part, jsonFactory));
			jsonRels.addAll(getInverseRels(resource, prr, CRMpe.PP8i_is_dataset_hosted_by, CRMpe.PP8_hosts_dataset, jsonFactory));
		}
		if(resource.hasProperty(RDF.type, CRMdig.D14_Software)) {
			jsonRels.addAll(getInverseRels(resource, prr, CRMpe.PP14i_is_run_by, CRMpe.PP14_runs_on_request, jsonFactory));
			jsonRels.addAll(getInverseRels(resource, prr, CRMpe.PP15i_is_delivered_by, CRMpe.PP15_delivers_on_request, jsonFactory));
			jsonRels.addAll(getInverseRels(resource, prr, CRMpe.PP21i_is_software_part_of, CRMpe.PP21_has_software_part, jsonFactory));
			//TODO: is this correct? Service --> uses access protocol --> Software ?
			jsonRels.addAll(getInverseRels(resource, prr, CRMpe.PP29i_is_access_protocol_used_by, CRMpe.PP29_uses_access_protocol, jsonFactory));
		}
		if(resource.hasProperty(RDF.type, CRMpe.PE19_Persistent_Digital_Object)) {
			jsonRels.addAll(getRels(resource, prr, CRMpe.PP16_has_persistent_digital_object_part, jsonFactory));
			jsonRels.addAll(getInverseRels(resource, prr, CRMpe.PP16i_is_persistent_digital_object_part_of, CRMpe.PP16_has_persistent_digital_object_part, jsonFactory));
			jsonRels.addAll(getInverseRels(resource, prr, CRMpe.PP17i_is_snapshot_of, CRMpe.PP17_has_snapshot, jsonFactory));
		}
		if(resource.hasProperty(RDF.type, CRMpe.PE20_Volatile_Digital_Object)) {
			jsonRels.addAll(getRels(resource, prr, CRMpe.PP17_has_snapshot, jsonFactory));
			jsonRels.addAll(getRels(resource, prr, CRMpe.PP18_has_digital_object_part, jsonFactory));
			jsonRels.addAll(getInverseRels(resource, prr, CRMpe.PP11i_is_volatile_digital_object_curated_by, CRMpe.PP11_curates_volatile_digital_object, jsonFactory));
		}
		if(resource.hasProperty(RDF.type, CRMpe.PE21_Persistent_Software)) {
			jsonRels.addAll(getRels(resource, prr, CRMpe.PP19_has_persistent_software_part, jsonFactory));
			jsonRels.addAll(getInverseRels(resource, prr, CRMpe.PP19i_is_persistent_software_part_of, CRMpe.PP19_has_persistent_software_part, jsonFactory));
			jsonRels.addAll(getInverseRels(resource, prr, CRMpe.PP22i_is_release_of, CRMpe.PP22_has_release, jsonFactory));
		}
		if(resource.hasProperty(RDF.type, CRMpe.PE22_Persistent_Dataset)) {
			jsonRels.addAll(getRels(resource, prr, CRMpe.PP20_has_persistent_dataset_part, jsonFactory));
			jsonRels.addAll(getRels(resource, prr, CRMpe.PP39_is_metadata_for, jsonFactory));
			jsonRels.addAll(getInverseRels(resource, prr, CRMpe.PP20i_is_persistent_dataset_part_of, CRMpe.PP20_has_persistent_dataset_part, jsonFactory));
			jsonRels.addAll(getInverseRels(resource, prr, CRMpe.PP24i_is_dataset_snapshot_of, CRMpe.PP24_has_dataset_snapshot, jsonFactory));
			jsonRels.addAll(getInverseRels(resource, prr, CRMpe.PP40i_is_deprecated_by, CRMpe.PP40_created_successor_of, jsonFactory));
		}
		if(resource.hasProperty(RDF.type, CRMpe.PE23_Volatile_Software)) {
			jsonRels.addAll(getRels(resource, prr, CRMpe.PP21_has_software_part, jsonFactory));
			jsonRels.addAll(getRels(resource, prr, CRMpe.PP22_has_release, jsonFactory));
			jsonRels.addAll(getInverseRels(resource, prr, CRMpe.PP12i_is_volatile_software_curated_by, CRMpe.PP12_curates_volatile_software, jsonFactory));
		}
		if(resource.hasProperty(RDF.type, CRMpe.PE24_Volatile_Dataset)) {
			jsonRels.addAll(getRels(resource, prr, CRMpe.PP23_has_dataset_part, jsonFactory));
			jsonRels.addAll(getRels(resource, prr, CRMpe.PP24_has_dataset_snapshot, jsonFactory));
			jsonRels.addAll(getRels(resource, prr, CRMpe.PP41_is_index_of, jsonFactory));
			jsonRels.addAll(getInverseRels(resource, prr, CRMpe.PP13i_is_volatile_dataset_curated_by, CRMpe.PP13_curates_volatile_dataset, jsonFactory));
		}
		if(resource.hasProperty(RDF.type, CRMpe.PE26_RI_Project)) {
			jsonRels.addAll(getRels(resource, prr, CRMpe.PP25_has_maintaining_RI, jsonFactory));
			jsonRels.addAll(getRels(resource, prr, CRMpe.PP1_currently_offers, jsonFactory));
		}
		if(resource.hasProperty(RDF.type, CRMpe.PE35_Project)) {
			jsonRels.addAll(getRels(resource, prr, CRMpe.PP43_supported_project_activity, jsonFactory));
			jsonRels.addAll(getRels(resource, prr, CRMpe.PP44_has_maintaining_team, jsonFactory));
		}
		if(resource.hasProperty(RDF.type, CRM.E39_Actor)) {
			jsonRels.addAll(getInverseRels(resource, prr, CRMpe.PP2i_provides, CRMpe.PP2_provided_by, jsonFactory));
		}
		if(resource.hasProperty(RDF.type, CRMpe.PE34_Team)) {
			jsonRels.addAll(getInverseRels(resource, prr, CRMpe.PP44i_is_maintaining_team_of, CRMpe.PP44_has_maintaining_team, jsonFactory));
		}
		if(resource.hasProperty(RDF.type, CRM.E70_Thing)) {
			jsonRels.addAll(getInverseRels(resource, prr, CRMpe.PP4i_is_object_hosted_by, CRMpe.PP4_hosts_object, jsonFactory));
		}
		if(resource.hasProperty(RDF.type, CRMpe.PE25_RI_Consortium)) {
			jsonRels.addAll(getInverseRels(resource, prr, CRMpe.PP25i_is_maintaining_RI_of, CRMpe.PP25_has_maintaining_RI, jsonFactory));
		}
		if(resource.hasProperty(RDF.type, CRMpe.PE28_Curation_Plan)) {
			jsonRels.addAll(getInverseRels(resource, prr, CRMpe.PP31i_is_curation_plan_used_by, CRMpe.PP31_uses_curation_plan, jsonFactory));
		}
		if(resource.hasProperty(RDF.type, CRMpe.PE32_Curated_Thing)) {
			jsonRels.addAll(getInverseRels(resource, prr, CRMpe.PP32i_is_curated_by, CRMpe.PP32_curates, jsonFactory));
		}
		if(resource.hasProperty(RDF.type, CRM.E65_Creation)) {
			jsonRels.addAll(getRels(resource, prr, CRMpe.PP40_created_successor_of, jsonFactory));
		}
		if(resource.hasProperty(RDF.type, CRM.E7_Activity)) {
			jsonRels.addAll(getInverseRels(resource, prr, CRMpe.PP43i_is_project_activity_supported_by, CRMpe.PP43_supported_project_activity, jsonFactory));
		}
		if(resource.hasProperty(RDF.type, CRMpe.PE36_Competency_Type)) {
			jsonRels.addAll(getInverseRels(resource, prr, CRMpe.PP45i_is_competency_of, CRMpe.PP45_has_competency, jsonFactory));
		}
		if(resource.hasProperty(RDF.type, CRMpe.PE37_Protocol_Type)) {
			jsonRels.addAll(getInverseRels(resource, prr, CRMpe.PP47i_is_protocol_type_of, CRMpe.PP47_has_protocol_type, jsonFactory));
		}
		if(resource.hasProperty(RDF.type, CRMpe.PE38_Schema)) {
			jsonRels.addAll(getInverseRels(resource, prr, CRMpe.PP47i_is_protocol_type_of, CRMpe.PP47_has_protocol_type, jsonFactory));
		}
		return jsonRels;
	}

	protected List<ParthenosRegistryRel> getRels(final Resource subject, final ParthenosRegistryResource subjectPrr, final Property rel, final JsonFactory jsonFactory) throws IOException, InvalidQueryException, ParthenosPublisherException, ResourceRegistryException {
		List<ParthenosRegistryRel> rels = Lists.newArrayList();
		StmtIterator it = subject.listProperties(rel);
		while (it.hasNext()) {
			Resource obj = it.nextStatement().getResource();
			if (obj == null) throw new ParthenosPublisherException(String.format("No target resource available for %s --> %s", subject.getURI(), rel.getURI()));
			ParthenosRegistryResource target = findInRegistry(obj.getURI());
			if(target == null){
				log.debug(String.format("Rel: %s - %s - %s: target is not yet registered", subject.getURI(), rel.getLocalName(), obj.getURI()));
			}
			else{
				rels.add(getRel(subjectPrr, rel, target, jsonFactory));
			}
		}
		return rels;
	}

	protected ParthenosRegistryRel getRel(final ParthenosRegistryResource subjectPrr, final Property rel, final ParthenosRegistryResource targetPrr, final JsonFactory jsonFactory ) throws IOException {
		String relJson = relWriter.writeRelationship(jsonFactory, rel.getLocalName(), subjectPrr.getUuid(), subjectPrr.getRegistryType(), targetPrr.getUuid(), targetPrr.getRegistryType());
		return new ParthenosRegistryRel().json(relJson).relName(rel.getLocalName());
	}

	protected List<ParthenosRegistryRel> getInverseRels(final Resource resource, final ParthenosRegistryResource resourcePRR, final Property rel, final Property inverseRel, final JsonFactory jsonFactory) throws IOException, InvalidQueryException, ResourceRegistryException {
		List<ParthenosRegistryRel> rels = Lists.newArrayList();
		StmtIterator it = resource.listProperties(rel);
		while (it.hasNext()) {
			Resource obj = it.nextStatement().getResource();
			String objURI = obj.getURI();
			ParthenosRegistryResource registryRes = findInRegistry(objURI);
			if (registryRes != null) {
				rels.add(getRel(registryRes, inverseRel, resourcePRR, jsonFactory));
			}
		}
		return rels;
	}

	/**
	 * Write the common properties: header, class for all resources
	 *
	 * @param resource  input Resource.
	 * @param className name of the class for the registry
	 * @param jg        JsonGenerator to write with.
	 * @param uuid      uuid of the resource
	 */
	protected void writeCommon(final Resource resource, final String className, final JsonGenerator jg, final String uuid)
			throws IOException {
		jg.writeObjectFieldStart("header");
		jg.writeStringField("uuid", uuid);
		jg.writeEndObject();
		jg.writeStringField("@class", StringUtils.remove(className, '-'));
	}

	/**
	 * Write the common facets: identifier and info facets
	 *
	 * @param res
	 * @param jg
	 */
	protected void writeCommonFacets(final Resource res, final JsonGenerator jg) throws IOException {
		facetWriter.writeIdentifierFacet(jg, res.getURI());
		facetWriter.writeP1Facets(jg, res);
		facetWriter.writeInfoFacet(jg, res);
	}

	public ParthenosRegistryResource findInRegistry(final String uri) throws ResourceRegistryException {
		Map<String, Object> queryMap = Maps.newHashMap();
		queryMap.put("value", uri);
		List<org.gcube.informationsystem.model.entity.Resource> resources = resourceRegistryClient.getFilteredResources(
				org.gcube.informationsystem.model.entity.Resource.class, IsIdentifiedBy.class, IdentifierFacet.class, true, queryMap);
		if(resources.isEmpty()) return null;
		org.gcube.informationsystem.model.entity.Resource resource = resources.get(0);
		if(resources.size() > 1){
			log.warn("More than one resource in registry with uri "+uri+" -- considering the first, with uuid: "+resource.getHeader().getUUID());
		}
		//build PRR from res, need to have type and uuid
		//TODO: check the type is what we want
		ParthenosRegistryResource prr = new ParthenosRegistryResource().setUuid(resource.getHeader().getUUID().toString()).setType(resource.getClass().getSimpleName());
		log.debug(prr.toString());
		return prr;
	}


	public FacetWriter getFacetWriter() {
		return facetWriter;
	}

	public void setFacetWriter(final FacetWriter facetWriter) {
		this.facetWriter = facetWriter;
	}

	public String getApplicationToken() {
		return applicationToken;
	}

	public void setApplicationToken(final String applicationToken) {
		this.applicationToken = applicationToken;
	}

	public String getRegistryBaseURL() {
		return registryBaseURL;
	}

	public void setRegistryBaseURL(final String registryBaseURL) {
		this.registryBaseURL = registryBaseURL;
	}

	public RelWriter getRelWriter() {
		return relWriter;
	}

	public void setRelWriter(final RelWriter relWriter) {
		this.relWriter = relWriter;
	}

	public ResourceRegistryPublisher getResourceRegistryPublisher() {
		return resourceRegistryPublisher;
	}

	public void setResourceRegistryPublisher(final ResourceRegistryPublisher resourceRegistryPublisher) {
		this.resourceRegistryPublisher = resourceRegistryPublisher;
	}

	public ResourceRegistryClient getResourceRegistryClient() {
		return resourceRegistryClient;
	}

	public void setResourceRegistryClient(final ResourceRegistryClient resourceRegistryClient) {
		this.resourceRegistryClient = resourceRegistryClient;
	}

	public ResourceReader getResourceReader() {
		return resourceReader;
	}

	public void setResourceReader(final ResourceReader resourceReader) {
		this.resourceReader = resourceReader;
	}
}
