package eu.dnetlib.parthenos.catalogue;

import java.io.BufferedOutputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URISyntaxException;
import java.net.URLDecoder;
import java.util.Iterator;
import java.util.Set;

import com.fasterxml.jackson.core.JsonEncoding;
import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonGenerator;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterators;
import com.google.common.collect.Sets;
import eu.dnetlib.parthenos.CRM;
import eu.dnetlib.parthenos.CRMdig;
import eu.dnetlib.parthenos.CRMpe;
import eu.dnetlib.parthenos.jrr.ParthenosRegistryResource;
import eu.dnetlib.parthenos.publisher.ParthenosPublisherException;
import eu.dnetlib.parthenos.rdf.ResourceReader;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.jena.rdf.model.RDFNode;
import org.apache.jena.rdf.model.Resource;
import org.apache.jena.rdf.model.StmtIterator;
import org.apache.jena.vocabulary.RDF;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

/**
 * Created by Alessia Bardi on 21/11/2017.
 *
 * @author Alessia Bardi
 */
@Component
public class CatalogueRegistrator {

	private static final Log log = LogFactory.getLog(CatalogueRegistrator.class);
	private final String PARTHENOS_BASE_URL = "http://parthenos.d4science.org";
	private final String CATALOGUE_NAME_REFIX = "parthenos_";

	@Autowired
	private ResourceReader resourceReader;

	@Autowired
	private CatalogueAPIClient catalogueAPIClient;

	private String groupTemplate = "{"
			+ "  \"name\":\"%s\","
			+ "  \"id\":\"%s\","
			+ "  \"title\": \"%s\""
			//+ "  \"description\": \"%s\""
			+ "}";



	public String register(final Resource resource, final Resource type, final String group)
			throws IOException, ParthenosPublisherException, URISyntaxException, InterruptedException {
		String resURI = resource.getURI();
		log.debug(String.format("Catalogue --> Processing resource : %s with type: %s", resURI, type.getLocalName()));
		//For the catalogue: Must be purely lowercase alphanumeric (ascii) characters and these symbols: -_
		//Replacing all non-alphanumeric characters with empty strings
		String resCatName = getNameForCatalogue(resURI.substring(resURI.lastIndexOf("handle/") + 7));

		ParthenosRegistryResource prr = catalogueAPIClient.getRegistered(resCatName);
		if(prr != null){
			//TODO: what to do if already registered?
			log.debug(resCatName+ " is already registered");
			return prr.getUuid();
		}
		else {
			//resource not yet registered
			String catGroup = ensureGroup(group);
			String json;
			//base types are {Project, Service, Actors, Datasets, Software, Physical Collections, ??Standards??} i.e. {PE35_Project, PE1_Service, E39_Actor, PE18_Dataset, D14_Software, E78_Collection, ??E29_Design_or_Procedure??)
			switch (type.getLocalName()) {
			case "PE35_Project":
				json = getJsonForProject(resource, resCatName, catGroup);
				break;
			case "PE1_Service":
				json = getJsonForService(resource, resCatName, catGroup);
				break;
//			case "E7_Activity":
//				json = getJsonForActivity(resource, resCatName, catGroup);
//				break;
			case "E29_Design_or_Procedure":
				json = getJsonForDesignProcedure(resource, resCatName, catGroup);
				break;
			case "E39_Actor":
				json = getJsonForActor(resource, resCatName, catGroup);
				break;
			case "PE18_Dataset":
				json = getJsonForDataset(resource, resCatName, catGroup);
				break;
			case "D14_Software":
				json = getJsonForSoftware(resource, resCatName, catGroup);
				break;
			case "E78_Collection":
				json = getJsonForCollection(resource, resCatName, catGroup);
				break;
//			case "E70_Thing":
//				json = getJsonForThing(resource, resCatName, catGroup);
//				break;
			default:
				throw new IllegalArgumentException(String.format("Type " + type.getLocalName() + " not supported"));
			}
			String uuid = catalogueAPIClient.doRegister(json, resCatName);
			if(StringUtils.isBlank(uuid)){
				log.warn(String.format("%s could not be registered, sleeping 1sec and then try again", resURI));
				Thread.sleep(1000);
				uuid = catalogueAPIClient.doRegister(json, resCatName);
				if(StringUtils.isBlank(uuid)) {
					log.warn(String.format("%s could not be registered even the second time, giving up", resURI));
					return null;
				}
			}
			log.debug(String.format("%s registered on the catalogue with uuid: %s", resURI, uuid));
			return uuid;
		}
	}

	protected boolean purge(final String resCatName) throws URISyntaxException, IOException {
		return catalogueAPIClient.purgeItem(resCatName);
	}

	/**
	 * The catalogue accepts names which are strings between 2 and 100 characters long, containing only lowercase alphanumeric characters, - and _
	 * @param name the original name
	 * @return a string which is name adapted for the catalogue
	 */
	protected String getNameForCatalogue(final String name) throws UnsupportedEncodingException {
		//need to double decode for URLs like: http://parthenos.d4science.org/handle/Parthenos/REG/Dataset/Appellation/Isidore%2520Dataset
		String n = URLDecoder.decode(URLDecoder.decode(name,"UTF-8" ).replaceAll("/", "_").toLowerCase().replaceAll(" ", "_"), "UTF-8");
		if(n.length() > 100){
			log.warn("Catalog name > 100: " + name);
			n = n.substring(0,99);
		}
		if(n.length() < 2){
			n = CATALOGUE_NAME_REFIX+n;
		}
		return n;
	}


	protected String ensureGroup(final String group) throws ParthenosPublisherException, IOException, URISyntaxException {
		log.debug("Ensuring group exist: "+group);
		if (StringUtils.isNotBlank(group)) {
			String groupName = getNameForCatalogue(group);
			if (!catalogueAPIClient.groupExist(groupName)) {
				String groupJson = String.format(groupTemplate, groupName, groupName, group);
				catalogueAPIClient.registerGroup(groupJson, groupName);
				log.info("NEW GROUP REGISTERED: " + groupName);
			}
			return groupName;
		}
		return null;
	}

	protected String getJsonForProject(final Resource res, final String resNameForCatalogue, final String catGroup) throws IOException {
		JsonFactory jsonFactory = new JsonFactory();
		final ByteArrayOutputStream out = new ByteArrayOutputStream();
		BufferedOutputStream bos = new BufferedOutputStream(out);
		JsonGenerator jg = jsonFactory.createGenerator(bos, JsonEncoding.UTF8);
		jg.writeStartObject();
		writeCommonFields(jg, res, resNameForCatalogue, catGroup);
		jg.writeStringField("maintainer", Joiner.on(", ").join(resourceReader.getMaintainersForManagementSection(res)));
		//TODO: it should be better to identify email contacts rather than generic contact labels of maintainer
		//jg.writeStringField("maintainer_email", Joiner.on(", ").join(resourceReader.getMaintainerContacts(res)));

		jg.writeArrayFieldStart("extras");
		//system:type must be one among {Project, Service, Actors, Datasets, Software, Physical Collections, ??Standards??} i.e. {PE35_Project, PE1_Service, E39_Actor, PE18_Dataset, D14_Software, E78_Collection, ??E29_Design_or_Procedure??)
		addExtra(jg, "system:type", CRMpe.PE35_Project.getLocalName());
		//specific class
		addExtra(jg, "instance of", resourceReader.findSpecificType(res, CRM.E7_Activity).getLocalName());
		if (res.getURI().startsWith(PARTHENOS_BASE_URL))
			addExtra(jg, "Parthenos URL", res.getURI());
		else addExtra(jg, "URL", res.getURI());
		addIdentifiers(res, jg);

		addExtra(jg, "started on", resourceReader.getStartTime(res));
		addExtra(jg, "maintaining team", Joiner.on(", ").join(resourceReader.getMaintainerUrls(res)));
		addExtra(jg, "offers", Joiner.on(", ").join(resourceReader.getOfferedServiceUrls(res)));

		jg.writeEndArray(); //end extras

		jg.writeEndObject();
		jg.close();
		return out.toString("UTF-8");
	}

	protected String getJsonForService(final Resource res, final String resNameForCatalogue, final String catGroup) throws IOException {
		JsonFactory jsonFactory = new JsonFactory();
		final ByteArrayOutputStream out = new ByteArrayOutputStream();
		BufferedOutputStream bos = new BufferedOutputStream(out);
		JsonGenerator jg = jsonFactory.createGenerator(bos, JsonEncoding.UTF8);
		jg.writeStartObject();
		writeCommonFields(jg, res, resNameForCatalogue, catGroup);
		jg.writeStringField("maintainer", Joiner.on(", ").join(resourceReader.getMaintainersForManagementSection(res)));
		//TODO: it should be better to identify email contacts rather than generic contact labels of maintainer
		//jg.writeStringField("maintainer_email", Joiner.on(", ").join(resourceReader.getMaintainerContacts(res)));

		jg.writeArrayFieldStart("extras");
		//system:type must be one among {Project, Service, Actors, Datasets, Software, Physical Collections, ??Standards??} i.e. {PE35_Project, PE1_Service, E39_Actor, PE18_Dataset, D14_Software, E78_Collection, ??E29_Design_or_Procedure??)
		addExtra(jg, "system:type", CRMpe.PE1_Service.getLocalName());
		//specific class
		addExtra(jg, "instance of", resourceReader.findSpecificType(res, CRM.E7_Activity).getLocalName());
		if (res.getURI().startsWith(PARTHENOS_BASE_URL))
			addExtra(jg, "Parthenos URL", res.getURI());
		else addExtra(jg, "URL", res.getURI());
		addIdentifiers(res, jg);
		addExtra(jg, "competence", Joiner.on(", ").join(resourceReader.getCompetences(res)));
		addExtra(jg, "activity type", Joiner.on(", ").join(resourceReader.getActivityTypes(res)));
		//condition of use (Rights)
		addExtra(jg, "condition of use", resourceReader.getConditionOfUse(res));
		addExtra(jg, "contact points", Joiner.on(", ").join(Iterators.concat(resourceReader.getResourceDirectContactPointsURI(res), resourceReader.getProviderContactPoints(res))));
		addExtra(jg, "provided by", Joiner.on(", ").join(resourceReader.getProviderUris(res)));
		addExtra(jg, "online access point", Joiner.on(", ").join(resourceReader.getAccessPoints(res)));
		addExtra(jg, "protocol", Joiner.on(", ").join(resourceReader.getProtocols(res)));
		addExtra(jg, "delivers on request", Joiner.on(", ").join(resourceReader.getDeliversOnRequest(res)));
		addExtra(jg, "runs on request", Joiner.on(", ").join(resourceReader.getRunsOnRequest(res)));
		addExtra(jg, "hosts", Joiner.on(", ").join(resourceReader.getHostedStuff(res)));
		addExtra(jg, "curates", Joiner.on(", ").join(resourceReader.getCuratedObjects(res)));
		addExtra(jg, "declared begin/end of operation", Joiner.on(", ").join(resourceReader.getDeclarativeTimes(res)));
		addExtra(jg, "availability", resourceReader.getAvailability(res));
		addExtra(jg, "uses curation plan", Joiner.on(", ").join(resourceReader.getCurationPlans(res)));
		addExtra(jg, "time of service", Joiner.on(", ").join(resourceReader.getDeclarativeTimes(res)));
		//TODO: where to get it?
		//addExtra(jg, "last confirmation", "");
		//TODO: where to get it?
		//addExtra(jg, "date of registration", "");

		jg.writeEndArray(); //end extras

		jg.writeEndObject();
		jg.close();
		return out.toString("UTF-8");
	}

	protected String getJsonForActor(final Resource res, final String resNameForCatalogue, final String catGroup) throws IOException {
		JsonFactory jsonFactory = new JsonFactory();
		final ByteArrayOutputStream out = new ByteArrayOutputStream();
		BufferedOutputStream bos = new BufferedOutputStream(out);
		JsonGenerator jg = jsonFactory.createGenerator(bos, JsonEncoding.UTF8);
		jg.writeStartObject();
		writeCommonFields(jg, res, resNameForCatalogue, catGroup);

		jg.writeArrayFieldStart("extras");
		addExtra(jg, "system:type", CRM.E39_Actor.getLocalName());
		//specific class
		addExtra(jg, "instance of", resourceReader.findSpecificType(res, CRM.E39_Actor).getLocalName());
		if (res.getURI().startsWith(PARTHENOS_BASE_URL))
			addExtra(jg, "Parthenos URL", res.getURI());
		else addExtra(jg, "URL", res.getURI());
		addIdentifiers(res, jg);
		addExtra(jg, "has member", Joiner.on(", ").join(resourceReader.getMemberUrls(res)));
		addExtra(jg, "is member of", Joiner.on(", ").join(resourceReader.isMemberOf(res)));
		addExtra(jg, "provides", Joiner.on(", ").join(resourceReader.getProvidedServiceUrls(res)));
		String contactPoints = "";
		StmtIterator it = res.listProperties(CRM.P76_has_contact_point);
		while(it.hasNext()) {
			Resource cp = it.next().getResource();
			Resource cpType = cp.getPropertyResourceValue(CRM.P2_has_type);
			String cpTypeLabel = resourceReader.getLabel(cpType);
			String cpLabel = resourceReader.getLabel(cp);
			if (StringUtils.isNotBlank(cpLabel)) {
				if (StringUtils.isNotBlank(cpTypeLabel)) {
					contactPoints += cpTypeLabel + ": ";
				}
				contactPoints += cpLabel + "; ";
			}
		}
		addExtra(jg,"contact points", contactPoints );
		addExtra(jg, "maintains", Joiner.on(", ").join(resourceReader.getMaintainedUrls(res)));

		jg.writeEndArray();

		jg.writeEndObject();
		jg.close();
		return out.toString("UTF-8");
	}

	protected String getJsonForDataset(final Resource res, final String resNameForCatalogue, final String catGroup) throws IOException {
		JsonFactory jsonFactory = new JsonFactory();
		final ByteArrayOutputStream out = new ByteArrayOutputStream();
		BufferedOutputStream bos = new BufferedOutputStream(out);
		JsonGenerator jg = jsonFactory.createGenerator(bos, JsonEncoding.UTF8);
		jg.writeStartObject();
		writeCommonFields(jg, res, resNameForCatalogue, catGroup);

		jg.writeArrayFieldStart("extras");
		addExtra(jg, "system:type", CRMpe.PE18_Dataset.getLocalName());
		//specific class
		addExtra(jg, "instance of", resourceReader.findSpecificType(res, CRM.E70_Thing).getLocalName());
		if (res.getURI().startsWith(PARTHENOS_BASE_URL)) {
			addExtra(jg, "Parthenos URL", res.getURI());
		}
		else addExtra(jg, "URL", res.getURI());
		addIdentifiers(res, jg);
		addExtra(jg, "has type", Joiner.on(", ").join(resourceReader.getHasTypeLabels(res)));
		addExtra(jg, "is part of", Joiner.on(", ").join(resourceReader.getIsPartOfUrls(res)));
		addExtra(jg, "has part", Joiner.on(", ").join(resourceReader.getHasPartUrls(res)));
		addExtra(jg, "curated by", Joiner.on(", ").join(resourceReader.getCuratorUrls(res)));
		addExtra(jg, "curation plan", Joiner.on(", ").join(resourceReader.getResourceCuratorCurationPlans(res)));
		addExtra(jg, "hosted by", Joiner.on(", ").join(resourceReader.getHostedBys(res)));
		addExtra(jg, "encoding type", Joiner.on(", ").join(resourceReader.getEncodings(res)));
		addExtra(jg, "creator", Joiner.on(", ").join(resourceReader.getCreatorsURIs(res)));
		addExtra(jg, "created with software", Joiner.on(", ").join(resourceReader.getUsedSoftware(res)));
		addExtra(jg, "subject", Joiner.on(", ").join(resourceReader.getSubjects(res)));
		addExtra(jg, "temporal coverage", Joiner.on(", ").join(resourceReader.getTemporalCoverages(res)));
		addExtra(jg, "spatial coverage", Joiner.on(", ").join(resourceReader.getSpatialCoverages(res)));
		addExtra(jg, "uses", Joiner.on(", ").join(resourceReader.getUsedSoftware(res)));
		addExtra(jg, "used by", Joiner.on(", ").join(resourceReader.getUsedBy(res)));
		addExtra(jg, "languages", Joiner.on(", ").join(resourceReader.getLanguages(res)));

		addExtra(jg, "has metadata", Joiner.on(", ").join(resourceReader.getMetadata(res)));
		addExtra(jg, "is metadata for", Joiner.on(", ").join(resourceReader.getDescribedDataset(res)));
		addExtra(jg, "has snapshot", Joiner.on(", ").join(resourceReader.getSnapshots(res)));
		addExtra(jg, "is snapshot of", Joiner.on(", ").join(resourceReader.getIsSnapshotOfs(res)));


		jg.writeEndArray();

		jg.writeEndObject();
		jg.close();
		return out.toString("UTF-8");
	}

	protected String getJsonForSoftware(final Resource res, final String resNameForCatalogue, final String catGroup) throws IOException {
		JsonFactory jsonFactory = new JsonFactory();
		final ByteArrayOutputStream out = new ByteArrayOutputStream();
		BufferedOutputStream bos = new BufferedOutputStream(out);
		JsonGenerator jg = jsonFactory.createGenerator(bos, JsonEncoding.UTF8);
		jg.writeStartObject();
		writeCommonFields(jg, res, resNameForCatalogue, catGroup);

		jg.writeArrayFieldStart("extras");
		addExtra(jg, "system:type", CRMdig.D14_Software.getLocalName());
		//specific class
		addExtra(jg, "instance of", resourceReader.findSpecificType(res, CRM.E70_Thing).getLocalName());
		if (res.getURI().startsWith(PARTHENOS_BASE_URL)) {
			addExtra(jg, "Parthenos URL", res.getURI());
		}
		else addExtra(jg, "URL", res.getURI());
		addIdentifiers(res, jg);
		addExtra(jg, "subject", Joiner.on(", ").join(resourceReader.getSubjects(res)));
		addExtra(jg, "has type", Joiner.on(", ").join(resourceReader.getHasTypeLabels(res)));
		addExtra(jg, "hosted by", Joiner.on(", ").join(resourceReader.getHostedBys(res)));
		addExtra(jg, "curated by", Joiner.on(", ").join(resourceReader.getCuratorUrls(res)));
		addExtra(jg, "has snapshot", Joiner.on(", ").join(resourceReader.getSnapshots(res)));
		addExtra(jg, "is snapshot of", Joiner.on(", ").join(resourceReader.getIsSnapshotOfs(res)));
		addExtra(jg, "is part of", Joiner.on(", ").join(resourceReader.getIsPartOfUrls(res)));
		addExtra(jg, "has part", Joiner.on(", ").join(resourceReader.getHasPartUrls(res)));
		addExtra(jg, "has release", Joiner.on(", ").join(resourceReader.getHasReleases(res)));
		addExtra(jg, "is release of", Joiner.on(", ").join(resourceReader.getIsReleaseOfs(res)));
		addExtra(jg, "used by", Joiner.on(", ").join(resourceReader.getUsedBy(res)));
		addExtra(jg, "creation time", resourceReader.getFirstCreationTime(res));

		jg.writeEndArray();

		jg.writeEndObject();
		jg.close();
		return out.toString("UTF-8");
	}

	protected String getJsonForCollection(final Resource res, final String resNameForCatalogue, final String catGroup) throws IOException {
		JsonFactory jsonFactory = new JsonFactory();
		final ByteArrayOutputStream out = new ByteArrayOutputStream();
		BufferedOutputStream bos = new BufferedOutputStream(out);
		JsonGenerator jg = jsonFactory.createGenerator(bos, JsonEncoding.UTF8);
		jg.writeStartObject();
		writeCommonFields(jg, res, resNameForCatalogue, catGroup);

		jg.writeArrayFieldStart("extras");
		addExtra(jg, "system:type", CRM.E78_Collection.getLocalName());
		//specific class
		addExtra(jg, "instance of", resourceReader.findSpecificType(res, CRM.E70_Thing).getLocalName());
		if (res.getURI().startsWith(PARTHENOS_BASE_URL)) {
			addExtra(jg, "Parthenos URL", res.getURI());
		}
		else addExtra(jg, "URL", res.getURI());
		addIdentifiers(res, jg);
		addExtra(jg, "has type", Joiner.on(", ").join(resourceReader.getHasTypeLabels(res)));
		addExtra(jg, "subject", Joiner.on(", ").join(resourceReader.getSubjects(res)));
		addExtra(jg, "temporal coverage", Joiner.on(", ").join(resourceReader.getTemporalCoverages(res)));
		addExtra(jg, "spatial coverage", Joiner.on(", ").join(resourceReader.getSpatialCoverages(res)));
		addExtra(jg, "has part", Joiner.on(", ").join(resourceReader.getHasPartUrls(res)));
		addExtra(jg, "hosted by", Joiner.on(", ").join(resourceReader.getHostedBys(res)));
		addExtra(jg, "curated by", Joiner.on(", ").join(resourceReader.getCuratorUrls(res)));
		addExtra(jg, "creator", Joiner.on(", ").join(resourceReader.getCreatorsURIs(res)));
		addExtra(jg, "languages", Joiner.on(", ").join(resourceReader.getLanguages(res)));

		jg.writeEndArray();

		jg.writeEndObject();
		jg.close();
		return out.toString("UTF-8");
	}


	protected String getJsonForDesignProcedure(final Resource res, final String resNameForCatalogue, final String catGroup) throws IOException {
		JsonFactory jsonFactory = new JsonFactory();
		final ByteArrayOutputStream out = new ByteArrayOutputStream();
		BufferedOutputStream bos = new BufferedOutputStream(out);
		JsonGenerator jg = jsonFactory.createGenerator(bos, JsonEncoding.UTF8);
		jg.writeStartObject();
		writeCommonFields(jg, res, resNameForCatalogue, catGroup);

		jg.writeArrayFieldStart("extras");
		addExtra(jg, "system:type", CRM.E29_Design_or_Procedure.getLocalName());
		//specific class
		addExtra(jg, "instance of", resourceReader.findSpecificType(res, CRM.E29_Design_or_Procedure).getLocalName());
		if (res.getURI().startsWith(PARTHENOS_BASE_URL)) {
			addExtra(jg, "Parthenos URL", res.getURI());
		}
		else addExtra(jg, "URL", res.getURI());
		addIdentifiers(res, jg);
		addExtra(jg, "used by", Joiner.on(", ").join(resourceReader.getUsedBy(res)));
		//TODO: add additional metadata for E29_Design_or_Procedure, if any
		jg.writeEndArray();

		jg.writeEndObject();
		jg.close();
		return out.toString("UTF-8");
	}

	protected void addIdentifiers(final Resource res, final JsonGenerator jg ) throws IOException {
		/*
		 <${subjectURL}> crm:P1_is_identified_by ?IDRes .
		 ?IDRes a crm:E42_Identifier .
		 ?IDRes rdfs:label ?ID_label .
		 */
		StmtIterator it = res.listProperties(CRM.P1_is_identified_by);
		Set<String> ids = Sets.newHashSet();
		while(it.hasNext()){
			RDFNode obj = it.next().getObject();
			if(obj.isLiteral()) ids.add(obj.asLiteral().getLexicalForm());
			else {
				Resource id = (Resource) obj;
				if (id.hasProperty(RDF.type, CRM.E42_Identifier)) {
					ids.add(resourceReader.getLabel(id));
				}
			}
		}
		addExtra(jg, "ID", String.join(",", ids));
	}

	protected void addExtra(final JsonGenerator jg, final String key, final String value) throws IOException {
		if(StringUtils.isNotBlank(value)) {
			jg.writeStartObject();
			jg.writeStringField("key", key);
			jg.writeStringField("value", value);
			jg.writeEndObject();
		}
	}



	protected void writeCommonFields(final JsonGenerator jg, final Resource res, final String resNameForCatalogue, final String catGroup) throws IOException {
		//end of URI
		jg.writeStringField("name", resNameForCatalogue);
		//default license
		jg.writeStringField("license_id", resourceReader.getCatalogueLicense(res).getId());
		String title = resourceReader.getTitle(res);
		if (StringUtils.isBlank(title))
			title = resNameForCatalogue;
		jg.writeStringField("title", title);
		//description
		jg.writeStringField("notes",Joiner.on(';').join(resourceReader.getDescriptions(res)));
		//the names of all superclasses of the entity
		jg.writeArrayFieldStart("tags");
		Iterator<String> classNames = resourceReader.getRDFClassNames(res);
		while (classNames.hasNext()) {
			jg.writeStartObject();
			jg.writeStringField("name", classNames.next());
			jg.writeEndObject();
		}
		jg.writeEndArray();
		jg.writeArrayFieldStart("groups");
		jg.writeStartObject();
		jg.writeStringField("name", catGroup);
		jg.writeEndObject();
		jg.writeEndArray();
	}


	public ResourceReader getResourceReader() {
		return resourceReader;
	}

	public void setResourceReader(final ResourceReader resourceReader) {
		this.resourceReader = resourceReader;
	}

	public CatalogueAPIClient getCatalogueAPIClient() {
		return catalogueAPIClient;
	}

	public void setCatalogueAPIClient(final CatalogueAPIClient catalogueAPIClient) {
		this.catalogueAPIClient = catalogueAPIClient;
	}
}
