package eu.dnetlib.parthenos.catalogue;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;

import com.google.common.collect.Lists;
import eu.dnetlib.parthenos.jrr.ParthenosRegistryResource;
import eu.dnetlib.parthenos.publisher.ParthenosPublisherException;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.http.client.utils.URIBuilder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.*;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

/**
 * Created by Alessia Bardi on 08/03/2018.
 *
 * @author Alessia Bardi
 */
@Component
public class CatalogueAPIClient {

	private static final Log log = LogFactory.getLog(CatalogueAPIClient.class);

	@Value("${gcube.catalogue.baseurl}")
	private String baseURL;
	@Value("${gcube.registry.application.token}")
	private String applicationToken;
	@Value("${gcube.uri.resolver}")
	private String uri_resolver;
	private String resolverBodyTemplate = "{ \"entity_name\": \"%s\" }";

	private final String createItemPath = "rest/api/items/create";
	private final String updateItemPath = "rest/api/items/update";
	private final String showItemPath = "rest/api/items/show"; //param id
	private final String purgeItemPath = "rest/api/items/purge";

	//private final String showGroupPath = "rest/api/groups/show"; //param id
	//private final String createGroupPath = "rest/api/groups/create";
	//private final String purgeGroupPath = "rest/api/groups/purge";
	private String purgeBodyTemplate = "{\"id\":\"%s\"}";

	@Autowired
	private RestTemplate jrrRestTemplate;

	/**
	 * Calls the parthenos resolver to get the name to use for the catalogue
	 * @param resName Given a Parthenos URL like: http://parthenos.d4science.org/handle/Ariadne/AriadnePortal/rhdsq5xm3e40, the resName is Ariadne/AriadnePortal/rhdsq5xm3e40.
	 * @return the name as computed by the Parthenos resolver. Null if the request was not successfull.
	 */
	public String getNameForCatalogue(final String resName) throws URISyntaxException {
		log.debug(String.format("Calling Parthenos resolver for "+resName));
		String body = String.format(resolverBodyTemplate, resName);
		HttpEntity<String> entity = new HttpEntity<String>(body, getHeaders());
		URI uri = new URIBuilder(getUri_resolver()).build();
		ResponseEntity<String> res = jrrRestTemplate.<String>exchange(uri, HttpMethod.POST, entity, String.class);
		if(res.getStatusCode().is2xxSuccessful()){
			String resolved = res.getBody();
			log.debug(String.format("Parthenos resolver resolved %s with %s", resName, resolved ));
			return resolved;
		}
		else{
			log.debug(String.format("Parthenos resolver returned %s with cause %s for %s", res.getStatusCodeValue(), res.getStatusCode().getReasonPhrase(), resName));
			return null;
		}
	}

	public ParthenosRegistryResource getRegistered(final String resCatName) throws ParthenosPublisherException {
		log.debug(String.format("Catalogue --> Checking if item %s exists", resCatName));
		HttpEntity<String> entity = new HttpEntity<String>(getHeaders());
		try {
			URI uri = new URIBuilder(getBaseURL()+showItemPath).addParameter("gcube-token", getApplicationToken()).addParameter("id", resCatName).build();
			ResponseEntity<String> res = jrrRestTemplate.<String>exchange(uri, HttpMethod.GET, entity, String.class);
			CatalogueAPIResponse response = new CatalogueAPIResponse();
			response.setResponseBody(res.getBody());
			if(response.isSuccess()){
				ParthenosRegistryResource r = response.getParthenosRegistryResource();
				log.debug(String.format("Resource %s is in the catalogue with uuid %s", resCatName, r.getUuid()));
				return r;
			}
			else{
				log.debug(String.format("Resource %s is not in the catalogue yet", resCatName));
				return null;
			}
		} catch (Throwable t) {
			throw new ParthenosPublisherException(t);
		}
	}

	/**
	 * @param json
	 * @param resCatName
	 * @return the uuid assigned by the catalogue to the registered resource
	 * @throws ParthenosPublisherException
	 */
	public String doRegister(final String json, final String resCatName) throws URISyntaxException, IOException {
		log.debug(String.format("Catalogue --> Registering item %s : %s", resCatName, json));
		return doCatalogueCall(json, resCatName, createItemPath);
	}

	public String doUpdate(final String json, final String resCatName) throws IOException, URISyntaxException {
		log.debug(String.format("Catalogue --> Updating item %s : %s", resCatName, json));
		return doCatalogueCall(json, resCatName, updateItemPath);
	}

	protected String doCatalogueCall(final String json, final String resCatName, final String actionPath) throws URISyntaxException, IOException {
		HttpEntity<String> entity = new HttpEntity<String>(json, getHeaders());
		URI uri = new URIBuilder(getBaseURL() + actionPath).addParameter("gcube-token", getApplicationToken()).build();
		ResponseEntity<String> res = jrrRestTemplate.exchange(uri, HttpMethod.POST, entity, String.class);
		CatalogueAPIResponse response = new CatalogueAPIResponse();
		if (res.getStatusCode() == HttpStatus.OK) {
			String body = res.getBody();
			response.setResponseBody(body);
			if (response.isSuccess()) {
				return response.getParthenosRegistryResource().getUuid();
			} else {
				log.warn(resCatName + " could not be registered/updated because of " + response.getErrorMessage());
				return null;
			}
		} else {
			log.warn(resCatName + " POST FAILED: " + res.getStatusCodeValue());
			return null;
		}
	}

	protected boolean purgeItem(final String resCatName) throws URISyntaxException, IOException {
		log.debug(String.format("Catalogue --> Purge Item %s", resCatName));
		return purge(purgeItemPath, resCatName);
	}


	private boolean purge(String purgePath, String id) throws URISyntaxException, IOException {
		String body = String.format(purgeBodyTemplate, id);
		log.warn("DEL BODY: "+body);
		HttpEntity<String> entity = new HttpEntity<String>(body, getHeaders());
		URI uri = new URIBuilder(getBaseURL()+purgePath).addParameter("gcube-token", getApplicationToken()).build();
		ResponseEntity<String> res = jrrRestTemplate.exchange(uri, HttpMethod.DELETE, entity, String.class);
		CatalogueAPIResponse response = new CatalogueAPIResponse();
		if(res.getStatusCode() == HttpStatus.OK){
			String resBody = res.getBody();
			response.setResponseBody(resBody);
			//TODO: may be useful to log something when the request did not succeeded
			return response.isSuccess();
		}
		else {
			log.error(res.getStatusCodeValue());
			return false;
		}
	}


	private HttpHeaders getHeaders() {
		HttpHeaders headers = new HttpHeaders();
		headers.setContentType(MediaType.APPLICATION_JSON);
		headers.setAccept(Lists.newArrayList(MediaType.APPLICATION_JSON));
		//headers.set("gcube-token", getApplicationToken() );
		return headers;
	}

	public String getBaseURL() {
		return baseURL;
	}

	public void setBaseURL(final String baseURL) {
		this.baseURL = baseURL;
	}

	public String getApplicationToken() {
		return applicationToken;
	}

	public void setApplicationToken(final String applicationToken) {
		this.applicationToken = applicationToken;
	}

	public String getCreateItemPath() {
		return createItemPath;
	}

	public String getShowItemPath() {
		return showItemPath;
	}

	public RestTemplate getJrrRestTemplate() {
		return jrrRestTemplate;
	}

	public void setJrrRestTemplate(final RestTemplate jrrRestTemplate) {
		this.jrrRestTemplate = jrrRestTemplate;
	}
	public String getUri_resolver() {
		return uri_resolver;
	}

	public void setUri_resolver(final String uri_resolver) {
		this.uri_resolver = uri_resolver;
	}

	public String getPurgeItemPath() {
		return purgeItemPath;
	}

	public String getPurgeBodyTemplate() {
		return purgeBodyTemplate;
	}

	public void setPurgeBodyTemplate(final String purgeBodyTemplate) {
		this.purgeBodyTemplate = purgeBodyTemplate;
	}

}
