package eu.dnetlib.parthenos.catalogue;

import java.io.BufferedOutputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.net.URISyntaxException;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import com.fasterxml.jackson.core.JsonEncoding;
import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonGenerator;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterators;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import eu.dnetlib.parthenos.CRM;
import eu.dnetlib.parthenos.publisher.ParthenosPublisherException;
import eu.dnetlib.parthenos.rdf.ResourceReader;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.jena.rdf.model.RDFNode;
import org.apache.jena.rdf.model.Resource;
import org.apache.jena.rdf.model.StmtIterator;
import org.apache.jena.vocabulary.RDF;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

/**
 * Created by Alessia Bardi on 21/11/2017.
 *
 * @author Alessia Bardi
 */
@Component
public class CatalogueRegistrator {

	private static final Log log = LogFactory.getLog(CatalogueRegistrator.class);
	private final String PARTHENOS_BASE_URL = "http://parthenos.d4science.org";

	@Autowired
	private ResourceReader resourceReader;

	@Autowired
	private CatalogueAPIClient catalogueAPIClient;

	public String register(final Resource resource, final Resource type, final String datasourceName)
			throws IOException, ParthenosPublisherException, URISyntaxException, InterruptedException {
		String resURI = resource.getURI();
		log.debug(String.format("Catalogue --> Processing resource : %s with type: %s from source: %s", resURI, type.getLocalName(), datasourceName));
		String resCatName = catalogueAPIClient.getNameForCatalogue(resURI.substring(resURI.lastIndexOf("handle/") + 7));
		if(catalogueAPIClient.isRegistered(resCatName)){
			log.debug(resCatName+ " is already registered");
			String json = getJson(type, resource, resCatName, datasourceName);
			catalogueAPIClient.doUpdate(json, resCatName);
		}
		else {
			//resource not yet registered
			String json = getJson(type, resource, resCatName, datasourceName);
			if(!catalogueAPIClient.doRegister(json, resCatName)){
				log.warn(String.format("%s could not be registered even the second time, giving up", resURI));
				return null;
			}
		}
		log.debug(String.format("%s registered on the catalogue with name: %s", resURI, resCatName));
		return resCatName;
	}

	protected boolean purge(final String resCatName) throws URISyntaxException, ParthenosPublisherException {
		return catalogueAPIClient.purgeItem(resCatName);
	}

	public int purgeAll(final int bulkSize) throws ParthenosPublisherException {
		return catalogueAPIClient.purgeAll(bulkSize);
	}



	protected String getJson(final Resource type, final Resource resource, final String resNameForCatalogue, final String datasourceName)
			throws IOException, ParthenosPublisherException {
		switch (type.getLocalName()) {
		case "E29_Design_or_Procedure":
			return getJsonForDesignProcedure(resource, resNameForCatalogue, datasourceName);
		case "D14_Software":
			return getJsonForSoftware(resource, resNameForCatalogue, datasourceName);
		case "PE35_Project":
				return getJsonForProject(resource, resNameForCatalogue, datasourceName);
		case "PE1_Service":
			return getJsonForService(resource, resNameForCatalogue, datasourceName);
		case "E39_Actor":
			return getJsonForActor(resource, resNameForCatalogue, datasourceName);
		case "PE18_Dataset":
			return getJsonForDataset(resource, resNameForCatalogue, datasourceName);
		case "E78_Collection":
			return getJsonForCollection(resource, resNameForCatalogue, datasourceName);
		default:
			throw new IllegalArgumentException(String.format("Type " + type.getLocalName() + " not supported"));
		}
	}

	protected String getJsonForProject(final Resource res, final String resNameForCatalogue, final String datasourceName)
			throws IOException, ParthenosPublisherException {
		JsonFactory jsonFactory = new JsonFactory();
		final ByteArrayOutputStream out = new ByteArrayOutputStream();
		BufferedOutputStream bos = new BufferedOutputStream(out);
		JsonGenerator jg = jsonFactory.createGenerator(bos, JsonEncoding.UTF8);
		jg.writeStartObject();
		writeCommonFields(jg, res, resNameForCatalogue, datasourceName);
		//Only RI_Project are linked to RIs that must be used as groups
		List<String> maintainer_RIs = Lists.newArrayList(resourceReader.getMaintainersLabels(res));
		if(maintainer_RIs.size() > 0) {
			jg.writeArrayFieldStart("groups");
			for(String ri : maintainer_RIs) {
				String group = CKANUtils.getCkanGroup(ri);
				if(StringUtils.isNotBlank(group)) {
					jg.writeStartObject();
					jg.writeStringField("name", group);
					jg.writeEndObject();
				}
			}
			jg.writeEndArray();
		}

		jg.writeStringField("maintainer", Joiner.on(", ").join(resourceReader.getMaintainersLabels(res)));
		//TODO: it should be better to identify email contacts rather than generic contact labels of maintainer
		//jg.writeStringField("maintainer_email", Joiner.on(", ").join(resourceReader.getMaintainerContacts(res)));

		jg.writeArrayFieldStart("extras");
		addExtra(jg, "system:type", CKANUtils.Project_type);
		//specific class
		addExtra(jg, "instance of", resourceReader.findSpecificType(res, CRM.E7_Activity).getLocalName());
		if (res.getURI().startsWith(PARTHENOS_BASE_URL))
			addExtra(jg, "Parthenos URL", res.getURI());
		else addExtra(jg, "URL", res.getURI());
		addIdentifiers(res, jg);

		addExtra(jg, "started on", resourceReader.getStartTime(res));
		int idx = 1;
		Iterator<String> maintainers = resourceReader.getMaintainerUrls(res);
		while(maintainers.hasNext()){
			addExtra(jg, String.format("maintaining team (%d)", idx), maintainers.next());
			idx++;
		}
		//addExtra(jg, "maintaining team", Joiner.on(", ").join(resourceReader.getMaintainerUrls(res)));
		idx = 1;
		Iterator<String> services = resourceReader.getOfferedServiceUrls(res);
		while(services.hasNext()){
			addExtra(jg, String.format("offers (%d)", idx), services.next());
			idx++;
		}
		//addExtra(jg, "offers", Joiner.on(", ").join(resourceReader.getOfferedServiceUrls(res)));

		jg.writeEndArray(); //end extras

		jg.writeEndObject();
		jg.close();
		return out.toString("UTF-8");
	}

	protected String getJsonForService(final Resource res, final String resNameForCatalogue, final String datasourceName)
			throws IOException, ParthenosPublisherException {
		JsonFactory jsonFactory = new JsonFactory();
		final ByteArrayOutputStream out = new ByteArrayOutputStream();
		BufferedOutputStream bos = new BufferedOutputStream(out);
		JsonGenerator jg = jsonFactory.createGenerator(bos, JsonEncoding.UTF8);
		jg.writeStartObject();
		writeCommonFields(jg, res, resNameForCatalogue, datasourceName);
		jg.writeStringField("maintainer", Joiner.on(", ").join(resourceReader.getMaintainersLabels(res)));
		//TODO: it should be better to identify email contacts rather than generic contact labels of maintainer
		//jg.writeStringField("maintainer_email", Joiner.on(", ").join(resourceReader.getMaintainerContacts(res)));

		jg.writeArrayFieldStart("extras");
		addExtra(jg, "system:type", CKANUtils.Service_type);
		//specific class
		addExtra(jg, "instance of", resourceReader.findSpecificType(res, CRM.E7_Activity).getLocalName());
		if (res.getURI().startsWith(PARTHENOS_BASE_URL))
			addExtra(jg, "Parthenos URL", res.getURI());
		else addExtra(jg, "URL", res.getURI());
		addIdentifiers(res, jg);
		addExtra(jg, "competence", Joiner.on(", ").join(resourceReader.getCompetences(res)));
		addExtra(jg, "activity type", Joiner.on(", ").join(resourceReader.getActivityTypes(res)));
		//condition of use (Rights)
		addExtra(jg, "condition of use", resourceReader.getConditionOfUse(res));
		int idx = 1;
		Iterator<String> contacts = Iterators.concat(resourceReader.getResourceDirectContactPointsURI(res), resourceReader.getProviderContactPoints(res));
		while(contacts.hasNext()){
			addExtra(jg, String.format("contact points (%d)", idx), contacts.next());
			idx++;
		}
		//addExtra(jg, "contact points", Joiner.on(", ").join(Iterators.concat(resourceReader.getResourceDirectContactPointsURI(res), resourceReader.getProviderContactPoints(res))));
		idx = 1;
		Iterator<String> providers = resourceReader.getProviderUris(res);
		while(providers.hasNext()){
			addExtra(jg, String.format("provided by (%d)", idx), providers.next());
			idx++;
		}
		//addExtra(jg, "provided by", Joiner.on(", ").join(resourceReader.getProviderUris(res)));
		idx = 1;
		Iterator<String> points = resourceReader.getAccessPoints(res);
		while(points.hasNext()){
			addExtra(jg, String.format("online access point (%d)", idx), points.next());
			idx++;
		}
		//addExtra(jg, "online access point", Joiner.on(", ").join(resourceReader.getAccessPoints(res)));
		addExtra(jg, "protocol", Joiner.on(", ").join(resourceReader.getProtocols(res)));
		idx = 1;
		Iterator<String> delivers = resourceReader.getDeliversOnRequest(res);
		while(delivers.hasNext()){
			addExtra(jg, String.format("delivers on request (%d)", idx), delivers.next());
			idx++;
		}
		//addExtra(jg, "delivers on request", Joiner.on(", ").join(resourceReader.getDeliversOnRequest(res)));
		idx = 1;
		Iterator<String> runs = resourceReader.getRunsOnRequest(res);
		while(runs.hasNext()){
			addExtra(jg, String.format("runs on request (%d)", idx), runs.next());
			idx++;
		}
		//addExtra(jg, "runs on request", Joiner.on(", ").join(resourceReader.getRunsOnRequest(res)));
		idx = 1;
		Iterator<String> hosts = resourceReader.getHostedStuff(res);
		while(hosts.hasNext()){
			addExtra(jg, String.format("hosts (%d)", idx), hosts.next());
			idx++;
		}
		//addExtra(jg, "hosts", Joiner.on(", ").join(resourceReader.getHostedStuff(res)));
		idx = 1;
		Iterator<String> curates = resourceReader.getCuratedObjects(res);
		while(curates.hasNext()){
			addExtra(jg, String.format("curates (%d)", idx), curates.next());
			idx++;
		}
		//addExtra(jg, "curates", Joiner.on(", ").join(resourceReader.getCuratedObjects(res)));
		addExtra(jg, "declared begin/end of operation", Joiner.on(", ").join(resourceReader.getDeclarativeTimes(res)));
		addExtra(jg, "availability", resourceReader.getAvailability(res));
		idx = 1;
		Iterator<String> plans = resourceReader.getCurationPlans(res);
		while(plans.hasNext()){
			addExtra(jg, String.format("uses curation plan (%d)", idx), plans.next());
			idx++;
		}
		//addExtra(jg, "uses curation plan", Joiner.on(", ").join(resourceReader.getCurationPlans(res)));
		addExtra(jg, "time of service", Joiner.on(", ").join(resourceReader.getDeclarativeTimes(res)));
		//TODO: where to get it?
		//addExtra(jg, "last confirmation", "");
		//TODO: where to get it?
		//addExtra(jg, "date of registration", "");

		jg.writeEndArray(); //end extras

		jg.writeEndObject();
		jg.close();
		return out.toString("UTF-8");
	}

	protected String getJsonForActor(final Resource res, final String resNameForCatalogue, final String datasourceName)
			throws IOException, ParthenosPublisherException {
		JsonFactory jsonFactory = new JsonFactory();
		final ByteArrayOutputStream out = new ByteArrayOutputStream();
		BufferedOutputStream bos = new BufferedOutputStream(out);
		JsonGenerator jg = jsonFactory.createGenerator(bos, JsonEncoding.UTF8);
		jg.writeStartObject();
		writeCommonFields(jg, res, resNameForCatalogue, datasourceName);

		jg.writeArrayFieldStart("extras");
		addExtra(jg, "system:type", CKANUtils.Actor_type);
		//specific class
		addExtra(jg, "instance of", resourceReader.findSpecificType(res, CRM.E39_Actor).getLocalName());
		if (res.getURI().startsWith(PARTHENOS_BASE_URL))
			addExtra(jg, "Parthenos URL", res.getURI());
		else addExtra(jg, "URL", res.getURI());
		addIdentifiers(res, jg);
		addExtra(jg, "has type", Joiner.on(", ").join(resourceReader.getHasTypeLabels(res)));
		int idx = 1;
		Iterator<String> hasMembers = resourceReader.getMemberUrls(res);
		while(hasMembers.hasNext()){
			addExtra(jg, String.format("has member (%d)", idx), hasMembers.next());
			idx++;
		}
		//addExtra(jg, "has member", Joiner.on(", ").join(resourceReader.getMemberUrls(res)));
		idx = 1;
		Iterator<String> isMembers = resourceReader.isMemberOf(res);
		while(isMembers.hasNext()){
			addExtra(jg, String.format("is member of (%d)", idx), isMembers.next());
			idx++;
		}
		//addExtra(jg, "is member of", Joiner.on(", ").join(resourceReader.isMemberOf(res)));
		idx = 1;
		Iterator<String> provides = resourceReader.getProvidedServiceUrls(res);
		while(provides.hasNext()){
			addExtra(jg, String.format("provides (%d)", idx), provides.next());
			idx++;
		}
		//addExtra(jg, "provides", Joiner.on(", ").join(resourceReader.getProvidedServiceUrls(res)));

		idx = 1;
		String contactPoints = "";
		StmtIterator it = res.listProperties(CRM.P76_has_contact_point);
		while(it.hasNext()) {
			Resource cp = it.next().getResource();
			Resource cpType = cp.getPropertyResourceValue(CRM.P2_has_type);
			String cpTypeLabel = resourceReader.getLabel(cpType);
			String cpLabel = resourceReader.getLabel(cp);
			if (StringUtils.isNotBlank(cpLabel)) {
				if (StringUtils.isNotBlank(cpTypeLabel)) {
					addExtra(jg,String.format("contact point (%d) - %s ", idx, cpTypeLabel), cpLabel );
					idx++;
					//contactPoints += cpTypeLabel + ": ";
				}
				else{
					addExtra(jg,String.format("contact point (%d)", idx), cpLabel );
					idx++;
					//contactPoints += cpLabel + "; ";
				}

			}
			else{
				addExtra(jg,String.format("contact point (%d)", idx), cp.getURI());
				idx++;
			}
		}
	//	addExtra(jg,"contact points", contactPoints );
		idx = 1;
		Iterator<String> maintains = resourceReader.getMaintainedUrls(res);
		while(maintains.hasNext()){
			addExtra(jg, String.format("maintains (%d)", idx), maintains.next());
			idx++;
		}
		//addExtra(jg, "maintains", Joiner.on(", ").join(resourceReader.getMaintainedUrls(res)));

		jg.writeEndArray();

		jg.writeEndObject();
		jg.close();
		return out.toString("UTF-8");
	}

	protected String getJsonForDataset(final Resource res, final String resNameForCatalogue, final String datasourceName)
			throws IOException, ParthenosPublisherException {
		JsonFactory jsonFactory = new JsonFactory();
		final ByteArrayOutputStream out = new ByteArrayOutputStream();
		BufferedOutputStream bos = new BufferedOutputStream(out);
		JsonGenerator jg = jsonFactory.createGenerator(bos, JsonEncoding.UTF8);
		jg.writeStartObject();
		writeCommonFields(jg, res, resNameForCatalogue, datasourceName);

		jg.writeArrayFieldStart("extras");
		addExtra(jg, "system:type", CKANUtils.Dataset_type);
		//specific class
		addExtra(jg, "instance of", resourceReader.findSpecificType(res, CRM.E70_Thing).getLocalName());
		if (res.getURI().startsWith(PARTHENOS_BASE_URL)) {
			addExtra(jg, "Parthenos URL", res.getURI());
		}
		else addExtra(jg, "URL", res.getURI());
		addIdentifiers(res, jg);
		addExtra(jg, "has type", Joiner.on(", ").join(resourceReader.getHasTypeLabels(res)));
		int idx = 1;
		Iterator<String> isPartOf = resourceReader.getIsPartOfUrls(res);
		while(isPartOf.hasNext()){
			addExtra(jg, String.format("is part of (%d)", idx), isPartOf.next());
			idx++;
		}
		//addExtra(jg, "is part of", Joiner.on(", ").join(resourceReader.getIsPartOfUrls(res)));
		idx = 1;
		Iterator<String> hasPart = resourceReader.getHasPartUrls(res);
		while(hasPart.hasNext()){
			addExtra(jg, String.format("has part (%d)", idx), hasPart.next());
			idx++;
		}
		//addExtra(jg, "has part", Joiner.on(", ").join(resourceReader.getHasPartUrls(res)));
		idx = 1;
		Iterator<String> curators = resourceReader.getCuratorUrls(res);
		while(curators.hasNext()){
			addExtra(jg, String.format("curated by (%d)", idx), curators.next());
			idx++;
		}
		//addExtra(jg, "curated by", Joiner.on(", ").join(resourceReader.getCuratorUrls(res)));
		idx = 1;
		Iterator<String> curationplans = resourceReader.getResourceCuratorCurationPlans(res);
		while(curationplans.hasNext()){
			addExtra(jg, String.format("curation plan (%d)", idx), curationplans.next());
			idx++;
		}
		//addExtra(jg, "curation plan", Joiner.on(", ").join(resourceReader.getResourceCuratorCurationPlans(res)));
		idx = 1;
		Iterator<String> hostedbys = resourceReader.getHostedBys(res);
		while(hostedbys.hasNext()){
			addExtra(jg, String.format("hosted by (%d)", idx), hostedbys.next());
			idx++;
		}
		//addExtra(jg, "hosted by", Joiner.on(", ").join(resourceReader.getHostedBys(res)));
		addExtra(jg, "encoding type", Joiner.on(", ").join(resourceReader.getEncodings(res)));
		idx = 1;
		Iterator<String> creators = resourceReader.getCreatorsURIs(res);
		while(creators.hasNext()){
			addExtra(jg, String.format("creator (%d)", idx), creators.next());
			idx++;
		}
		//addExtra(jg, "creator", Joiner.on(", ").join(resourceReader.getCreatorsURIs(res)));
		idx = 1;
		Iterator<String> sw = resourceReader.getUsedSoftware(res);
		while(sw.hasNext()){
			addExtra(jg, String.format("used software (%d)", idx), sw.next());
			idx++;
		}

		addExtra(jg, "subject", Joiner.on(", ").join(resourceReader.getSubjects(res)));
		addExtra(jg, "temporal coverage", Joiner.on(", ").join(resourceReader.getTemporalCoverages(res)));
		addExtra(jg, "spatial coverage", Joiner.on(", ").join(resourceReader.getSpatialCoverages(res)));
		idx = 1;
		Iterator<String> usedbys = resourceReader.getUsedBy(res);
		while(usedbys.hasNext()){
			addExtra(jg, String.format("used by (%d)", idx), usedbys.next());
			idx++;
		}
		//addExtra(jg, "used by", Joiner.on(", ").join(resourceReader.getUsedBy(res)));
		addExtra(jg, "languages", Joiner.on(", ").join(resourceReader.getLanguages(res)));
		idx = 1;
		Iterator<String> metadata = resourceReader.getMetadata(res);
		while(metadata.hasNext()){
			addExtra(jg, String.format("has metadata (%d)", idx), metadata.next());
			idx++;
		}
		//addExtra(jg, "has metadata", Joiner.on(", ").join(resourceReader.getMetadata(res)));
		idx = 1;
		Iterator<String> metadataFor = resourceReader.getDescribedDataset(res);
		while(metadataFor.hasNext()){
			addExtra(jg, String.format("is metadata for (%d)", idx), metadataFor.next());
			idx++;
		}
		//addExtra(jg, "is metadata for", Joiner.on(", ").join(resourceReader.getDescribedDataset(res)));
		idx = 1;
		Iterator<String> snaphsots = resourceReader.getSnapshots(res);
		while(snaphsots.hasNext()){
			addExtra(jg, String.format("has snapshot (%d)", idx), snaphsots.next());
			idx++;
		}
		//addExtra(jg, "has snapshot", Joiner.on(", ").join(resourceReader.getSnapshots(res)));
		idx = 1;
		Iterator<String> issnaphsots = resourceReader.getIsSnapshotOfs(res);
		while(issnaphsots.hasNext()){
			addExtra(jg, String.format("is snapshot of (%d)", idx), issnaphsots.next());
			idx++;
		}
		//addExtra(jg, "is snapshot of", Joiner.on(", ").join(resourceReader.getIsSnapshotOfs(res)));

		jg.writeEndArray();

		jg.writeEndObject();
		jg.close();
		return out.toString("UTF-8");
	}

	protected String getJsonForSoftware(final Resource res, final String resNameForCatalogue, final String datasourceName)
			throws IOException, ParthenosPublisherException {
		JsonFactory jsonFactory = new JsonFactory();
		final ByteArrayOutputStream out = new ByteArrayOutputStream();
		BufferedOutputStream bos = new BufferedOutputStream(out);
		JsonGenerator jg = jsonFactory.createGenerator(bos, JsonEncoding.UTF8);
		jg.writeStartObject();
		writeCommonFields(jg, res, resNameForCatalogue, datasourceName);

		jg.writeArrayFieldStart("extras");
		addExtra(jg, "system:type", CKANUtils.Software_type);
		//specific class
		addExtra(jg, "instance of", resourceReader.findSpecificType(res, CRM.E70_Thing).getLocalName());
		if (res.getURI().startsWith(PARTHENOS_BASE_URL)) {
			addExtra(jg, "Parthenos URL", res.getURI());
		}
		else addExtra(jg, "URL", res.getURI());
		addIdentifiers(res, jg);
		addExtra(jg, "subject", Joiner.on(", ").join(resourceReader.getSubjects(res)));
		addExtra(jg, "has type", Joiner.on(", ").join(resourceReader.getHasTypeLabels(res)));
		int idx = 1;
		Iterator<String> hosted = resourceReader.getHostedBys(res);
		while(hosted.hasNext()){
			addExtra(jg, String.format("hosted by (%d)", idx), hosted.next());
			idx++;
		}
		//addExtra(jg, "hosted by", Joiner.on(", ").join(resourceReader.getHostedBys(res)));
		idx = 1;
		Iterator<String> curated = resourceReader.getCuratorUrls(res);
		while(curated.hasNext()){
			addExtra(jg, String.format("curated by (%d)", idx), curated.next());
			idx++;
		}
		//addExtra(jg, "curated by", Joiner.on(", ").join(resourceReader.getCuratorUrls(res)));
		idx = 1;
		Iterator<String> hasSnapshot = resourceReader.getSnapshots(res);
		while(hasSnapshot.hasNext()){
			addExtra(jg, String.format("has snapshot (%d)", idx), hasSnapshot.next());
			idx++;
		}
		//addExtra(jg, "has snapshot", Joiner.on(", ").join(resourceReader.getSnapshots(res)));
		idx = 1;
		Iterator<String> isSnapshot = resourceReader.getIsSnapshotOfs(res);
		while(isSnapshot.hasNext()){
			addExtra(jg, String.format("is snapshot (%d)", idx), isSnapshot.next());
			idx++;
		}
		//addExtra(jg, "is snapshot of", Joiner.on(", ").join(resourceReader.getIsSnapshotOfs(res)));
		idx = 1;
		Iterator<String> isPart = resourceReader.getIsPartOfUrls(res);
		while(isPart.hasNext()){
			addExtra(jg, String.format("is part of (%d)", idx), isPart.next());
			idx++;
		}
		//addExtra(jg, "is part of", Joiner.on(", ").join(resourceReader.getIsPartOfUrls(res)));
		idx = 1;
		Iterator<String> hasPart = resourceReader.getHasPartUrls(res);
		while(hasPart.hasNext()){
			addExtra(jg, String.format("has part (%d)", idx), hasPart.next());
			idx++;
		}
		//addExtra(jg, "has part", Joiner.on(", ").join(resourceReader.getHasPartUrls(res)));
		idx = 1;
		Iterator<String> hasRelease = resourceReader.getHasReleases(res);
		while(hasRelease.hasNext()){
			addExtra(jg, String.format("has release (%d)", idx), hasRelease.next());
			idx++;
		}
		//addExtra(jg, "has release", Joiner.on(", ").join(resourceReader.getHasReleases(res)));
		idx = 1;
		Iterator<String> isRelease = resourceReader.getIsReleaseOfs(res);
		while(isRelease.hasNext()){
			addExtra(jg, String.format("is release (%d)", idx), isRelease.next());
			idx++;
		}
		//addExtra(jg, "is release of", Joiner.on(", ").join(resourceReader.getIsReleaseOfs(res)));
		idx = 1;
		Iterator<String> used = resourceReader.getUsedBy(res);
		while(used.hasNext()){
			addExtra(jg, String.format("used by (%d)", idx), used.next());
			idx++;
		}
		//addExtra(jg, "used by", Joiner.on(", ").join(resourceReader.getUsedBy(res)));
		addExtra(jg, "creation time", resourceReader.getFirstCreationTime(res));

		jg.writeEndArray();

		jg.writeEndObject();
		jg.close();
		return out.toString("UTF-8");
	}

	protected String getJsonForCollection(final Resource res, final String resNameForCatalogue, final String datasourceName)
			throws IOException, ParthenosPublisherException {
		JsonFactory jsonFactory = new JsonFactory();
		final ByteArrayOutputStream out = new ByteArrayOutputStream();
		BufferedOutputStream bos = new BufferedOutputStream(out);
		JsonGenerator jg = jsonFactory.createGenerator(bos, JsonEncoding.UTF8);
		jg.writeStartObject();
		writeCommonFields(jg, res, resNameForCatalogue, datasourceName);

		jg.writeArrayFieldStart("extras");
		addExtra(jg, "system:type", CKANUtils.Collection_type);
		//specific class
		addExtra(jg, "instance of", resourceReader.findSpecificType(res, CRM.E70_Thing).getLocalName());
		if (res.getURI().startsWith(PARTHENOS_BASE_URL)) {
			addExtra(jg, "Parthenos URL", res.getURI());
		}
		else addExtra(jg, "URL", res.getURI());
		addIdentifiers(res, jg);
		addExtra(jg, "has type", Joiner.on(", ").join(resourceReader.getHasTypeLabels(res)));
		addExtra(jg, "subject", Joiner.on(", ").join(resourceReader.getSubjects(res)));
		addExtra(jg, "temporal coverage", Joiner.on(", ").join(resourceReader.getTemporalCoverages(res)));
		addExtra(jg, "spatial coverage", Joiner.on(", ").join(resourceReader.getSpatialCoverages(res)));
		int idx = 1;
		Iterator<String> hasPart = resourceReader.getHasPartUrls(res);
		while(hasPart.hasNext()){
			addExtra(jg, String.format("has part (%d)", idx), hasPart.next());
			idx++;
		}
		//addExtra(jg, "has part", Joiner.on(", ").join(resourceReader.getHasPartUrls(res)));
		idx = 1;
		Iterator<String> hosted = resourceReader.getHostedBys(res);
		while(hosted.hasNext()){
			addExtra(jg, String.format("hosted by (%d)", idx), hosted.next());
			idx++;
		}
		//addExtra(jg, "hosted by", Joiner.on(", ").join(resourceReader.getHostedBys(res)));
		idx = 1;
		Iterator<String> curated = resourceReader.getCuratorUrls(res);
		while(curated.hasNext()){
			addExtra(jg, String.format("curated by (%d)", idx), curated.next());
			idx++;
		}
		//addExtra(jg, "curated by", Joiner.on(", ").join(resourceReader.getCuratorUrls(res)));
		idx = 1;
		Iterator<String> creators = resourceReader.getCreatorsURIs(res);
		while(creators.hasNext()){
			addExtra(jg, String.format("creator (%d)", idx), creators.next());
			idx++;
		}
		//addExtra(jg, "creator", Joiner.on(", ").join(resourceReader.getCreatorsURIs(res)));
		addExtra(jg, "languages", Joiner.on(", ").join(resourceReader.getLanguages(res)));

		jg.writeEndArray();

		jg.writeEndObject();
		jg.close();
		return out.toString("UTF-8");
	}


	protected String getJsonForDesignProcedure(final Resource res, final String resNameForCatalogue, final String datasourceName)
			throws IOException, ParthenosPublisherException {
		JsonFactory jsonFactory = new JsonFactory();
		final ByteArrayOutputStream out = new ByteArrayOutputStream();
		BufferedOutputStream bos = new BufferedOutputStream(out);
		JsonGenerator jg = jsonFactory.createGenerator(bos, JsonEncoding.UTF8);
		jg.writeStartObject();
		writeCommonFields(jg, res, resNameForCatalogue, datasourceName);

		jg.writeArrayFieldStart("extras");
		addExtra(jg, "system:type", CKANUtils.DesignOrProcedure_type);
		//specific class
		addExtra(jg, "instance of", resourceReader.findSpecificType(res, CRM.E29_Design_or_Procedure).getLocalName());
		if (res.getURI().startsWith(PARTHENOS_BASE_URL)) {
			addExtra(jg, "Parthenos URL", res.getURI());
		}
		else addExtra(jg, "URL", res.getURI());
		addIdentifiers(res, jg);
		int idx = 1;
		Iterator<String> used = resourceReader.getUsedBy(res);
		while(used.hasNext()){
			addExtra(jg, String.format("used by (%d)", idx), used.next());
			idx++;
		}
		//addExtra(jg, "used by", Joiner.on(", ").join(resourceReader.getUsedBy(res)));
		//TODO: add additional metadata for E29_Design_or_Procedure, if any
		jg.writeEndArray();

		jg.writeEndObject();
		jg.close();
		return out.toString("UTF-8");
	}

	protected void addIdentifiers(final Resource res, final JsonGenerator jg ) throws IOException {
		/*
		 <${subjectURL}> crm:P1_is_identified_by ?IDRes .
		 ?IDRes a crm:E42_Identifier .
		 ?IDRes rdfs:label ?ID_label .
		 */
		StmtIterator it = res.listProperties(CRM.P1_is_identified_by);
		Set<String> ids = Sets.newHashSet();
		while(it.hasNext()){
			RDFNode obj = it.next().getObject();
			if(obj.isLiteral()) ids.add(obj.asLiteral().getLexicalForm());
			else {
				Resource id = (Resource) obj;
				if (id.hasProperty(RDF.type, CRM.E42_Identifier)) {
					ids.add(resourceReader.getLabel(id));
				}
			}
		}
		addExtra(jg, "ID", String.join(",", ids));
	}

	protected void addExtra(final JsonGenerator jg, final String key, final String value) throws IOException {
		if(StringUtils.isNotBlank(value)) {
			jg.writeStartObject();
			jg.writeStringField("key", key);
			jg.writeStringField("value", value);
			jg.writeEndObject();
		}
	}



	protected void writeCommonFields(final JsonGenerator jg, final Resource res, final String resNameForCatalogue, final String datasourceName)
			throws IOException, ParthenosPublisherException {
		String ckanOrg = CKANUtils.getCKanOrg(datasourceName);
		//id is available only for updates
		if(StringUtils.isBlank(ckanOrg)) throw new ParthenosPublisherException(String.format("Cannot register %s : blank ckan org for data source with name %s", resNameForCatalogue, datasourceName));
		//the owning organization, i.e. the data souce from which this resource has been collected from
		jg.writeStringField("owner_org", ckanOrg);
		jg.writeStringField("name", resNameForCatalogue);
		//default license
		jg.writeStringField("license_id", resourceReader.getCatalogueLicense(res).getId());
		String title = resourceReader.getTitle(res);
		if (StringUtils.isBlank(title))
			title = resNameForCatalogue;
		jg.writeStringField("title", title);
		//description
		jg.writeStringField("notes",Joiner.on(';').join(resourceReader.getDescriptions(res)));
		//the names of all superclasses of the entity
		jg.writeArrayFieldStart("tags");
		Iterator<String> classNames = resourceReader.getRDFClassNames(res);
		while (classNames.hasNext()) {
			jg.writeStartObject();
			jg.writeStringField("name", classNames.next());
			jg.writeEndObject();
		}
		jg.writeEndArray();
	}


	public ResourceReader getResourceReader() {
		return resourceReader;
	}

	public void setResourceReader(final ResourceReader resourceReader) {
		this.resourceReader = resourceReader;
	}

	public CatalogueAPIClient getCatalogueAPIClient() {
		return catalogueAPIClient;
	}

	public void setCatalogueAPIClient(final CatalogueAPIClient catalogueAPIClient) {
		this.catalogueAPIClient = catalogueAPIClient;
	}
}
