package eu.dnetlib.parthenos.catalogue;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;

import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.Lists;
import eu.dnetlib.parthenos.jrr.ParthenosRegistryResource;
import eu.dnetlib.parthenos.publisher.ParthenosPublisherException;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.http.client.utils.URIBuilder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.*;
import org.springframework.stereotype.Component;
import org.springframework.web.client.HttpClientErrorException;
import org.springframework.web.client.RestTemplate;

import javax.annotation.PostConstruct;

/**
 * Created by Alessia Bardi on 08/03/2018.
 *
 * @author Alessia Bardi
 */
@Component
public class CatalogueAPIClient {

	private static final Log log = LogFactory.getLog(CatalogueAPIClient.class);

	@Value("${gcube.catalogue.baseurl}")
	private String baseURL;
	private final String itemPath = "items/";
	@Value("${gcube.registry.application.token}")
	private String applicationToken;
	@Value("${gcube.uri.resolver}")
	private String uriResolver;
	private String resolverBodyTemplate = "{ \"entity_name\": \"%s\" }";
	private String purgeBodyTemplate = "{\"id\":\"%s\"}";

	private HttpHeaders headersForResolver;
	private HttpHeaders headersForCatalogue;

	@Autowired
	private RestTemplate restTemplate;


	public CatalogueAPIClient(){}

	@PostConstruct
	public void init(){
		headersForResolver = new HttpHeaders();
		headersForResolver.setContentType(MediaType.APPLICATION_JSON);

		headersForCatalogue = new HttpHeaders();
		headersForCatalogue.setContentType(MediaType.APPLICATION_JSON);
		headersForCatalogue.setAccept(Lists.newArrayList(MediaType.APPLICATION_JSON));
		headersForCatalogue.set("gcube-token", getApplicationToken());
	}


	/**
	 * Calls the parthenos resolver to get the name to use for the catalogue
	 * @param resName Given a Parthenos URL like: http://parthenos.d4science.org/handle/Ariadne/AriadnePortal/rhdsq5xm3e40, the resName is Ariadne/AriadnePortal/rhdsq5xm3e40.
	 * @return the name as computed by the Parthenos resolver. Null if the request was not successfull.
	 */
	public String getNameForCatalogue(final String resName) throws URISyntaxException {
		log.debug("Calling Parthenos resolver for "+resName);
		String body = String.format(resolverBodyTemplate, resName);
		HttpEntity<String> entity = new HttpEntity<String>(body, headersForResolver);
		URI uri = new URIBuilder(getUriResolver()).build();
		log.debug("Resolver at "+getUriResolver()+" post body: "+body);
		try {
			ResponseEntity<String> res = restTemplate.exchange(uri, HttpMethod.POST, entity, String.class);
			if (res.getStatusCode().is2xxSuccessful()) {
				String resolved = res.getBody();
				log.debug(String.format("Parthenos resolver resolved %s with %s", resName, resolved));
				return resolved;
			} else {
				log.debug(String.format("Parthenos resolver returned %s with cause %s for %s", res.getStatusCodeValue(), res.getStatusCode().getReasonPhrase(), resName));
				return null;
			}
		}catch(HttpClientErrorException e){
			log.debug(String.format("Parthenos resolver returned %s with cause %s for %s", e.getRawStatusCode(), e.getStatusCode().getReasonPhrase(), resName));
			return null;
		}
	}

	public boolean isRegistered(final String resCatName) throws ParthenosPublisherException {
		log.debug(String.format("Catalogue --> Checking if item %s exists", resCatName));
		HttpEntity<String> entity = new HttpEntity<String>(headersForCatalogue);
		try {
			URI uri = new URIBuilder(getBaseURL()+itemPath+resCatName).build();
			log.info(uri);
			log.info(headersForCatalogue.toString());
			ResponseEntity<String> response
					= restTemplate.exchange(uri, HttpMethod.GET, entity, String.class);
			return response.getStatusCode().is2xxSuccessful();
		}catch(HttpClientErrorException e){
			if(e.getRawStatusCode() == 404){
				return false;
			}
			else throw new ParthenosPublisherException(e);
		} catch (URISyntaxException e) {
			throw new ParthenosPublisherException(e);
		}
	}

	@Deprecated
	public ParthenosRegistryResource getRegistered(final String resCatName) throws ParthenosPublisherException {
		log.debug(String.format("Catalogue --> Checking if item %s exists", resCatName));
		HttpEntity<String> entity = new HttpEntity<String>(headersForCatalogue);
		try {
			URI uri = new URIBuilder(getBaseURL()+itemPath).build();
			ResponseEntity<String> response
					= restTemplate.exchange(uri +resCatName, HttpMethod.GET, entity, String.class);
			if(response.getStatusCode().is2xxSuccessful()){
				CatalogueAPIResponse body = new CatalogueAPIResponse();
				body.setResponseBody(response.getBody());
				ParthenosRegistryResource r = body.getParthenosRegistryResource();
				log.debug(String.format("Resource %s is in the catalogue with uuid %s", resCatName, r.getUuid()));
				return r;
			}
			else{
				log.debug(String.format("Resource %s is not in the catalogue yet", resCatName));
				return null;
			}
		} catch (Throwable t) {
			throw new ParthenosPublisherException(t);
		}
	}

	public boolean doRegister(final String json, final String resCatName) throws URISyntaxException, IOException {
		log.debug(String.format("Catalogue --> Registering item %s : %s", resCatName, json));
		return doCatalogueCall(json, resCatName, HttpMethod.POST);
	}

	public boolean doUpdate(final String json, final String resCatName) throws IOException, URISyntaxException {
		log.debug(String.format("Catalogue --> Updating item %s : %s", resCatName, json));
		return doCatalogueCall(json, resCatName, HttpMethod.PUT);
	}

	protected boolean doCatalogueCall(final String json, final String resCatName, final HttpMethod method) throws URISyntaxException, IOException {
		HttpEntity<String> entity = new HttpEntity<String>(json, headersForCatalogue);
		URI uri = new URIBuilder(getBaseURL()+itemPath).build();
		try {
			ResponseEntity<String> res = restTemplate.exchange(uri, method, entity, String.class);
			if (res.getStatusCode().is2xxSuccessful()) {
				return true;
			} else {
				log.error(String.format("Method %s on resource %s failed with code %s, reason: %s", method.name(), resCatName, res.getStatusCodeValue(), res.getBody()));
				return false;
			}
		}catch(HttpClientErrorException e){
			log.error(String.format("Method %s on resource %s failed with code %s, reason: %s", method.name(), resCatName, e.getRawStatusCode(), e.getStatusCode().getReasonPhrase()));
			return false;
		}
	}

	protected boolean purgeItem(final String resCatName) throws URISyntaxException, ParthenosPublisherException {
		log.debug(String.format("Catalogue --> Purge Item %s", resCatName));
		HttpEntity<String> entity = new HttpEntity<String>(headersForCatalogue);
		URI uri = new URIBuilder(getBaseURL()+itemPath+resCatName).addParameter("purge", "true").build();
		try{
			ResponseEntity<String> res
					= restTemplate.exchange(uri, HttpMethod.DELETE, entity, String.class);
			if (res.getStatusCode().is2xxSuccessful()) {
				return true;
			} else {
				log.error(String.format("Cannot purge item %s. HTTP error code %s, reason: %s", resCatName, res.getStatusCodeValue(), res.getStatusCode().getReasonPhrase()));
				return false;
			}
		} catch(HttpClientErrorException e){
			log.error(String.format("Cannot purge item %s. HTTP error code %s, reason: %s", resCatName, e.getRawStatusCode(), e.getStatusCode().getReasonPhrase()));
			return false;
		} catch (Throwable t) {
			throw new ParthenosPublisherException(t);
		}
	}

	protected int purgeAll(int bulkSize) throws ParthenosPublisherException {
		log.debug("Catalogue --> Purge All Item");
		int count = 0;
		try {
			List<String> items = list(bulkSize);
			while (items.size() > 0){
				for(String itemName : items) {
					purgeItem(itemName);
					count++;

						Thread.sleep(50);

				}
				items = list(200);
				Thread.sleep(TimeUnit.SECONDS.toMillis(5));
			}
			} catch (Throwable t) {
				throw new ParthenosPublisherException(t);
		}
		log.debug(String.format("Catalogue --> Purged all %d Items", count));
		return count;
	}

	protected List<String> list(int limit) throws ParthenosPublisherException {
		log.debug(String.format("Getting list of items"));
		ObjectMapper mapper = new ObjectMapper();
		JavaType listType = mapper.getTypeFactory().constructCollectionType(ArrayList.class, String.class);
		HttpEntity<String> entity = new HttpEntity<String>(headersForCatalogue);
		try {
			URI uri = new URIBuilder(getBaseURL()+itemPath).addParameter("limit", String.valueOf(limit)).addParameter("offset", "0").build();
			ResponseEntity<String> response
					= restTemplate.exchange(uri, HttpMethod.GET, entity, String.class);
			if(response.getStatusCode().is2xxSuccessful()){
				String res = response.getBody();
				return mapper.readValue(res, listType);
			}
			throw new ParthenosPublisherException("Cannot get list of items to purge");
		} catch (Throwable t) {
			throw new ParthenosPublisherException(t);
		}
	}


	public String getBaseURL() {
		return baseURL;
	}

	public void setBaseURL(final String baseURL) {
		this.baseURL = baseURL;
	}

	public String getApplicationToken() {
		return applicationToken;
	}

	public void setApplicationToken(final String applicationToken) {
		this.applicationToken = applicationToken;
	}

	public RestTemplate getRestTemplate() {
		return restTemplate;
	}

	public void setRestTemplate(final RestTemplate restTemplate) {
		this.restTemplate = restTemplate;
	}

	public String getUriResolver() {
		return uriResolver;
	}

	public void setUriResolver(final String uri_resolver) {
		this.uriResolver = uriResolver;
	}

	public String getPurgeBodyTemplate() {
		return purgeBodyTemplate;
	}

	public void setPurgeBodyTemplate(final String purgeBodyTemplate) {
		this.purgeBodyTemplate = purgeBodyTemplate;
	}

}
