package eu.dnetlib.functionality.modular.ui.patcheditor.record;

import com.mongodb.client.MongoCollection;
import eu.dnetlib.data.mdstore.modular.connector.MDStoreManagerInfo;
import eu.dnetlib.data.mdstore.modular.mongodb.MDStoreTransactionManagerImpl;
import eu.dnetlib.enabling.locators.UniqueServiceLocator;
import eu.dnetlib.functionality.modular.ui.patcheditor.exceptions.PatchEditorException;
import eu.dnetlib.functionality.modular.ui.workflows.util.ISLookupClient;
import eu.dnetlib.miscutils.datetime.DateUtils;
import eu.dnetlib.msro.logging.DnetLogger;
import eu.dnetlib.msro.workflows.util.WorkflowsConstants;
import eu.dnetlib.rmi.data.MDStoreServiceException;
import eu.dnetlib.rmi.enabling.ISLookUpException;
import eu.dnetlib.rmi.enabling.ISLookUpService;
import java.io.StringReader;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.annotation.Resource;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.bson.Document;
import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;
import org.springframework.beans.factory.annotation.Autowired;

public class PatchManagerImpl implements PatchManager {

	@Autowired
	private UniqueServiceLocator serviceLocator;

	@Autowired
	private ISLookupClient isLookupClient;

	@Autowired
	private MDStoreTransactionManagerImpl mdstoreTransactionManager;

	@Resource(name = "msroWorkflowLogger")
	private DnetLogger dnetLogger;

	private final SAXReader reader = new SAXReader();

	private static final Log log = LogFactory.getLog(PatchManager.class); // NOPMD by marko on 11/24/08 5:02 PM

	@Override
	public void commit(PatchedRecord record) throws PatchEditorException {

		try {
			final String patchMdStoreId = getPatchMDStoreId(record.getRepositoryId());
			final MongoCollection<Document> collection = getPatchMdStoreMongoCollection(patchMdStoreId);
			final Document mongoQuery = new Document().append("id", record.getId());
			final Document storePatches = collection.find(mongoQuery).first();
			if (storePatches == null) {
				final Document patched_document = mongoQuery
						.append("body", record.asXML())
						.append("timestamp", System.currentTimeMillis());
				collection.insertOne(patched_document);
			} else {
				final String oldPatches = storePatches.getString("body");

				collection.findOneAndUpdate(mongoQuery, new Document("$set", new Document("body", mergePatches(record.asXML(), oldPatches))));
			}

			log.info("Committed patches on patchMdStoreId" + patchMdStoreId);
		} catch (final Throwable e) {
			throw new PatchEditorException(e);
		}

	}

	public String mergePatches(String newPatch, final String oldPatches) {
		try {
			final org.dom4j.Document newDoc = reader.read(new StringReader(newPatch));

			final org.dom4j.Document oldDoc = reader.read(new StringReader(oldPatches));
			final Element e = (Element) oldDoc.selectSingleNode("//patches");
			for (final Object o : newDoc.selectNodes("//patch")) {
				e.add(((Element) o).createCopy());
			}
			oldDoc.selectSingleNode("//*[local-name()='dateOfCollection']").setText(DateUtils.now_ISO8601());
			return oldDoc.asXML();
		} catch (final Exception e) {
			return newPatch;
		}
	}

	private String getPatchMDStoreId(final String repositoryId) throws ISLookUpException {

		final String getPatchMdstoreIdQuery = "for $x in collection('/db/DRIVER/WorkflowDSResources/WorkflowDSResourceType') where "
				+ "($x//DATASOURCE/@id/string()='%s') return $x//PARAM[./@name='patchMdstoreId']/text()";

		final ISLookUpService lookUpClient = serviceLocator.getService(ISLookUpService.class);

		final List<String> patchMdStoreIds = lookUpClient.quickSearchProfile(String.format(getPatchMdstoreIdQuery, repositoryId));

		if (patchMdStoreIds == null || patchMdStoreIds.size() != 1) { throw new RuntimeException(
				"Unexpected number of result executing query " + String.format(getPatchMdstoreIdQuery, patchMdStoreIds.get(0)) + " expected: 1"); }

		return patchMdStoreIds.get(0);
	}

	private MongoCollection<Document> getPatchMdStoreMongoCollection(final String mdStoreId) throws MDStoreServiceException {

		final MDStoreManagerInfo info = mdstoreTransactionManager.getInfoForCurrentMdStore(mdStoreId);
		return mdstoreTransactionManager.getDb().getCollection(info.getCurrentId());
	}

	private String retrieveOldPatchesXml(String repositoryId, String recordId) throws PatchEditorException {
		String patchMdStoreId;
		try {
			patchMdStoreId = getPatchMDStoreId(repositoryId);
			MongoCollection<Document> collection;

			collection = getPatchMdStoreMongoCollection(patchMdStoreId);
			final Document mongoQuery = new Document().append("id", recordId);
			final Document storePatches = collection.find(mongoQuery).first();
			if (storePatches != null) {
				final String oldPatches = storePatches.getString("body");

				return oldPatches;
			}
		} catch (final ISLookUpException | MDStoreServiceException e) {
			throw new PatchEditorException(e);
		}
		return "";
	}

	private String getLastIndexDate(final String repositoryId) throws ISLookUpException {

		final String getWorkflowIdQuery = "for $x in collection('/db/DRIVER/WorkflowDSResources/WorkflowDSResourceType') where "
				+ "($x//DATASOURCE/@id/string()='%s') return  $x//RESOURCE_IDENTIFIER/@value/string()";

		final ISLookUpService lookUpClient = serviceLocator.getService(ISLookUpService.class);

		final List<String> workflowId = lookUpClient.quickSearchProfile(String.format(getWorkflowIdQuery, repositoryId));

		if (workflowId == null || workflowId.size() != 1) { throw new RuntimeException(
				"Unexpected number of result executing query " + String.format(getWorkflowIdQuery, repositoryId) + " expected: 1"); }
		final String wfId = workflowId.get(0);

		final List<Map<String, String>> subWfs = isLookupClient.obtainSubWorkflows(wfId);
		String lastIndexDate = "";
		for (final Map<String, String> map : subWfs) {
			if ("true".equalsIgnoreCase(map.get("isTemplate")) && "index".equalsIgnoreCase(map.get("name"))) {
				final Map<String, Object> query = new HashMap<>();
				query.put(WorkflowsConstants.LOG_WF_PARENT, wfId);
				query.put(WorkflowsConstants.LOG_WF_PROFILE_TEMPLATE_ID, map.get("id"));
				final Map<String, String> log = dnetLogger.findOne(query);
				if (log != null && !log.isEmpty()) {
					lastIndexDate = log.get(WorkflowsConstants.LOG_WF_PROCESS_END_DATE);
				}
			}
		}
		return lastIndexDate;
	}

	private boolean existOneNotIndexedPatch(final String storedPatches, String lastIndexDatePlain) throws DocumentException {

		final DateUtils dateUtils = new DateUtils();
		final Date lastIndexDate = dateUtils.parse(DateUtils.calculate_ISO8601(Long.parseLong(lastIndexDatePlain)));

		final SAXReader reader = new SAXReader();
		final org.dom4j.Document patchesDoc = reader.read(new StringReader(storedPatches));

		for (final Object o : patchesDoc.selectNodes("//patch")) {
			final String xpath = ((Element) o).valueOf("@xpath");
			final String value = ((Element) o).valueOf("@value");
			final String creationDatePlain = ((Element) o).valueOf("@creationDate");
			if (StringUtils.isNoneEmpty(creationDatePlain)) {
				final Date creationDate = dateUtils.parse(creationDatePlain);
				if (creationDate.after(lastIndexDate)) { return true; }
			}
		}
		return false;
	}

	@Override
	public boolean existPatchNotIndexed(String repositoryId, String recordId) throws PatchEditorException {

		final String oldPatchesXml = retrieveOldPatchesXml(repositoryId, recordId);
		if (!StringUtils.isEmpty(oldPatchesXml)) {
			log.info("Patches found for recordId: " + recordId + " on repositoryId: " + repositoryId);
		} else {
			log.info("No patches found for recordId: " + recordId + " on repositoryId: " + repositoryId);
			return false;
		}
		String lastIndexDate = null;
		try {
			lastIndexDate = getLastIndexDate(repositoryId);
			final boolean found = existOneNotIndexedPatch(oldPatchesXml, lastIndexDate);
			if (found) {
				log.info("Found not indexed patches");
			}
			return found;
		} catch (final ISLookUpException | DocumentException e) {
			throw new PatchEditorException(e);
		}
	}
}
