package eu.dnetlib.functionality.modular.ui.patcheditor;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.gson.Gson;
import eu.dnetlib.enabling.locators.UniqueServiceLocator;
import eu.dnetlib.functionality.modular.ui.AbstractAjaxController;
import eu.dnetlib.functionality.modular.ui.patcheditor.client.IndexPatchEditorClient;
import eu.dnetlib.functionality.modular.ui.patcheditor.converters.EfgAvCreationConverter;
import eu.dnetlib.functionality.modular.ui.patcheditor.converters.EfgNonAvCreationConverter;
import eu.dnetlib.functionality.modular.ui.patcheditor.exceptions.PatchEditorException;
import eu.dnetlib.functionality.modular.ui.patcheditor.record.LSCH;
import eu.dnetlib.functionality.modular.ui.patcheditor.record.Patch;
import eu.dnetlib.functionality.modular.ui.patcheditor.record.PatchManager;
import eu.dnetlib.functionality.modular.ui.patcheditor.record.PatchedRecord;
import eu.dnetlib.functionality.modular.ui.patcheditor.record.Record;
import eu.dnetlib.functionality.modular.ui.patcheditor.record.UIAction;
import eu.dnetlib.functionality.modular.ui.patcheditor.record.UIAction.EntityTypes;
import eu.dnetlib.functionality.modular.ui.vocabularies.model.Relation;
import eu.dnetlib.functionality.modular.ui.vocabularies.model.Synonym;
import eu.dnetlib.functionality.modular.ui.vocabularies.model.Term;
import eu.dnetlib.functionality.modular.ui.vocabularies.persistence.VocabularyException;
import eu.dnetlib.rmi.enabling.ISLookUpException;
import eu.dnetlib.rmi.enabling.ISLookUpService;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import javax.annotation.Resource;
import javax.servlet.http.HttpSession;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.ModelMap;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

@Controller
public class PatchEditorController extends AbstractAjaxController {

	private static final Log log = LogFactory.getLog(PatchEditorController.class);

	@Resource
	private PatchManager patchManager;

	@Autowired
	private UniqueServiceLocator serviceLocator;

	@Autowired
	private IndexPatchEditorClient indexPatchEditorClient;

	@RequestMapping(path = "/ui/patcheditor/titleTypes", method = RequestMethod.GET)
	public @ResponseBody List<Term> getTitleTypes() throws Exception {
		final List<Term> terms = getVocabularies("AVTitleType");
		return terms;
	}

	@RequestMapping(path = "/ui/patcheditor/languages", method = RequestMethod.GET)
	public @ResponseBody List<Term> getLanguages() throws Exception {
		final List<Term> terms = getVocabularies("Names of Languages");
		return terms;
	}

	@RequestMapping(path = "/ui/patcheditor/countries", method = RequestMethod.GET)
	public @ResponseBody List<Term> getCountries() throws Exception {
		final List<Term> terms = getVocabularies("Iso3166-1CountryCS");
		return terms;
	}

	@RequestMapping(path = "/ui/patcheditor/keywordTypes", method = RequestMethod.GET)
	public @ResponseBody List<Term> getKeywordTypes() throws Exception {
		final List<Term> terms = getVocabularies("KeywordType");
		return terms;
	}

	@RequestMapping(path = "/ui/patcheditor/sounds", method = RequestMethod.GET)
	public @ResponseBody List<Term> getSounds() throws Exception {
		final List<Term> terms = getVocabularies("Sound");
		return terms;
	}

	@RequestMapping(path = "/ui/patcheditor/colours", method = RequestMethod.GET)
	public @ResponseBody List<Term> getColours() throws Exception {
		final List<Term> terms = getVocabularies("Colour");
		return terms;
	}

	@RequestMapping(path = "/ui/patcheditor/languageUsages", method = RequestMethod.GET)
	public @ResponseBody List<Term> getLanguageUsages() throws Exception {
		final List<Term> terms = getVocabularies("LanguageUsage");
		return terms;
	}

	@RequestMapping(path = "/ui/patcheditor/descriptionTypes", method = RequestMethod.GET)
	public @ResponseBody List<Term> getDescriptionTypes() throws Exception {
		final List<Term> terms = getVocabularies("DescriptionType");
		return terms;
	}

	@RequestMapping(path = "/ui/patcheditor/rightsStatus", method = RequestMethod.GET)
	public @ResponseBody List<Term> getRightsStatus() throws Exception {
		final List<Term> terms = getVocabularies("RightsStatus");
		return terms;
	}

	@RequestMapping(path = "/ui/patcheditor/gauge", method = RequestMethod.GET)
	public @ResponseBody List<Term> getGauge() throws Exception {
		final List<Term> terms = getVocabularies("Gauge");
		return terms;
	}

	@RequestMapping(path = "/ui/patcheditor/lsch", method = RequestMethod.GET)
	public @ResponseBody List<LSCH> getLSCH() throws Exception {
		final List<LSCH> lschs = loadLSCH();
		return lschs;
	}

	@RequestMapping(path = "/ui/patcheditor/record", method = RequestMethod.GET)
	public @ResponseBody Record getRecord(@RequestParam(value = "recordId", required = true) final String recordId) throws Exception {

		final PatchedRecord patchedRecord = createPatchedRecord(recordId);

		return patchedRecord.getRecord();
	}

	@RequestMapping(path = "/ui/patcheditor/existPatchNotIndexed", method = RequestMethod.GET)
	public @ResponseBody String existPatchNotIndexed(@RequestParam(value = "repositoryId", required = true) final String repositoryId,
			@RequestParam(value = "recordId", required = true) final String recordId) throws Exception {

		final boolean existPatchNotIndexed = patchManager.existPatchNotIndexed(repositoryId, recordId);
		if (existPatchNotIndexed) { return "true"; }
		return "";
	}

	@RequestMapping(path = "/ui/applyPatches.do", method = RequestMethod.POST)
	public @ResponseBody String applyPatches(ModelMap map,
			HttpSession httpSession,
			@RequestParam(value = "id", required = true) String id,
			@RequestParam(value = "patches", required = true) String patches) throws Exception {
		generateActions(map, httpSession, id, patches);
		return "ok";
	}

	private List<Term> getVocabularies(final String codeValue) throws PatchEditorException {
		List<Term> terms = null;
		try {
			final String query =
					"for $x in collection('/db/DRIVER/VocabularyDSResources/VocabularyDSResourceType') where ($x//VOCABULARY_NAME/@code='" + codeValue
							+ "') return $x";
			final List<String> countries = serviceLocator.getService(ISLookUpService.class).quickSearchProfile(query);
			terms = getTerms(countries.get(0));
		} catch (final ISLookUpException e) {
			throw new PatchEditorException(e);
		}
		return terms;
	}

	private List<Term> getTerms(final String vocabulary) throws PatchEditorException {
		try {
			final Document doc = new SAXReader().read(new StringReader(vocabulary));
			final Map<String, Term> terms = Maps.newHashMap();
			for (final Object t : doc.selectNodes("//TERM")) {
				final Element termNode = (Element) t;
				final String code = termNode.valueOf("@code");
				if (!terms.containsKey(code)) {
					final Term term = new Term();
					term.setEnglishName(termNode.valueOf("@english_name"));
					term.setNativeName(termNode.valueOf("@native_name"));
					term.setEncoding(termNode.valueOf("@encoding"));
					term.setCode(code);
					term.setSynonyms(new ArrayList<Synonym>());
					term.setRelations(new ArrayList<Relation>());
					terms.put(code, term);
				}
				final Term term = terms.get(code);
				for (final Object s : termNode.selectNodes(".//SYNONYM")) {
					final Element synNode = (Element) s;
					final Synonym syn = new Synonym();
					syn.setTerm(synNode.valueOf("@term"));
					syn.setEncoding(synNode.valueOf("@encoding"));
					term.getSynonyms().add(syn);
				}
				for (final Object r : termNode.selectNodes(".//RELATION")) {
					final Element relNode = (Element) r;
					final Relation rel = new Relation();
					rel.setCode(relNode.valueOf("@code"));
					rel.setType(relNode.valueOf("@type"));
					term.getRelations().add(rel);
				}
				Collections.sort(term.getSynonyms());
				Collections.sort(term.getRelations());
			}
			final List<Term> list = Lists.newArrayList(terms.values());
			Collections.sort(list);
			return list;
		} catch (final DocumentException e) {
			throw new PatchEditorException(new VocabularyException(e));
		}
	}

	private List<LSCH> loadLSCH() {
		try {
			final BufferedReader br =
					new BufferedReader(new InputStreamReader(getClass().getResourceAsStream("/eu/dnetlib/web/LSCH.csv")));
			final Gson g = new Gson();
			String s;

			s = br.readLine();

			final List<LSCH> lschs = new ArrayList<LSCH>();

			while (s != null) {
				final String[] lsch = s.split(";");
				lschs.add(new LSCH(lsch[1], lsch[0], lsch[2]));
				s = br.readLine();
			}
			return lschs;
		} catch (final IOException e) {

			return null;
		}

	}

	private PatchedRecord createPatchedRecord(final String recordId) throws PatchEditorException {
		try {
			final String recordPlain = indexPatchEditorClient.getDocument(recordId);

			final SAXReader reader = new SAXReader();
			final Document doc = reader.read(new StringReader(recordPlain));
			final String repoId = doc.valueOf("//*[local-name()='repositoryId']");
			final String type = doc.valueOf("local-name(//*[local-name()='efgEntity']/*)");
			Record record = null;

			if (type.equals("avcreation")) {
				final EfgAvCreationConverter conv = new EfgAvCreationConverter();
				record = conv.calculateRecordForPatchEdit(doc.getRootElement());
			} else if (type.equals("nonavcreation")) {
				final EfgNonAvCreationConverter conv = new EfgNonAvCreationConverter();
				record = conv.calculateRecordForPatchEdit(doc.getRootElement());
			} else {
				throw new PatchEditorException("patches can be only applied for avcreation and nonavcreation types");
			}
			record.setRepositoryId(repoId);
			record.setEntityType(type);
			return new PatchedRecord(recordId, repoId, record, type, new ArrayList<Patch>());
		} catch (final Exception e) {
			log.error(ExceptionUtils.getStackTrace(e));
			throw new PatchEditorException(e);
		}
	}

	private void generateActions(ModelMap map,
			HttpSession httpSession,
			String recordId,
			String jsonPatches) {
		try {
			final PatchedRecord patchedRecord = createPatchedRecord(recordId);
			final String type = patchedRecord.getType();
			final EntityTypes entityType = UIAction.EntityTypes.valueOf(type);

			final List<Patch> patches = new ArrayList<Patch>();

			UIAction[] arrayAction;
			arrayAction = new Gson().fromJson(jsonPatches, UIAction[].class);

			for (final UIAction element : arrayAction) {
				element.setEntityType(entityType);
				patches.addAll(element.asPatches());
			}

			patchedRecord.setPatches(patches);

			patchManager.commit(patchedRecord);

			map.addAttribute("response", "OK");
		} catch (final Exception e) {
			log.error(ExceptionUtils.getStackTrace(e));
			map.addAttribute("response", "FAIL");
		}
	}
}
