package eu.dnetlib.functionality.modular.ui.patcheditor.record;

import eu.dnetlib.functionality.modular.ui.patcheditor.record.model.Operation;
import eu.dnetlib.functionality.modular.ui.patcheditor.record.model.Patch;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;
import org.dom4j.Namespace;
import org.dom4j.QName;

/**
 * The Class UIAction.
 */
public class UIAction implements Serializable {

	/** The Constant serialVersionUID. */
	private static final long serialVersionUID = -5689869014308142684L;

	private static final Log log = LogFactory.getLog(UIAction.class);

	private static final Namespace EFG_NS = new Namespace("efg", "http://www.europeanfilmgateway.eu/efg");

	/**
	 * The Enum FieldTypes.
	 */
	public enum FieldTypes {
		isShownAt, isShownBy, aggregator, thumbnail, title, rightsStatus, rightsholderUrl, keyword, description, duration, provenance, language, colour, sound, gauge, countryOfReference, productionYear, specificType
	}

	public enum EntityTypes {
		avcreation, nonavcreation
	}

	/*
	 * Title { operation="ADDSUBTREE", fieldType="title", value="TEXT::-::TYPE::-::LANG"} keyword, LCSH { operation="ADDSUBTREE",
	 * fieldType="keyword", value="id::-::TERM::-::SCHEME::-::TYPE::-::LANG"} (undef if a term is not present) LCSH example:
	 * sh85147437::-::Women and war::-::http://id.loc.gov/authorities/subjects/sh85147437::-::LCSH::-::EN description {
	 * operation="ADDSUBTREE", fieldType="description", value="TEXT::-::TYPE::-::LANG"} language { operation="ADDSUBTREE",
	 * fieldType="language", value="LANG::-::USAGE"} isShownAt,isShownBy, aggregator, thumbnail, countryOfReference, productionYear,
	 * rightsStatus,rightholderUrl, duration, provenance, gauge, specificType { operation="EDIT/DELETENODE", fieldType="type",
	 * value="value"} colour, sound { operation="EDIT/DELETENODE", fieldType="type", value="colour::-::hascolour"}
	 */

	/** The operation. */
	private Operation operation;

	/** The field type. */
	private FieldTypes fieldType;

	/** The value. */
	private String value;

	/**
	 * The entity type should be at the moment avcreation or nonavcreation .
	 */
	private EntityTypes entityType;

	/**
	 * Instantiates a new uI action.
	 */
	public UIAction() {}

	/**
	 * Instantiates a new uI action.
	 *
	 * @param operation
	 *            the operation
	 * @param fieldType
	 *            the field type
	 * @param value
	 *            the value
	 */
	public UIAction(Operation operation, FieldTypes fieldType, String value) {
		super();
		this.operation = operation;
		this.fieldType = fieldType;
		this.value = value;
	}

	/**
	 * Gets the value.
	 *
	 * @return the value
	 */
	public String getValue() {
		return value;
	}

	/**
	 * Sets the value.
	 *
	 * @param value
	 *            the new value
	 */
	public void setValue(String value) {
		this.value = value;
	}

	/**
	 * Gets the operation.
	 *
	 * @return the operation
	 */
	public Operation getOperation() {
		return operation;
	}

	/**
	 * Sets the operation.
	 *
	 * @param operation
	 *            the new operation
	 */
	public void setOperation(Operation operation) {
		this.operation = operation;
	}

	/**
	 * Gets the field type.
	 *
	 * @return the field type
	 */
	public FieldTypes getFieldType() {
		return fieldType;
	}

	/**
	 * Sets the field type.
	 *
	 * @param fieldType
	 *            the new field type
	 */
	public void setFieldType(FieldTypes fieldType) {
		this.fieldType = fieldType;
	}

	public EntityTypes getEntityType() {
		return entityType;
	}

	public void setEntityType(EntityTypes entityType) {
		this.entityType = entityType;
	}

	/**
	 * As patches.
	 *
	 * @return the list
	 */
	public List<Patch> asPatches() {
		final List<Patch> out = new ArrayList<Patch>();

		switch (fieldType) {
		case title:
			processTitle(out);
			break;
		case description:
			processDescription(out);
			break;
		case keyword:
			processKeyword(out);
			break;
		case isShownAt:
		case isShownBy:
		case aggregator:
			processItemField(out);
			break;
		case colour:
		case sound:
		case gauge:
			processFormatField(out);
			break;
		case thumbnail:
		case rightsStatus:
		case duration:
		case provenance:
		case specificType:
			processAvManifestationField(out);
			break;
		case rightsholderUrl:
			processManifestationAttribute("rightsHolder", "URL", out);
			break;
		case countryOfReference:
		case productionYear:
			processField(out);
			break;
		case language:
			processLanguage(out);
			break;
		default:
			break;
		}

		return out;
	}

	private void processLanguage(List<Patch> out) {
		final String[] arr = value.split("::-::");
		final Element lang = DocumentHelper.createElement(new QName("language", EFG_NS));
		lang.setText(arr[0]);
		lang.addAttribute("usage", arr[1]);
		switch (entityType) {
		case avcreation:
			out.add(new Patch("//*[local-name()='avManifestation']", Operation.ADDSUBTREE, lang.asXML()));
			break;
		case nonavcreation:
			out.add(new Patch("//*[local-name()='nonAVManifestation']", Operation.ADDSUBTREE, lang.asXML()));
			break;

		default:
			break;
		}

	}

	private void processTitle(List<Patch> out) {
		final String[] arr = value.split("::-::");
		final Element title = DocumentHelper.createElement(new QName("title", EFG_NS));
		title.addElement(new QName("text", EFG_NS)).setText(arr[0]);
		title.addElement(new QName("relation", EFG_NS)).setText(arr[1]);
		title.addAttribute("lang", arr[2]);

		switch (entityType) {
		case avcreation:
			out.add(new Patch("//*[local-name()='avcreation']", Operation.ADDSUBTREE, title.asXML()));
			out.add(new Patch("//*[local-name()='avManifestation']", Operation.ADDSUBTREE, title.asXML()));
			break;
		case nonavcreation:
			out.add(new Patch("//*[local-name()='nonavcreation']", Operation.ADDSUBTREE, title.asXML()));
			out.add(new Patch("//*[local-name()='nonAVManifestation']", Operation.ADDSUBTREE, title.asXML()));
			break;
		default:
			break;
		}
	}

	private void processKeyword(List<Patch> out) {
		final String[] arr = value.split("::-::");
		final Element subject = DocumentHelper.createElement(new QName("keywords", EFG_NS));
		final Element term = subject.addElement(new QName("term", EFG_NS));
		if (!arr[2].equals("undef")) {
			subject.addAttribute("scheme", arr[2]);
		}
		if (!arr[3].equals("undef")) {
			subject.addAttribute("type", arr[3]);
		}
		if (!arr[4].equals("undef")) {
			subject.addAttribute("lang", arr[4]);
		}
		if (!arr[0].equals("undef")) {
			term.addAttribute("id", arr[0]);
		}
		term.setText(arr[1].trim());

		switch (entityType) {
		case avcreation:
			out.add(new Patch("//*[local-name()='avcreation']", Operation.ADDSUBTREE, subject.asXML()));
			break;
		case nonavcreation:
			out.add(new Patch("//*[local-name()='nonavcreation']", Operation.ADDSUBTREE, subject.asXML()));
			break;

		default:
			break;
		}

	}

	private void processField(List<Patch> out) {
		switch (operation) {
		case EDIT:
			out.add(new Patch("//*[local-name()='avcreation']", Operation.VERIFYNODE, fieldType.toString()));
			out.add(new Patch("//*[local-name()='" + fieldType.toString() + "']", Operation.EDIT, value));
			break;
		case DELETENODE:
			out.add(new Patch("//*[local-name()='format']", Operation.DELETENODE, fieldType.toString()));
			break;
		default:
			break;
		}
	}

	private void processDescription(List<Patch> out) {
		final String[] arr = value.split("::-::");
		final Element descr = DocumentHelper.createElement(new QName("description", EFG_NS));
		descr.addAttribute("type", arr[1]);
		descr.addAttribute("lang", arr[2]);
		descr.setText(arr[0]);

		switch (entityType) {
		case avcreation:
			out.add(new Patch("//*[local-name()='avcreation']", Operation.ADDSUBTREE, descr.asXML()));
			break;
		case nonavcreation:
			out.add(new Patch("//*[local-name()='nonavcreation']", Operation.ADDSUBTREE, descr.asXML()));
			break;
		default:
			break;
		}

	}

	private void processItemField(List<Patch> out) {

		switch (operation) {

		case EDIT:
			switch (entityType) {
			case avcreation:
				out.add(new Patch("//*[local-name()='avManifestation']", Operation.VERIFYNODE, "item"));
				break;
			case nonavcreation:
				out.add(new Patch("//*[local-name()='nonAVManifestation']", Operation.VERIFYNODE, "item"));
				break;
			default:
				break;
			}
			out.add(new Patch("//*[local-name()='item']", Operation.VERIFYNODE, fieldType.toString()));
			out.add(new Patch("//*[local-name()='" + fieldType.toString() + "']", Operation.EDIT, value));
			break;
		case DELETENODE:
			out.add(new Patch("//*[local-name()='item']", Operation.DELETENODE, fieldType.toString()));
			break;
		default:
			break;
		}
	}

	private void processFormatField(List<Patch> out) {
		switch (operation) {
		case EDIT:
			switch (entityType) {
			case avcreation:
				out.add(new Patch("//*[local-name()='avManifestation']", Operation.VERIFYNODE, "format"));
				break;
			case nonavcreation:
				out.add(new Patch("//*[local-name()='nonAVManifestation']", Operation.VERIFYNODE, "format"));
				break;
			default:
				break;
			}

			out.add(new Patch("//*[local-name()='format']", Operation.VERIFYNODE, fieldType.toString()));
			if (fieldType == FieldTypes.colour) {
				final String[] arr = value.split("::-::");

				out.add(new Patch("//*[local-name()='" + fieldType.toString() + "']", Operation.EDIT, arr[0]));
				out.add(new Patch("//*[local-name()='" + fieldType.toString() + "']", Operation.VERIFYATTRIBUTE, "hasColor"));
				out.add(new Patch("//*[local-name()='" + fieldType.toString() + "']/@hasColor", Operation.EDIT, arr[1]));
			} else if (fieldType == FieldTypes.sound) {
				final String[] arr = value.split("::-::");
				out.add(new Patch("//*[local-name()='" + fieldType.toString() + "']", Operation.EDIT, arr[0]));
				out.add(new Patch("//*[local-name()='" + fieldType.toString() + "']", Operation.VERIFYATTRIBUTE, "hasSound"));
				out.add(new Patch("//*[local-name()='" + fieldType.toString() + "']/@hasSound", Operation.EDIT, arr[1]));
			} else {
				out.add(new Patch("//*[local-name()='" + fieldType.toString() + "']", Operation.EDIT, value));
			}
			break;
		case DELETENODE:
			out.add(new Patch("//*[local-name()='format']", Operation.DELETENODE, fieldType.toString()));
			break;
		default:
			break;
		}
	}

	private void processAvManifestationField(List<Patch> out) {
		switch (operation) {
		case EDIT:
			switch (entityType) {
			case avcreation:
				out.add(new Patch("//*[local-name()='avManifestation']", Operation.VERIFYNODE, fieldType.toString()));
				break;
			case nonavcreation:
				out.add(new Patch("//*[local-name()='nonAVManifestation']", Operation.VERIFYNODE, fieldType.toString()));
				break;

			default:
				break;
			}
			out.add(new Patch("//*[local-name()='" + fieldType.toString() + "']", Operation.EDIT, value));
			break;
		case DELETENODE:
			switch (entityType) {
			case avcreation:
				out.add(new Patch("//*[local-name()='avManifestation']", Operation.DELETENODE, fieldType.toString()));
				break;
			case nonavcreation:
				out.add(new Patch("//*[local-name()='nonAVManifestation']", Operation.DELETENODE, fieldType.toString()));
				break;
			default:
				break;
			}
			break;

		default:
			break;
		}
	}

	private void processManifestationAttribute(String nodeName, String attributeName, List<Patch> out) {
		switch (operation) {
		case EDIT:
			switch (entityType) {
			case avcreation:
				out.add(new Patch("//*[local-name()='avManifestation']", Operation.VERIFYNODE, nodeName));
				break;
			case nonavcreation:
				out.add(new Patch("//*[local-name()='nonAVManifestation']", Operation.VERIFYNODE, nodeName));
				break;

			default:
				break;
			}
			out.add(new Patch("//*[local-name()='" + nodeName + "']", Operation.VERIFYATTRIBUTE, attributeName));
			if (fieldType == FieldTypes.rightsholderUrl) {
				final String[] arr = value.split("::-::");
				if (arr[0] != null && arr[0].length() > 0 && "undef".equals(arr[0]) == false) {
					out.add(new Patch("//*[local-name()='" + nodeName + "']", Operation.EDIT, arr[0]));
				}
				if (arr[1] != null && arr[1].length() > 0 && "undef".equals(arr[1]) == false) {
					out.add(new Patch("//*[local-name()='" + nodeName + "']/@" + attributeName, Operation.EDIT, arr[1]));
				}
			} else {
				out.add(new Patch("//*[local-name()='" + nodeName + "']/@" + attributeName, Operation.EDIT, value));
			}
			break;
		case DELETENODE:
			switch (entityType) {
			case avcreation:
				out.add(new Patch("//*[local-name()='avManifestation']", Operation.DELETENODE, fieldType.toString()));
				break;
			case nonavcreation:
				out.add(new Patch("//*[local-name()='nonAVManifestation']", Operation.DELETENODE, fieldType.toString()));
				break;
			default:
				break;
			}

			break;

		default:
			break;
		}
	}

}
