package eu.dnetlib.functionality.modular.ui.patcheditor.client;

import eu.dnetlib.clients.index.client.IndexClient;
import eu.dnetlib.clients.index.client.IndexClientException;
import eu.dnetlib.clients.index.client.ResolvingIndexClientFactory;
import eu.dnetlib.clients.index.client.response.LookupResponse;
import eu.dnetlib.enabling.locators.UniqueServiceLocator;
import eu.dnetlib.functionality.modular.ui.lightui.objects.IndexConfiguration;
import eu.dnetlib.functionality.modular.ui.patcheditor.exceptions.PatchEditorException;
import eu.dnetlib.rmi.enabling.ISLookUpException;
import eu.dnetlib.rmi.enabling.ISLookUpService;
import java.io.IOException;
import java.io.StringReader;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Node;
import org.dom4j.io.SAXReader;
import org.springframework.beans.factory.annotation.Autowired;

/**
 * Main class that handles solr client
 *
 */
public class IndexPatchEditorClient {

	private static final Log log = LogFactory.getLog(IndexPatchEditorClient.class);
	@Autowired
	private ResolvingIndexClientFactory indexClientFactory;
	private Map<String, IndexClient> indexClientMap;
	@Autowired
	private UniqueServiceLocator serviceLocator;

	/**
	 * Converts solr client parameters in the requested object
	 *
	 * @param configuration
	 *            parameters for solr client
	 * @return
	 * @throws IndexClientException
	 */
	private IndexClient getIndexClient(final IndexConfiguration configuration) throws IndexClientException {
		if (indexClientMap == null) {
			indexClientMap = new HashMap<>();
		}
		if (indexClientMap.containsKey(configuration.getBackendId())) { return indexClientMap.get(configuration.getBackendId()); }
		final IndexClient index = indexClientFactory.getClient(configuration.getFormat(), configuration.getLayout(), configuration.getInterpretation(),
				configuration.getBackendId());
		indexClientMap.put(configuration.getBackendId(), index);

		return index;
	}

	/**
	 * Retrieves repository id from RepositoryServiceResources by its prefix
	 *
	 * @param prefix
	 *            identifies repository
	 * @return
	 * @throws ISLookUpException
	 */
	private String getRepositoryId(final String prefix) throws ISLookUpException {

		final String getRepositoryIdQuery =
				"for $x in collection('/db/DRIVER/RepositoryServiceResources/RepositoryServiceResourceType') where ($x//value/text()='anf') return $x//RESOURCE_IDENTIFIER/@value/string()";

		final ISLookUpService lookUpClient = serviceLocator.getService(ISLookUpService.class);

		final List<String> repositoryIds = lookUpClient.quickSearchProfile(String.format(getRepositoryIdQuery, prefix));

		if (repositoryIds == null || repositoryIds.size() != 1) { throw new RuntimeException(
				"Unexpected number of result executing query " + String.format(getRepositoryIdQuery, repositoryIds.get(0)) + " expected: 1"); }

		return repositoryIds.get(0);
	}

	/**
	 * Retrieves index id from WorkflowDSResources by repositoryId
	 *
	 * @param repositoryId
	 * @return
	 * @throws ISLookUpException
	 */
	private String getIndexId(final String repositoryId) throws ISLookUpException {

		final String getIndexIdQuery = "for $x in collection('/db/DRIVER/WorkflowDSResources/WorkflowDSResourceType') where "
				+ "($x//DATASOURCE/@id/string()='%s') return $x//PARAM[./@name='indexId']/text()";

		final ISLookUpService lookUpClient = serviceLocator.getService(ISLookUpService.class);

		final List<String> indexIds = lookUpClient.quickSearchProfile(String.format(getIndexIdQuery, repositoryId));

		if (indexIds == null || indexIds.size() != 1) { throw new RuntimeException(
				"Unexpected number of result executing query " + String.format(getIndexIdQuery, indexIds.get(0)) + " expected: 1"); }

		return indexIds.get(0);
	}

	/**
	 * Creates an IndexConfiguration object containing configuration data
	 *
	 * @param node
	 *            xml representation of index client configuration
	 * @return
	 * @throws DocumentException
	 * @throws IOException
	 */
	private IndexConfiguration calculateIndexConfiguration(final Node node) throws DocumentException, IOException {
		final String id = node.valueOf("./BACKEND/@ID");
		final String format = node.valueOf("./METADATA_FORMAT/text()");
		final String layout = node.valueOf("./METADATA_FORMAT_LAYOUT/text()");
		final String interpretation = node.valueOf("./METADATA_FORMAT_INTERPRETATION/text()");
		final String backendId = node.valueOf("./BACKEND/text()");

		return new IndexConfiguration(id, format, layout, interpretation, backendId);
	}

	/**
	 * Retrieves index client configuration from IndexDSResources by indexId
	 *
	 * @param indexId
	 * @return
	 * @throws ISLookUpException
	 * @throws DocumentException
	 */
	private Node getConfigurationNode(final String indexId) throws ISLookUpException, DocumentException {

		final String getIndexConfigurationQuery =
				"for $x in collection('/db/DRIVER/IndexDSResources/IndexDSResourceType')  where ($x//RESOURCE_IDENTIFIER/@value/string()='%s') return $x";
		final ISLookUpService lookUpClient = serviceLocator.getService(ISLookUpService.class);
		final List<String> indexConfigurations = lookUpClient.quickSearchProfile(String.format(getIndexConfigurationQuery, indexId));
		if (indexConfigurations == null || indexConfigurations.size() != 1) { throw new RuntimeException(
				"Unexpected number of result executing query " + String.format(getIndexConfigurationQuery, indexConfigurations.get(0)) + " expected: 1"); }
		final SAXReader reader = new SAXReader();
		final Document doc = reader.read(new StringReader(indexConfigurations.get(0)));
		return doc.selectSingleNode("//CONFIGURATION");
	}

	/**
	 * Retrieves document record from solr
	 *
	 * @param recordId
	 * @return
	 * @throws PatchEditorException
	 */
	public String getDocument(final String recordId) throws PatchEditorException {

		final String record;
		try {
			final String field = "objidentifier";

			final String repositoryId = getRepositoryId(recordId.split("::")[0]);

			final String indexId = getIndexId(repositoryId);

			final Node configurationNode = getConfigurationNode(indexId);

			final IndexConfiguration idx = calculateIndexConfiguration(configurationNode);

			final String query = field + " exact \"" + recordId.trim() + "\"";
			final IndexClient indexClient = getIndexClient(idx);
			final LookupResponse lookupResponse = indexClient.lookup(query, null, 0, 1);

			final int total = (int) lookupResponse.getTotal();
			if (total != 1) {
				log.error("Invalid number of results (" + total + ") for query: " + query);
				throw new IllegalArgumentException("Invalid number of results (" + total + ") for query: " + query);
			}

			record = lookupResponse.getRecords().get(0);

		} catch (final ISLookUpException | DocumentException | IOException | IndexClientException e) {
			throw new PatchEditorException(e);
		}

		return record;
	}
}
