package eu.dnetlib.pid.resolver;

import eu.dnetlib.pid.resolver.model.ResolvedObject;
import eu.dnetlib.pid.resolver.store.ResolverStore;
import net.sf.ehcache.Cache;
import net.sf.ehcache.Element;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.http.HttpEntity;
import org.apache.http.client.ResponseHandler;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.util.EntityUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Required;

import java.io.IOException;

/**
 * Created by sandro on 9/8/16.
 */
public abstract class AbstractPIDResolver implements PIDResolver {

	private static final Log log = LogFactory.getLog(AbstractPIDResolver.class);
	private Cache cache;
	protected String name;
	protected int order;

	protected boolean availableOffline;

	@Autowired
	private ResolverStore storeManager;

    private static ResponseHandler<String> responseHandler = response -> {
		int status = response.getStatusLine().getStatusCode();
		if (status >= 200 && status < 300) {
			HttpEntity entity = response.getEntity();
			return entity != null ? EntityUtils.toString(entity) : null;
		} else {
			log.debug("Unexpected response status: " + status);
			return null;
		}
	};

	public ResolvedObject retrievePIDfromCache(final String pid, final String pidType) {
		final String dnetIdentifier = storeManager.generateDNetIdentifier(pid, pidType);
		final Element item = cache.get(dnetIdentifier);

		if (item != null) return (ResolvedObject) item.getObjectValue();

		return storeManager.getRecord(dnetIdentifier);
	}

	public ResolvedObject retrievePID(final String pid, final String pidType, boolean offline) {
		if (offline && !isAvailableOffline())
			return null;
		if (!canResolvePid(pidType)) return null;
		final String dnetIdentifier = storeManager.generateDNetIdentifier(pid, pidType);
		final Element item = cache.get(dnetIdentifier);

		if (item != null) return (ResolvedObject) item.getObjectValue();

		ResolvedObject resolvedObject = storeManager.getRecord(dnetIdentifier);

		if (resolvedObject != null) {
			cache.put(new Element(storeManager.generateDNetIdentifier(pid, pidType), resolvedObject));
			return resolvedObject;
		}

		log.debug("Cache miss for pid:" + pid + "  and type :" + pidType);
		resolvedObject = resolve(pid, pidType);
		if (resolvedObject != null) {
			storeManager.insertRecord(getName(), resolvedObject);
			cache.put(new Element(storeManager.generateDNetIdentifier(pid, pidType), resolvedObject));
			return resolvedObject;
		}
		return null;
	}

	public String retrieveDnetID(final String pid, final String pidType, boolean offline) {
		if (offline && !isAvailableOffline())
			return null;
		if (!canResolvePid(pidType)) return null;
		final String dnetIdentifier = storeManager.generateDNetIdentifier(pid, pidType);
		final Element item = cache.get(dnetIdentifier);

		if (item != null) return dnetIdentifier;

        String resolvedIdentifier = storeManager.getRecordIdentifier(dnetIdentifier);

		if (resolvedIdentifier != null && !StringUtils.isBlank(resolvedIdentifier))
			return resolvedIdentifier;

		ResolvedObject resolvedObject = resolve(pid, pidType);
		if (resolvedObject != null) {
			storeManager.insertRecord(getName(), resolvedObject);
			cache.put(new Element(dnetIdentifier, resolvedObject));
			return dnetIdentifier;
		}

		return null;
	}

	public String getName() {
		return this.getClass().getSimpleName();
	}

	public void setName(final String name) {
		this.name = name;
	}

	protected abstract boolean canResolvePid(final String pidType);

	protected abstract ResolvedObject resolve(final String pid, final String pidType);

	public Cache getCache() {
		return cache;
	}

	@Required
	public void setCache(final Cache cache) {
		this.cache = cache;
	}


	public static String requestURL(final String url, final int timeout, int numberOfRetry) {
        RequestConfig config = RequestConfig.custom()
                .setConnectTimeout(timeout * 1000)
                .setConnectionRequestTimeout(timeout * 1000)
                .setSocketTimeout(timeout * 1000).build();

        log.debug("Timeout: "+timeout);
        final CloseableHttpClient httpclient = HttpClientBuilder.create().setDefaultRequestConfig(config).build();
        try {
            return getRequest(url, httpclient, numberOfRetry);
        } catch (Throwable e) {
            log.debug("Error in getting url " + url, e);
            return null;
        } finally {
            try {
                httpclient.close();
            } catch (IOException e) {
                log.error("Error on closing httpclient", e);
            }
        }
    }



	public static String requestURL(final String url) {
        return requestURL(url,4,1);
	}

    private static String getRequest(String url, CloseableHttpClient httpclient, int numberOfRetry)  {
        try {
            log.debug("requesting URL: "+url);
            HttpGet httpGet = new HttpGet(url);
            return httpclient.execute(httpGet, responseHandler);
        } catch (Exception e) {
            log.debug("Error in getting url " + url, e);
            if (numberOfRetry > 0) {
                log.debug("Try again");
                return getRequest(url, httpclient, --numberOfRetry);
            }
            return null;
        }
    }


    @Override
	public int getOrder() {
		return this.order;
	}

	@Required
	public AbstractPIDResolver setOrder(int order) {
		this.order = order;
		return this;
	}


	@Override
	public int compareTo(PIDResolver o) {
		return Integer.compare(getOrder(), o.getOrder());
	}


	public boolean isAvailableOffline() {
		return availableOffline;
	}

	@Required
	public AbstractPIDResolver setAvailableOffline(boolean availableOffline) {
		this.availableOffline = availableOffline;
		return this;
	}
}
