package eu.dnetlib.pid.resolver.mdstore.plugin;

import com.mongodb.DBObject;
import com.mongodb.client.MongoCollection;
import eu.dnetlib.pid.resolver.PIDResolver;
import eu.dnetlib.pid.resolver.model.ObjectRelation;
import eu.dnetlib.pid.resolver.model.ObjectType;
import eu.dnetlib.pid.resolver.model.PID;
import eu.dnetlib.pid.resolver.model.ResolvedObject;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.util.List;
import java.util.Map;
import java.util.concurrent.BlockingQueue;
import java.util.function.Function;

public abstract class AbstractRecordResolver implements RecordResolver {

    protected static final Log log = LogFactory.getLog(AbstractRecordResolver.class);
    protected final Long timestamp;
    protected List<PIDResolver> pluginResolver;
    protected BlockingQueue<DBObject> inputQueue;
    protected MongoCollection<DBObject> outputCollection;
    protected ResolverSerializer serializer;
    protected boolean offline;
    protected boolean forceResolver = false;

    public AbstractRecordResolver(final long ts) {
        this.timestamp = ts;
    }

    protected String resolveRelations(String inputRecord, ResolvedObject inputObject, boolean shouldUpdate, Function<String,String> getInverseRelation) {
        if (inputObject.getRelations() != null) {
            for (ObjectRelation rel : inputObject.getRelations()) {
                final Map<String, ObjectType> resolvedRelation = tryToResolveRelation(rel.getTargetPID());
                if (resolvedRelation != null && !resolvedRelation.isEmpty()) {
                    resolvedRelation.entrySet()
                            .forEach(e -> {
                                rel.setTargetPID(new PID(e.getKey(), "dnet"));
                                rel.setTargetType(e.getValue());
                                rel.setInverseRelation(getInverseRelation.apply(rel.getRelationSemantics()));

                            });
                    shouldUpdate = true;
                }
            }
            if (shouldUpdate) {
                final String newXML = serializer.serializeReplacingXML(inputRecord, inputObject);
                return newXML;
            }
        }
        return null;
    }

    @Override
    public Boolean call() throws Exception {

        log.info("START HERE! Ressolving offline:"+offline);

        DBObject currentObject = inputQueue.take();
        int i = 0;
        String currentRecord = null;
        double sumTotal = 0;
        while (currentObject != ResolverMDStorePlugin.DONE) {
            try {
                currentRecord = (String) currentObject.get("body");
                if (forceResolver || currentObject.get("resolved_ts") == null) {
                    final double start = System.currentTimeMillis();
                    final String resolvedRecord = resolve(currentRecord);
                    if (resolvedRecord != null) {
                        currentObject.put("body", resolvedRecord);
                        currentObject.removeField("_id");
                        currentObject.put("resolved_ts", timestamp);
                        outputCollection.insertOne(currentObject);
                    }
                    final double total = System.currentTimeMillis() - start;
                    sumTotal += total;

                }

                currentObject = inputQueue.take();
                if (i++ % 100 == 0) {
                    log.debug(Thread.currentThread().getId() + " total object resolved: " + i + "in average time " + (sumTotal / 100) + "ms");
                    sumTotal = 0;
                }
            } catch (Throwable e) {
                log.error("Error on resolving objects " + currentRecord, e);
                return false;
            }
        }
        if (currentObject == ResolverMDStorePlugin.DONE) {
            inputQueue.put(currentObject);
        }
        return true;

    }

    protected abstract Map<String, ObjectType> tryToResolveRelation(PID targetPID);

    public void setOffline(boolean offline) {
        this.offline = offline;
    }
}
