package eu.dnetlib.pid.resolver.mdstore.plugin;

import com.google.common.collect.Lists;
import com.mongodb.BasicDBObject;
import com.mongodb.BasicDBObjectBuilder;
import com.mongodb.DBObject;
import com.mongodb.client.FindIterable;
import com.mongodb.client.MongoCollection;
import com.mongodb.client.MongoDatabase;
import com.mongodb.client.model.*;
import eu.dnetlib.data.mdstore.modular.action.DoneCallback;
import eu.dnetlib.data.mdstore.modular.action.FailedCallback;
import eu.dnetlib.data.mdstore.modular.action.MDStorePlugin;
import eu.dnetlib.data.mdstore.modular.connector.MDStoreDao;
import eu.dnetlib.data.mdstore.modular.mongodb.MDStoreTransactionManagerImpl;
import eu.dnetlib.pid.resolver.PIDResolver;
import eu.dnetlib.rmi.data.MDStoreServiceException;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.bson.Document;
import org.bson.conversions.Bson;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.*;
import java.util.concurrent.*;

import static com.mongodb.client.model.Filters.*;
import static com.mongodb.client.model.Filters.gt;
import static com.mongodb.client.model.Filters.lt;

/**
 * Created by sandro on 9/9/16.
 */
public class ResolverMDStorePlugin implements MDStorePlugin {

	private static final Log log = LogFactory.getLog(ResolverMDStorePlugin.class);
	public static DBObject DONE = new BasicDBObject();



	@Autowired
	private List<PIDResolver> pluginResolver;


	@Autowired
	private ResolverSerializer resolverSerializer;

	@Autowired
	private RecordResolverFactory recordResolverFactory;


	@Autowired
	private MDStoreTransactionManagerImpl transactionManager;

	public static void save(MongoCollection<DBObject> collection, DBObject document) {
		Object id = document.get("_id");
		if (id == null) {
			collection.insertOne(document);
		} else {
			collection.replaceOne(eq("_id", id), document, new UpdateOptions().upsert(true));
		}
	}

	@Override
	public void run(final MDStoreDao dao, final Map<String, String> params, final DoneCallback doneCallback, final FailedCallback failedCallback) {
		try {
			final String id = params.get("mdStoreId");
			final boolean offline = "true".equals(params.get("offline"));

            int numberOfThreads = 4;

            final String numOfThreadsString = params.get("numberOfThreads");

			final String resolvingMode = params.get("collectionMode");

			final String lastResolveDate = params.get("lastResolveDate");

            try {
                if (!StringUtils.isEmpty(numOfThreadsString)) {
                    numberOfThreads = Integer.parseInt(numOfThreadsString);
                }

            } catch (Throwable e) {
                log.error("Number of threads Param is not an int value it will apply by default 4");
            }

			final boolean refresh = "refresh".equalsIgnoreCase(resolvingMode);

			final String internalId = transactionManager.readMdStore(id);

			final MongoDatabase db = transactionManager.getDb();

			final MongoCollection<DBObject> currentMdStoreCollection = db.getCollection(internalId, DBObject.class);

			final MongoCollection<DBObject> resolvedRecord = db.getCollection("resolved_" + StringUtils.substringBefore(id, "_"), DBObject.class);

            final BasicDBObject idx = new BasicDBObject();
            idx.put("resolved_ts", 1);
            resolvedRecord.createIndex(idx);

//			if (refresh) {
//				resolvedRecord.drop();
//			}

            if (!"INCREMENTAL".equalsIgnoreCase(resolvingMode))
            	upsertResolved(currentMdStoreCollection, resolvedRecord, 0);


			final FindIterable<DBObject> mdstoreRecords = "refresh".equalsIgnoreCase(resolvingMode)?currentMdStoreCollection.find(): currentMdStoreCollection.find(dateQuery(lastResolveDate==null?0:Long.parseLong(lastResolveDate),null));

            mdstoreRecords.noCursorTimeout(true);

			final BlockingQueue<DBObject> queue = new ArrayBlockingQueue<>(100);

			final List<Future<Boolean>> responses = new ArrayList<>();

			final ExecutorService executor = Executors.newFixedThreadPool(100);

			final long total = "refresh".equalsIgnoreCase(resolvingMode)?currentMdStoreCollection.count():currentMdStoreCollection.count(dateQuery(lastResolveDate==null?0:Long.parseLong(lastResolveDate),null));

			int previousPrintValue = -1;
            int currentPerc;

            final long ts = System.currentTimeMillis();


			Collections.sort(pluginResolver);

			for (int i = 0; i < numberOfThreads; i++) {
				final RecordResolver resolver = recordResolverFactory.createResolver(ts, queue, resolvedRecord, resolverSerializer, pluginResolver, offline, false);
				responses.add(executor.submit(resolver));
			}

			int parsed = 0;

			for (DBObject currentMdStoreRecord : mdstoreRecords) {
				queue.put(currentMdStoreRecord);

				currentPerc = Math.round(((float) ++parsed / (float) total) * 100.0F);

				if (currentPerc != previousPrintValue) {
					log.info("Resolving process " + currentPerc + " %");
					previousPrintValue = currentPerc;
				}
			}
			queue.put(DONE);

			for (Future<Boolean> response : responses) {
				response.get();
			}
            upsertResolved(currentMdStoreCollection, resolvedRecord, ts - 1);
            doneCallback.call(params);
		} catch (Throwable e) {
            log.error(e);
            throw new RuntimeException("Error on resolving records ", e);
		}
	}


	private Bson dateQuery(final Long from, final Long until) {
		if (from != null & until != null) {
			return and(gt("timestamp", from), lt("timestamp", until));
		}
		if (from != null) {
			return gt("timestamp", from);
		}
		if (until != null) {
			return lt("timestamp", until);
		}
		return null;
	}


	private void upsertResolved(MongoCollection<DBObject> currentMdStoreCollection, MongoCollection<DBObject> resolvedRecord, final long timestamp) {
        log.info("Updating resolved objects");

        final Bson queryByTs = Filters.gte("resolved_ts", Long.valueOf(timestamp));
        int i = 0;
        final FindIterable<DBObject> dbObjects = timestamp == 0 ? resolvedRecord.find() : resolvedRecord.find(queryByTs);
		final UpdateOptions f = new UpdateOptions().upsert(true);
		final List<WriteModel<DBObject>> upsertList = new ArrayList<>();
		final BulkWriteOptions writeOptions = new BulkWriteOptions().ordered(false);
		final int bulkSize = 1000;
		long validOpCounter  =0;
		for (DBObject object : dbObjects) {
			if (StringUtils.isNotBlank(object.get("id").toString())) {
				final DBObject replacedObj = BasicDBObjectBuilder.start()
						.add("body", object.get("body").toString())
						.add("id", object.get("id").toString())
						.add("resolved_ts", object.get("resolved_ts"))
						.get();

				upsertList.add(new ReplaceOneModel(new BasicDBObject("id",  object.get("id").toString()), replacedObj, f));
				validOpCounter++;
				if (((validOpCounter % bulkSize) == 0) && (validOpCounter != 0)) {
					currentMdStoreCollection.bulkWrite(upsertList, writeOptions);
					upsertList.clear();
					log.info("Transaction commit: Upserting: "+validOpCounter);
				}
			}
		}
		if (upsertList.size()>0){
			currentMdStoreCollection.bulkWrite(upsertList, writeOptions);
		}
//
//
//
//        for (DBObject object : dbObjects) {
//			Bson query = Filters.eq("id", object.get("id").toString());
//			final DBObject replacedObj = BasicDBObjectBuilder.start()
//					.add("body", object.get("body").toString())
//                    .add("resolved_ts", object.get("resolved_ts"))
//                    .get();
//			Bson newDocument = new Document("$set", replacedObj);
//			currentMdStoreCollection.findOneAndUpdate(query, newDocument);
//            i++;
//
//		}

        log.info("Updated " + i);
    }


}
