package eu.dnetlib.wds.collector.plugins;

import java.net.URL;
import java.util.ArrayList;
import java.util.List;

import com.google.gson.*;
import eu.dnetlib.rmi.data.CollectorServiceException;
import eu.dnetlib.rmi.data.InterfaceDescriptor;
import eu.dnetlib.rmi.data.plugin.AbstractCollectorPlugin;
import eu.dnetlib.wds.utils.WDSUtils;
import org.antlr.stringtemplate.StringTemplate;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringEscapeUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * Created by sandro on 7/1/16.
 */
public class CMRDatasourcePlugin extends AbstractCollectorPlugin {

    private static final String BASE_URL = "https://cmr.earthdata.nasa.gov/search/keywords/data_centers?pretty=true";

    private static final Log log = LogFactory.getLog(CMRDatasourcePlugin.class);

    @Override
    public Iterable<String> collect(final InterfaceDescriptor interfaceDescriptor, final String fromDate, final String untilDate)
            throws CollectorServiceException {

        return generateListOfDatasource();

    }


    private void manageLevel(final JsonObject level, final List<String> result, final Gson g) {

        if ((level == null) || level.get("subfields") == null)
            return;
        final JsonArray subfields = level.get("subfields").getAsJsonArray();

        subfields.forEach(field -> {
            if ("short_name".equals(field.getAsString())) {
                level.get(field.getAsString()).getAsJsonArray()
                        .forEach(it -> manageShortName(it.getAsJsonObject(), result, g));
            } else {
                level.get(field.getAsString()).getAsJsonArray()
                        .forEach(it -> manageLevel(it.getAsJsonObject(), result, g));

            }
        });


    }


    private String getElement(final JsonObject element, final String key) {

        if (element.get(key) != null)
            return element.get(key).getAsString();
        return null;

    }

    private void manageShortName(final JsonObject item, final List<String> result, final Gson g) {
        if (item != null && item.get("subfields") == null) {
            final String value = getElement(item, "value");
            final String uuid = getElement(item, "uuid");

            addDatasource(result, g, uuid, uuid, value, null);
            return;
        }
        final JsonArray subfields = item.get("subfields").getAsJsonArray();
        subfields.forEach(field -> {
            if ("short_name".equals(field.getAsString())) {
                item.get(field.getAsString()).getAsJsonArray()
                        .forEach(it -> manageShortName(it.getAsJsonObject(), result, g));
            }
            if ("long_name".equals(field.getAsString())) {
                manageLongName(g, result, item);
            }
            if ("url".equals(field.getAsString())) {
                manageURL(item, result, g);

            }


        });
    }

    private void manageURL(final JsonObject item, final List<String> result, final Gson g) {
        final String id = getElement(item, "value");
        final JsonObject firstItem = item.get("url").getAsJsonArray().get(0).getAsJsonObject();
        final String url = getElement(firstItem, "value");
        final String uuid = getElement(firstItem, "uuid");
        addDatasource(result, g, id, uuid, id, url);
    }

    private void addDatasource(final List<String> result, final Gson g, final String id, final String uuid, final String value, final String url) {
        final GenericDatasource currentDatasource = new GenericDatasource();
        try {
            currentDatasource.setId(uuid);
            currentDatasource.setName(value);
            currentDatasource.getExtrafields().put("archive_center", StringEscapeUtils.escapeXml11(id));
            currentDatasource.setNamespacePrefix(WDSUtils.generateWDSNsPrefix(uuid, "cmr_", 12));
            if (StringUtils.isNotEmpty(url)) {
                currentDatasource.setUrl(url);
            }
            result.add(g.toJson(currentDatasource));
        } catch (Throwable e1) {
            log.error("Error on creating list of CMR Datasources", e1);
        }
    }

    private void manageLongName(final Gson g, List<String> results, JsonObject item) {
        final JsonObject currentItem = item.getAsJsonObject();
        final String id = currentItem.get("value").getAsString();
        final JsonElement jsonInfo = currentItem.get("long_name");
        if (jsonInfo != null) {
            String name;

            JsonObject infoObject;
            if (jsonInfo.isJsonArray()) {
                infoObject = jsonInfo.getAsJsonArray().get(0).getAsJsonObject();
            } else infoObject = jsonInfo.getAsJsonObject();

            name = infoObject.get("value").getAsString();
            final JsonElement urlObject = infoObject.get("url");

            String url = "";
            String uuid = "";
            if (urlObject != null) {
                if (urlObject.isJsonArray()) {
                    url = urlObject.getAsJsonArray().get(0).getAsJsonObject().get("value").getAsString();
                    uuid = urlObject.getAsJsonArray().get(0).getAsJsonObject().get("uuid").getAsString();
                } else {
                    url = urlObject.getAsJsonObject().get("value").getAsString();
                    uuid = urlObject.getAsJsonObject().get("uuid").getAsString();
                }
            }
            addDatasource(results, g, id, uuid, name, url);

        }
    }


    public List<String> generateListOfDatasource() throws CollectorServiceException {
        try {

            URL myUrl = new URL(BASE_URL);
            final Gson g = new Gson();

            JsonElement element = new JsonParser().parse(IOUtils.toString(myUrl.openStream()));

            final JsonObject root = element.getAsJsonObject();

            final JsonArray level_0 = root.get("level_0").getAsJsonArray();

            List<String> results = new ArrayList<>();

            for (JsonElement levelItem : level_0) {

                manageLevel(levelItem.getAsJsonObject(), results, new Gson());


//                final JsonElement short_name = levelItem.getAsJsonObject().get("short_name");
//
//                if (short_name != null) {
//                    final JsonArray level_1 = short_name.getAsJsonArray();
//
//                    for (JsonElement item : level_1) {
//
//                        String subfields = "";
//                        JsonElement subfield = item.getAsJsonObject().get("subfields");
//                        if (subfield!= null) {
//                            subfields =  subfield.getAsJsonArray().get(0).getAsString();
//                        }
//
//
//                        if (subfields.equals("long_name"))
//                            manageLongName(g, results, item);
//                        if (subfields.equals("short_name"))
//                        {
//                            JsonArray shortName = item.getAsJsonObject().get("short_name").getAsJsonArray();
//                            shortName.forEach(it -> manageLongName(g, results, it));
//                        }
//                    }
//                }
            }
            return results;
        } catch (Throwable e) {
            throw new CollectorServiceException("Error on iterating CMR data centers", e);
        }
    }


}
