package gr.cite.regional.data.collection.application.controllers;

import gr.cite.regional.data.collection.application.core.EntityDtoMapper;
import gr.cite.regional.data.collection.application.dtos.DomainDto;
import gr.cite.regional.data.collection.application.dtos.UserReferenceDto;
import gr.cite.regional.data.collection.application.endpoint.ServiceDiscoveryException;
import gr.cite.regional.data.collection.application.endpoint.social.SocialNetworkingService;
import gr.cite.regional.data.collection.application.endpoint.social.UserProfile;
import gr.cite.regional.data.collection.dataaccess.entities.Domain;
import gr.cite.regional.data.collection.dataaccess.entities.UserReference;
import gr.cite.regional.data.collection.dataaccess.exceptions.ServiceException;
import gr.cite.regional.data.collection.dataaccess.services.DomainService;
import gr.cite.regional.data.collection.dataaccess.services.UserReferenceService;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.CrossOrigin;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;

import java.util.List;
import java.util.NoSuchElementException;

@Controller
@CrossOrigin
@RequestMapping("/" + UserReferenceController.USER_REFERENCE_ENDPOINT)
public class UserReferenceController extends BaseController {
	private static final Logger logger = LogManager.getLogger(UserReferenceController.class);
	static final String USER_REFERENCE_ENDPOINT = "userReferences";
	private static final String VRE_MANAGER_ROLE = "VRE-Manager";

	private String hostname;
	private EntityDtoMapper entityDtoMapper;
	
	private UserReferenceService userReferenceService;
	private DomainService domainService;
	private SocialNetworkingService socialNetworkingService;

	@Autowired
	public UserReferenceController(UserReferenceService userReferenceService, DomainService domainService, SocialNetworkingService socialNetworkingService, String hostname, EntityDtoMapper entityDtoMapper) {
		this.hostname = hostname;
		this.entityDtoMapper = entityDtoMapper;

		this.userReferenceService = userReferenceService;
		this.domainService = domainService;
		this.socialNetworkingService = socialNetworkingService;
	}

	@RequestMapping(value = "", method = RequestMethod.GET, produces = MediaType.APPLICATION_JSON_VALUE)
	public ResponseEntity<List<UserReferenceDto>> getAllUserReferences() throws ServiceException {
		List<UserReference> userReferences = this.userReferenceService.getAllUserReferences();
		return ResponseEntity.ok(this.entityDtoMapper.entitiesToDtos(userReferences, UserReferenceDto.class));
	}
	
	@RequestMapping(value = "", method = RequestMethod.POST, produces = MediaType.APPLICATION_JSON_VALUE)
	public ResponseEntity<String> addUserReference(@RequestBody UserReferenceDto userReferenceDto) throws ServiceException {
		UserReference userReferenceEntity = this.userReferenceService.addUserReference(this.entityDtoMapper.dtoToEntity(userReferenceDto, UserReference.class));
		
		return ResponseEntity.ok("User Reference " + userReferenceEntity.getLabel() + " [" + userReferenceEntity.getId() + "] successfully created");
		/*URI location = UriComponentsBuilder.fromHttpUrl(this.hostname).pathSegment(UserReferenceController.USER_REFERENCE_ENDPOINT, "{id}")
				.buildAndExpand(userReferenceEntity.getId()).toUri();
		return ResponseEntity.created(location).body("User Reference " + userReferenceEntity.getLabel() + " [" + userReferenceEntity.getId() + "] successfully created");*/
	}
	
	@RequestMapping(value = "/login", method = RequestMethod.POST, produces = MediaType.APPLICATION_JSON_VALUE)
	public ResponseEntity<UserReferenceDto> login(@RequestBody UserReferenceDto userReferenceDto) throws ServiceException {
		String scope = this.getGCubeScope();
		
		try {
			this.domainService.getDomainByLabel(scope);
		} catch (NoSuchElementException e) {
			throw new ServiceException(scope + "does not exist");
		}
		
		UserReference userReferenceEntity = this.userReferenceService.createUserReferenceIfNotExists(this.entityDtoMapper.dtoToEntity(userReferenceDto, UserReference.class));
		return ResponseEntity.ok(this.entityDtoMapper.entityToDto(userReferenceEntity, UserReferenceDto.class));
	}
	
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, produces = MediaType.APPLICATION_JSON_VALUE)
	public ResponseEntity<String> updateUserReference(@PathVariable("id") Integer id, @RequestBody UserReferenceDto userReferenceDto) throws ServiceException {
		userReferenceDto.setId(id);
		UserReference userReferenceEntity = this.userReferenceService.updateUserReference(this.entityDtoMapper.dtoToEntity(userReferenceDto, UserReference.class));
		
		return ResponseEntity.ok("User Reference " + userReferenceEntity.getLabel() + " [" + userReferenceEntity.getId() + "] successfully updated");
	}
	
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, produces = MediaType.APPLICATION_JSON_VALUE)
	public ResponseEntity<UserReferenceDto> getUserReference(@PathVariable("id") Integer id) {
		UserReference userReference = this.userReferenceService.getUserReference(id);
		return ResponseEntity.ok(this.entityDtoMapper.entityToDto(userReference, UserReferenceDto.class));
	}
	
	@RequestMapping(value = "/admin/register", method = RequestMethod.POST, consumes = MediaType.APPLICATION_JSON_VALUE, produces = MediaType.APPLICATION_JSON_VALUE)
	public ResponseEntity<UserReferenceDto> registerAdminUser() throws ServiceException {
		String scope = this.getGCubeScope();
		String token = this.getGCubeToken();

		//if (user == null || user.getDomain() == null) throw new IllegalArgumentException("User and Domain must be defined");
		UserProfile userProfile;
		try {
			userProfile = this.socialNetworkingService.getUserProfile(token, scope);
		} catch (ServiceDiscoveryException e) {
			logger.error(e.getMessage(), e);
			throw new ServiceException("Failed at retrieving user profile");
		}
		if (userProfile == null) throw new IllegalArgumentException("No user with token [" + token + "]");

		Domain domain = this.domainService.getOrCreateDomainByLabel(scope);
		UserReferenceDto user = getOrCreateUserReference(userProfile, domain);

		return ResponseEntity.ok(user);
	}

	@RequestMapping(value = "/register", method = RequestMethod.POST, consumes = MediaType.APPLICATION_JSON_VALUE, produces = MediaType.APPLICATION_JSON_VALUE)
	public ResponseEntity<UserReferenceDto> registerUser() throws ServiceException {
		String scope = this.getGCubeScope();
		String token = this.getGCubeToken();

		UserProfile userProfile = null;
		try {
			userProfile = this.socialNetworkingService.getUserProfile(token, scope);
		} catch (ServiceDiscoveryException e) {
			logger.error(e.getMessage(), e);
			throw new ServiceException("Failed at retrieving user profile");
		}
		if (userProfile == null) throw new IllegalArgumentException("No user with token [" + token + "]");

		Domain domain = this.domainService.getDomainByLabel(scope);
		UserReferenceDto user = getOrCreateUserReference(userProfile, domain);

		return ResponseEntity.ok(user);
	}

	private UserReferenceDto getOrCreateUserReference(UserProfile userProfile, Domain domain) throws ServiceException {
		UserReference userReference = mapUserProfileToUserReference(userProfile);
		userReference = this.userReferenceService.getOrCreateUserReferenceByLabel(userReference);

		UserReferenceDto user = this.entityDtoMapper.entityToDto(userReference, UserReferenceDto.class);
		user.setDomain(this.entityDtoMapper.entityToDto(domain, DomainDto.class));

		return user;
	}

	private UserReference mapUserProfileToUserReference(UserProfile userProfile) {
		UserReference userReference = null;

		if (userProfile != null) {
			userReference = new UserReference();
			userReference.setLabel(userProfile.getUsername());
			userReference.setEmail(userProfile.getEmail());
			userReference.setFullName(userProfile.getFullname());
			userReference.setUri(userProfile.getUri());
		}
		
		return userReference;
	}
	
}
