package gr.cite.regional.data.collection.application.controllers;

import gr.cite.regional.data.collection.application.core.EntityDtoMapper;
import gr.cite.regional.data.collection.application.dtos.*;
import gr.cite.regional.data.collection.application.endpoint.ServiceDiscoveryException;
import gr.cite.regional.data.collection.application.endpoint.social.SocialNetworkingService;
import gr.cite.regional.data.collection.application.endpoint.social.UserProfile;
import gr.cite.regional.data.collection.application.services.DataSubmissionHandler;
import gr.cite.regional.data.collection.dataaccess.dsd.DsdProcessing;
import gr.cite.regional.data.collection.dataaccess.dsd.Field;
import gr.cite.regional.data.collection.dataaccess.entities.Cdt;
import gr.cite.regional.data.collection.dataaccess.entities.DataSubmission;
import gr.cite.regional.data.collection.dataaccess.entities.Domain;
import gr.cite.regional.data.collection.dataaccess.entities.UserReference;
import gr.cite.regional.data.collection.dataaccess.exceptions.ServiceException;
import gr.cite.regional.data.collection.dataaccess.services.DataCollectionService;
import gr.cite.regional.data.collection.dataaccess.services.DataSubmissionService;
import gr.cite.regional.data.collection.dataaccess.services.DomainService;
import gr.cite.regional.data.collection.dataaccess.services.UserReferenceService;
import gr.cite.regional.data.collection.dataaccess.types.DataSubmissionStatusType;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.CrossOrigin;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.xml.bind.JAXBException;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileWriter;
import java.io.IOException;
import java.util.Comparator;
import java.util.List;
import java.util.stream.Collectors;

@Controller
@CrossOrigin
@RequestMapping("/" + DataSubmissionController.DATA_SUBMISSION_ENDPOINT)
public class DataSubmissionController extends BaseController {
	static final String DATA_SUBMISSION_ENDPOINT = "dataSubmissions";
	private static final String DATA_SUBMISSION_AS_CSV_FILE_ENDPOINT = "{id}/data/file";
	private static final Logger logger = LogManager.getLogger(DataSubmissionController.class);
	
	private String hostname;
	private EntityDtoMapper entityDtoMapper;
	
	private DsdProcessing dsdProcessing;
	private DataSubmissionService dataSubmissionService;
	private DataCollectionService dataCollectionService;
	private DomainService domainService;
	private UserReferenceService userReferenceService;
	private SocialNetworkingService socialNetworkingService;
	private DataSubmissionHandler dataSubmissionHandler;
	
	@Autowired
	public DataSubmissionController(DataSubmissionService dataSubmissionService, UserReferenceService userReferenceService, DomainService domainService,
									DataCollectionService dataCollectionService, DsdProcessing dsdProcessing, DataSubmissionHandler dataSubmissionHandler,
									String hostname, EntityDtoMapper entityDtoMapper, SocialNetworkingService socialNetworkingService) {
		this.hostname = hostname;
		this.entityDtoMapper = entityDtoMapper;
		
		this.dataSubmissionService = dataSubmissionService;
		this.dataCollectionService = dataCollectionService;
		this.userReferenceService = userReferenceService;
		this.domainService = domainService;
		this.dsdProcessing = dsdProcessing;
		this.socialNetworkingService = socialNetworkingService;
		this.dataSubmissionHandler = dataSubmissionHandler;
	}
	
	@RequestMapping(value = "", method = RequestMethod.POST, consumes = MediaType.APPLICATION_JSON_VALUE, produces = MediaType.APPLICATION_JSON_VALUE)
	public ResponseEntity<DataSubmissionDto> addDataSubmission(@RequestBody DataSubmissionDto dataSubmissionDto, HttpServletRequest request) throws ServiceException {
		String token = request.getHeader("gcube-token");
		AttributesDto attributesDto = dataSubmissionDto.getAttributes();

		DataSubmission dataSubmission = this.dataSubmissionHandler.addDataSubmissionAndNotify(token, attributesDto, this.entityDtoMapper.dtoToEntity(dataSubmissionDto, DataSubmission.class));

		DataSubmissionDto newDataSubmission = new DataSubmissionDto();
		newDataSubmission.setId(dataSubmission.getId());

		return ResponseEntity.ok(newDataSubmission);
		/*URI location = UriComponentsBuilder.fromHttpUrl(this.hostname).pathSegment(DataSubmissionController.DATA_SUBMISSION_ENDPOINT, "{id}")
				.buildAndExpand(dataSubmissionEntity.getId()).toUri();
		return ResponseEntity.created(location).body("Data Submission [" + dataSubmissionEntity.getId() + "] successfully inserted");*/
	}
	
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, produces = MediaType.TEXT_PLAIN_VALUE)
	public ResponseEntity<String> updateDataSubmission(@PathVariable("id") Integer id, @RequestBody DataSubmissionDto dataSubmissionDto, HttpServletRequest request) throws ServiceException {
		String token = request.getHeader("gcube-token");

		dataSubmissionDto.setId(id);

		DataSubmission dataSubmissionEntity = this.entityDtoMapper.dtoToEntity(dataSubmissionDto, DataSubmission.class);
		AttributesDto attributesDto = dataSubmissionDto.getAttributes();
		try {
			this.dataSubmissionHandler.updateDataSubmission(token, attributesDto, dataSubmissionEntity);
		}catch (IllegalStateException ex){
			return ResponseEntity.status(HttpStatus.LOCKED).body(ex.getMessage());
		}
		return ResponseEntity.ok(id.toString());
		/*URI location = UriComponentsBuilder.fromHttpUrl(this.hostname).pathSegment(DataSubmissionController.DATA_SUBMISSION_ENDPOINT, "{id}")
				.buildAndExpand(dataSubmissionEntity.getId()).toUri();
		return ResponseEntity.created(location).body("Data Submission [" + dataSubmissionEntity.getId() + "] successfully replaced");*/
	}
	


	
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, produces = MediaType.APPLICATION_JSON_VALUE)
	public ResponseEntity<DataSubmissionDto> getDataSubmission(@PathVariable("id") Integer id) throws ServiceException {
		DataSubmission dataSubmission = this.dataSubmissionService.getDataSubmission(id);
		return ResponseEntity.ok(this.entityDtoMapper.entityToDto(dataSubmission, DataSubmissionDto.class));
	}
	
	@RequestMapping(value = "", method = RequestMethod.GET, produces = MediaType.APPLICATION_JSON_VALUE)
	public ResponseEntity<List<DataSubmissionDto>> getDataSubmissions(@RequestParam(name = "dataCollection", required = false) Integer dataCollectionId,
			  @RequestParam(name = "owner", required = false) Integer ownerId) throws ServiceException {
		
		List<DataSubmission> dataSubmissions;
		if (dataCollectionId == null) {
			dataSubmissions = this.dataSubmissionService.getAllDataSubmissions();
		} else if (ownerId == null) {
			dataSubmissions = this.dataSubmissionService.getDataSubmissionsByDataCollectionId(dataCollectionId);
		} else {
			dataSubmissions = this.dataSubmissionService.getDataSubmissionsByDataCollectionAndOwner(dataCollectionId, ownerId);
		}
		dataSubmissions.forEach(dataSubmission -> dataSubmission.getDataCollection().setDataModel(null));
		
		return ResponseEntity.ok(this.entityDtoMapper.entitiesToDtos(dataSubmissions, DataSubmissionDto.class));
	}
	
	@RequestMapping(value = "/addin", method = RequestMethod.GET, produces = MediaType.APPLICATION_JSON_VALUE)
	public ResponseEntity<List<DataSubmissionDto>> getDataSubmissions(@RequestParam(name = "dataCollection", required = false) Integer dataCollectionId) throws ServiceException {
		
		String scope = this.getGCubeScope();
		String token = this.getGCubeToken();
		
		if (dataCollectionId == null) throw new IllegalArgumentException("No data collection defined");
		
		// TODO Uncomment !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
		UserProfile userProfile = null;
		try {
			userProfile = this.socialNetworkingService.getUserProfile(token, scope);
		} catch (ServiceDiscoveryException e) {
			logger.error(e.getMessage(), e);
			throw new ServiceException("Failed at retrieving user profile");
		}
		/*UserProfile userProfile = new UserProfile();
		userProfile.setUsername("mister.pink");*/
		UserReference user = this.userReferenceService.getUserReferenceByLabel(userProfile.getUsername());
		Domain domain = this.domainService.getDomainByLabel(scope);
		
		if (user == null) throw new ServiceException("No user [" + userProfile.getUsername() + "] exists");
		if (domain == null) throw new ServiceException("No domain [" + scope + "] exists");
		
		List<DataSubmission> dataSubmissions = this.dataSubmissionService.getDataSubmissionsByDataCollectionDomainAndOwner(dataCollectionId, domain.getId(), user.getId());
		dataSubmissions.forEach(dataSubmission -> dataSubmission.getDataCollection().setDataModel(null));
		
		return ResponseEntity.ok(this.entityDtoMapper.entitiesToDtos(dataSubmissions, DataSubmissionDto.class));
	}
	
	@RequestMapping(value = "/{id}/data", method = RequestMethod.GET, produces = MediaType.APPLICATION_JSON_VALUE)
	public ResponseEntity<DataSubmissionDataDto> getDataSubmissionData(@PathVariable("id") Integer id) throws ServiceException {
		DataSubmission dataSubmission = this.dataSubmissionService.getDataSubmission(id);
		
		List<String> headers = getFieldNamesInOrder(dataSubmission.getDataCollection().getDataModel().getDefinition());
		
		DataSubmissionDataDto dataSubmissionData = new DataSubmissionDataDto();
		dataSubmissionData.setHeaders(headers);
		dataSubmissionData.setData(this.entityDtoMapper.entitiesToDtos(dataSubmission.getData(), CdtDto.class));
		try {
			dataSubmissionData.setAttributes(AttributesDto.fromXml(dataSubmission.getAttributes()));
		} catch (JAXBException e) {
			throw new ServiceException("Error while extracting metadata");
		}
		
		return ResponseEntity.ok(dataSubmissionData);
	}
	
	@RequestMapping(value = "/{id}/data", method = RequestMethod.GET, produces = "text/csv")
	public ResponseEntity<String> getDataSubmissionAsCsv(@PathVariable("id") Integer id) throws ServiceException {
		DataSubmission dataSubmission = this.dataSubmissionService.getDataSubmission(id);
		List<String> fieldNames = getFieldNamesInOrder(dataSubmission.getDataCollection().getDataModel().getDefinition());
		
		String csv;
		try {
			csv = constructDataSubmissionCsv(fieldNames, dataSubmission.getData());
		} catch (Exception e) {
			throw new ServiceException("An error occurred while processing csv data", e);
		}
		
		return ResponseEntity.ok(csv);
	}
	
	@RequestMapping(value = "/" + DATA_SUBMISSION_AS_CSV_FILE_ENDPOINT, method = RequestMethod.GET, produces = "text/csv")
	public void getFileOfDataSubmissions(@PathVariable("id") Integer id, HttpServletRequest request, HttpServletResponse response) throws ServiceException {
		DataSubmission dataSubmission = this.dataSubmissionService.getDataSubmission(id);
		
		if (dataSubmission.getStatus() != DataSubmissionStatusType.VALIDATED.code()) {
			throw new ServiceException("Non validated file");
		}
		List<String> fieldNames = getFieldNamesInOrder(dataSubmission.getDataCollection().getDataModel().getDefinition());
		
		String csv;
		try {
			csv = constructDataSubmissionCsv(fieldNames, dataSubmission.getData());
		} catch (Exception e) {
			throw new ServiceException("An error occurred while processing csv data", e);
		}
		
		File file = new File("DataSubmission" + dataSubmission.getId().toString() + ".csv");
		FileWriter fileWriter = null;
		try {
			fileWriter = new FileWriter(file);
			fileWriter.write(csv);
			fileWriter.flush();
			fileWriter.close();
		} catch (IOException e) {
			e.printStackTrace();
			throw new ServiceException("An error occurred while creating the csv file");
		}
		
		response.setContentType("text/csv");
		response.setContentLength((int) file.length());
		response.setHeader("Content-Disposition", "attachment; filename=\"" + file.getName() + "\"");
		
		BufferedInputStream input = null;
		BufferedOutputStream output = null;
		
		FileInputStream fis = null;
		try {
			fis = new FileInputStream(file);
		} catch (FileNotFoundException e) {
			e.printStackTrace();
			throw new ServiceException("An error occurred while creating the stream for the csv file");
		}
		
		try {
			input = new BufferedInputStream(fis);
			output = new BufferedOutputStream(response.getOutputStream());
			byte[] buffer = new byte[1048576];
			int length;
			while ((length = input.read(buffer)) > 0) {
				output.write(buffer, 0, length);
			}
		} catch (IOException e) {
			logger.error("There are errors in reading/writing the stream " + e.getMessage());
		} finally {
			if (output != null)
				try {
					output.close();
				} catch (IOException ignore) {
					ignore.printStackTrace();
				}
			if (input != null)
				try {
					input.close();
				} catch (IOException ignore) {
					ignore.printStackTrace();
				}
		}
	}
	
	private List<String> getFieldNamesInOrder(String dsd) {
		return this.dsdProcessing.getDefinitionForExcelAddIn(dsd).getFields().stream()
				.sorted(Comparator.comparingInt(Field::getOrder)).map(Field::getLabel).collect(Collectors.toList());
	}
	
	private String constructDataSubmissionCsv(List<String> fieldNames, List<Cdt> data) throws Exception {
		String csvHeaders = fieldNames.stream().collect(Collectors.joining(",", "", "\n"));
		String csvData = data.stream()
				.map(cdt -> fieldNames.stream().map(field ->
						(cdt.getData().get(field) == null) ? "" : cdt.getData().get(field).toString()).collect(Collectors.joining(",")))
				.collect(Collectors.joining("\n"));
		
		return csvHeaders + csvData;
	}
	
	@RequestMapping(value = "/{id}/status", method = RequestMethod.POST, consumes = MediaType.APPLICATION_JSON_VALUE, produces = MediaType.APPLICATION_JSON_VALUE)
	public ResponseEntity<DataSubmissionDto> changeDataSubmissionStatus(@PathVariable("id") Integer id, @RequestBody DataSubmissionDto dataSubmissionDto) throws ServiceException {
		dataSubmissionDto.setId(id);
		DataSubmission dataSubmissionEntity = this.entityDtoMapper.dtoToEntity(dataSubmissionDto, DataSubmission.class);
		
		dataSubmissionEntity = this.dataSubmissionService.updateDataSubmission(dataSubmissionEntity);
		
		return ResponseEntity.ok(this.entityDtoMapper.entityToDto(dataSubmissionEntity, DataSubmissionDto.class));
	}
	
	
}

