package org.gcube.data.streams;

import java.net.URI;
import java.util.Iterator;
import java.util.NoSuchElementException;

import org.gcube.data.streams.exceptions.StreamContingencyException;
import org.gcube.data.streams.exceptions.StreamException;
import org.gcube.data.streams.exceptions.StreamOpenException;

/**
 * An iterator over the elements of a dataset of arbitrary origin, including memory, secondary storage, and network. 
 * <p>
 * Streams are:
 * 
 * <ul>
 * <li>addressable, i.e. return a <em>locator</em> that references their address. The use of locators is implementation-dependent;
 * <li>close-able, i.e. their implementation <em>may</em> release resources;
 * </ul>
 * 
 * Streams may raise a broader range of failures than plain {@link Iterator}s, including {@link StreamException}s.
 * Independently from failure handling strategies, the recommended idiom for stream consumption is the following:
 * 
 * <pre>
 * Stream&lt;T> stream = ...
 * try {
 *   while (stream.hasNext())
 *     ....stream.next()...
 * }
 * finally {
 *  stream.close();
 * }
 * </pre>
 * 
 * 
 * 
 * @param <E> the type of elements iterated over 
 * 
 * @author Fabio Simeoni
 * 
 */
public interface Stream<E> extends Iterator<E> {

	
	/**
	 * 
	 * Returns {@code true} if the stream has more elements. 
	 * 
	 * @return {@code true} if the stream has more elements <em>and</em> has not been closed. 
	 * 
	 */
	boolean hasNext();
	
	/**
	 * @throws NoSuchElementException if the stream has no more elements or it has been closed.
	 * @throws StreamContingencyException if the element cannot be returned due to a recoverable error
	 * @throws StreamOpenException if the stream cannot be opened for iteration (unrecoverable)
	 * @throws RuntimeException if the element cannot be returned due to a generic unrecoverable outage or error
	 * 
	 */
	E next() throws StreamException;

	/**
	 * Returns the stream locator.
	 * @return the locator
	 * @throws IllegalStateException if the stream is no longer addressable at the time of invocation. 
	 */
	URI locator() throws IllegalStateException; 
	
	
	/**
	 * Closes the stream unconditionally, releasing any resources that it may be using.
	 * <p>
	 * Subsequent invocations of this method have no effect.<br> 
	 * Subsequents invocations of {@link #hasNext()} return {@code false}.<br> 
	 * Subsequent invocations of {@link #next()}
	 * throw {@link NoSuchElementException}s.
	 * <p>
	 * Failures are logged by implementations and are otherwise suppressed.
	 */
	void close();
	
}