package org.gcube.data.streams.dsl;

import gr.uoa.di.madgik.grs.record.Record;

import java.net.URI;
import java.util.Iterator;

import org.gcube.data.streams.Stream;
import org.gcube.data.streams.adapters.IteratorAdapter;
import org.gcube.data.streams.adapters.IteratorStream;
import org.gcube.data.streams.dsl.fold.InClause;
import org.gcube.data.streams.dsl.from.RsOfClause;
import org.gcube.data.streams.dsl.guard.GuardWithClause;
import org.gcube.data.streams.dsl.listen.MonitorWithClause;
import org.gcube.data.streams.dsl.pipe.PipeThroughClause;
import org.gcube.data.streams.dsl.publish.PublishRsUsingClause;
import org.gcube.data.streams.dsl.publish.PublishRsWithClause;
import org.gcube.data.streams.dsl.unfold.UnfoldThroughClause;
import org.gcube.data.streams.generators.LoggingListener;
import org.gcube.data.streams.generators.NoOpGenerator;
import org.gcube.data.streams.handlers.FaultHandler;
import org.gcube.data.streams.handlers.IgnoreHandler;
import org.gcube.data.streams.handlers.RethrowHandler;
import org.gcube.data.streams.handlers.RethrowUnrecoverableHandler;
import org.gcube.data.streams.handlers.StopFastHandler;
import org.gcube.data.streams.publishers.RsStringRecordFactory;

/**
 * 
 * The definitions of an eDSL of stream and stream-related expressions. 
 * 
 * @author Fabio Simeoni
 *
 */
public class Streams {

	
	// ADAPT

	/**
	 * Adapts an {@link Iterator} to a {@link Stream}.
	 * @param itarator the iterator
	 * @return the stream
	 */
	public static <E> Stream<E> convert(Iterator<E> itarator) {
		return new IteratorStream<E>(itarator);
	}
	
	/**
	 * Adapts an {@link Iterator} to a {@link Stream} using a custom {@link IteratorAdapter}.
	 * @param adapter the adapter
	 * @return the stream
	 */
	public static <E> IteratorStream<E> convert(IteratorAdapter<E> adapter) {
		return new IteratorStream<E>(adapter);
	}
	
	/**
	 * Adapts an {@link Iterable} to a {@link Stream}.
	 * @param iterable the iterable
	 * @return the stream
	 */
	public static <E> Stream<E> convert(Iterable<E> iterable) {
		return convert(iterable.iterator());
	}
	
	/**
	 * Starts a sentence to convert a resultset into a {@link Stream}.
	 * @param locator the locator of the resultset
	 * @return the next clause of the sentence
	 */
	public static RsOfClause<Record> convert(URI locator) {
		return new RsOfClause<Record>(locator);
	}
	
	/**
	 * Returns a {@link Stream} of strings extracted from a resultset of {@link RsStringRecordFactory#STRING_RECORD}s.
	 * @param locator the locator of the resultset
	 * @return the stream
	 */
	public static Stream<String> stringsIn(URI locator) {
		return convert(locator).ofStrings().withDefaults();
	}
	
	// PIPE
	
	/**
	 * Starts a sentence to produce a {@link Stream} generated from another {@link Stream}.
	 * @param stream the input stream.
	 * @return the next clause of the sentence
	 */
	public static <E> PipeThroughClause<E> pipe(Stream<E> stream) {
		return new PipeThroughClause<E>(stream);
	}
	
	
	// FOLD
	
	/**
	 * Starts a sentence to produce a {@link Stream} that groups of elements of another {@link Stream}.
	 * @param stream the input stream.
	 * @return the next clause of the sentence
	 */
	public static <E> InClause<E> fold(Stream<E> stream) {
		return new InClause<E>(stream);
	}
	
	// UNFOLD
	
	/**
	 * Starts a sentence to produce a {@link Stream} that unfolds the elements of another {@link Stream}.
	 * @param stream the input stream.
	 * @return the next clause of the sentence
	 */
	public static <E> UnfoldThroughClause<E> unfold(Stream<E> stream) {
		return new UnfoldThroughClause<E>(stream);
	}
	
	// GUARD
	
	/**
	 * Starts a sentence to produce a {@link Stream} that controls the error raised by another {@link Stream}.
	 * @param stream the input stream.
	 * @return the next clause of the sentence
	 */
	public static <E> GuardWithClause<E> guard(Stream<E> stream) {
		return new GuardWithClause<E>(stream);
	}
	
	// GUARD
	
	/**
	 * Starts a sentence to produce a {@link Stream} that notifies key events in the iteration of another {@link Stream}.
	 * @param stream the input stream.
	 * @return the next clause of the sentence
	 */
	public static <E> MonitorWithClause<E> monitor(Stream<E> stream) {
		return new MonitorWithClause<E>(stream);
	}
	
	// PUBLISH
	
	/**
	 * Starts a sentence to publish a {@link Stream} as a resultset.
	 * @param stream the stream
	 * @return the next clause of the sentence
	 */
	public static <E> PublishRsUsingClause<E> publish(Stream<E> stream) {
		return new PublishRsUsingClause<E>(stream);
	}
	
	/**
	 * Starts a sentence to publish a {@link Stream} as a resultset.
	 * @param stream the stream
	 * @return the next clause of the sentence
	 */
	public static PublishRsWithClause<String> publishStringsIn(Stream<String> stream) {
		return new PublishRsUsingClause<String>(stream).using(no_serialiser);
	}
	
	/**
	 * Returns a {@link Stream} that logs the throughput of an input {@link Stream}.
	 * @param stream the input stream
	 * @return the output stream
	 */
	public static <E> Stream<E> log(Stream<E> stream) {
		LoggingListener<E> listener = new LoggingListener<E>();
		return monitor(pipe(stream).through(listener)).with(listener);
	}
	
	// GENERATORS
	
	/**
	 * Return a {@link NoOpGenerator}.
	 * @return the generator
	 */
	public static NoOpGenerator<String> no_serialiser = new NoOpGenerator<String>();
	
	/**
	 * Return a {@link NoOpGenerator}.
	 * @return the generator
	 */
	public static <E> NoOpGenerator<E> no_op(Stream<E> stream) {
		return new NoOpGenerator<E>();
	}
	
	// HANDLERS
	/**
	 * Returns a {@link RethrowHandler} for failure handling.
	 * @return the handler
	 */
	public static FaultHandler RETHROW_POLICY = new RethrowHandler();
	
	/**
	 * Returns a {@link RethrowUnrecoverableHandler} for failure handling.
	 * @return the handler
	 */
	public static FaultHandler RETHROW_UNRECOVERABLE_POLICY = new RethrowUnrecoverableHandler();
	
	/**
	 * Returns a {@link StopFastHandler} for failure handling.
	 * @return the handler
	 */
	public static FaultHandler STOPFAST_POLICY= new StopFastHandler();
	
	/**
	 * Returns a {@link IgnoreHandler} for failure handling.
	 * @return the handler
	 */
	public static FaultHandler IGNORE_POLICY = new IgnoreHandler();
}
