package org.gcube.data.analysis.tabulardata.cube.metadata;

import java.util.List;

import javax.enterprise.inject.Default;
import javax.inject.Inject;
import javax.inject.Singleton;
import javax.persistence.EntityManager;
import javax.persistence.TypedQuery;

import org.gcube.data.analysis.tabulardata.cube.metadata.exceptions.NoSuchTableException;
import org.gcube.data.analysis.tabulardata.cube.metadata.model.JPATable;
import org.gcube.data.analysis.tabulardata.cube.metadata.model.JPATableFactory;
import org.gcube.data.analysis.tabulardata.cube.metadata.model.TableFactory;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.model.table.TableId;
import org.gcube.data.analysis.tabulardata.model.table.TableType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Function;
import com.google.common.collect.Lists;

@Default
@Singleton
public class JPACubeMetadataWrangler implements CubeMetadataWrangler {

	Logger log = LoggerFactory.getLogger(JPACubeMetadataWrangler.class);

	EntityManager em = null;

	ISEntityManagerProvider emp;

	@Inject
	public JPACubeMetadataWrangler(ISEntityManagerProvider emp) {
		super();
		this.emp = emp;
	}

	// public Table update(Table table){
	// TableConsistencyChecker.checkTableConsistency(table);
	// JPATable jpaTable = JPATableFactory.createJPATable(table);
	// mergeEntity(jpaTable);
	// return TableFactory.createTable(jpaTable);
	// }

	@Override
	public Table save(Table table) {
		TableConsistencyChecker.checkTableConsistency(table);
		JPATable jpaTable = JPATableFactory.createJPATable(table);
		persistEntity(jpaTable);
		return TableFactory.createTable(jpaTable);
	}

	private void initializeIfNot() {
		if (em == null)
			em = emp.get();
	}

	@Override
	public Table get(TableId id) throws NoSuchTableException {
		return TableFactory.createTable(getJPATableById(id.getValue()));
	}

	@Override
	public List<Table> getAll() {
		return Lists.transform(getAllJPATables(), new Function<JPATable, Table>() {

			@Override
			public Table apply(JPATable input) {
				return TableFactory.createTable(input);
			}

		});
	}

	@Override
	public List<Table> getAll(TableType tableType) {
		return Lists.transform(getAllJPATablesByType(tableType), new Function<JPATable, Table>() {

			@Override
			public Table apply(JPATable input) {
				return TableFactory.createTable(input);
			}

		});
	}

	@Override
	public void remove(TableId id) throws NoSuchTableException {
		removeEntity(getJPATableById(id.getValue()));
	}

	private JPATable getJPATableById(long id) throws NoSuchTableException {
		TypedQuery<JPATable> query = getEntityManager().createNamedQuery("Table.findById", JPATable.class);
		query.setParameter("Id", id);
		List<JPATable> queryResult = query.getResultList();
		if (queryResult.size() > 0)
			return queryResult.get(0);
		throw new NoSuchTableException(id);
	}

	private List<JPATable> getAllJPATables() {
		TypedQuery<JPATable> query = getEntityManager().createNamedQuery("Table.findAll", JPATable.class);
		return query.getResultList();
	}

	private List<JPATable> getAllJPATablesByType(TableType tableType) {
		TypedQuery<JPATable> query = getEntityManager().createNamedQuery("Table.findAllByType", JPATable.class);
		query.setParameter("TableType", tableType);
		return query.getResultList();
	}

	private void persistEntity(Object entity) {
		getEntityManager().getTransaction().begin();
		getEntityManager().persist(entity);
		getEntityManager().getTransaction().commit();
		log.debug("Saved entity: " + entity);
	}

	// private void mergeEntity(Object entity){
	// log.debug("Updating entity: " + entity);
	// getEntityManager().getTransaction().begin();
	// getEntityManager().merge(entity);
	// getEntityManager().getTransaction().commit();
	// log.debug("Updated entity: " + entity);
	// }

	private void removeEntity(Object entity) {
		getEntityManager().getTransaction().begin();
		getEntityManager().remove(entity);
		getEntityManager().getTransaction().commit();
	}

	private EntityManager getEntityManager() {
		initializeIfNot();
		return em;
	}

}
