package org.gcube.data.analysis.tabulardata.cube.tablemanagers.hcl;

import java.util.Collection;
import java.util.List;

import org.gcube.data.analysis.tabulardata.cube.data.DatabaseWrangler;
import org.gcube.data.analysis.tabulardata.cube.exceptions.TableCreationException;
import org.gcube.data.analysis.tabulardata.cube.metadata.CubeMetadataWrangler;
import org.gcube.data.analysis.tabulardata.cube.tablemanagers.DefaultTableCreator;
import org.gcube.data.analysis.tabulardata.cube.tablemanagers.TableManager;
import org.gcube.data.analysis.tabulardata.model.column.Column;
import org.gcube.data.analysis.tabulardata.model.column.ColumnType;
import org.gcube.data.analysis.tabulardata.model.column.IdColumn;
import org.gcube.data.analysis.tabulardata.model.idioms.ColumnIsOfType;
import org.gcube.data.analysis.tabulardata.model.table.HierarchicalCodelist;
import org.gcube.data.analysis.tabulardata.model.table.Table;

import com.google.common.collect.Collections2;
import com.google.common.collect.Lists;

public class HCLCreator extends DefaultTableCreator {

	// Columns
	private List<Column> newColumns = Lists.newArrayList();

	// These are used for cloning
	private Table clonedHCL;
	private List<Column> columnsToRemove = Lists.newArrayList();
	private boolean withData;

	public HCLCreator(DatabaseWrangler dbWrangler, CubeMetadataWrangler cmWrangler, TableManager tableManager) {
		super(dbWrangler, cmWrangler, tableManager);
	}

	@Override
	protected Collection<Column> getAllColumns() {
		List<Column> columns = Lists.newArrayList();
		if (clonedHCL != null) {
			columns.addAll(clonedHCL.getColumns());
			columns.removeAll(columnsToRemove);
		}
		columns.addAll(newColumns);
		// Make sure that IdColumn is not on the list
		while (columns.remove(new IdColumn()))
			;
		return columns;
	}

	@Override
	protected Collection<Column> getNewColumns() {
		return newColumns;
	}

	@Override
	protected Table createBaseTable(String tableName, Collection<Column> columns) {
		return new HierarchicalCodelist(0, tableName, columns);
	}

	@Override
	protected void checkConsistency() throws TableCreationException {
		try {
			// Verify label uniqueness
			checkDuplicateLabels(getAllColumns());
		} catch (Exception e) {
			throw new TableCreationException(e.getMessage());
		}

		// Verify minimum constraints
		boolean hasChildRef = false;
		boolean hasParentRef = false;
		Collection<Column> columns = Collections2.filter(getAllColumns(), new ColumnIsOfType(ColumnType.CODELISTREF));
		if (columns.size() != 2)
			throw new TableCreationException("Invalid number of columns of type " + ColumnType.CODELISTREF
					+ ". Only 2 must be provided");
		for (Column column : columns) {
			if (column.getRelationship().isChildParentRelationship())
				hasChildRef = true;
			else
				hasParentRef = true;
		}
		if (!hasChildRef || !hasParentRef)
			throw new TableCreationException("Must provide a column of type " + ColumnType.CODELISTREF
					+ " for both a child-parent relationship and a parent-child relationship.");

		// Verify CodelistRefColumn IDs and referenced column names
		try {
			checkColumnsRelationship(getAllColumns());
		} catch (Exception e) {
			throw new TableCreationException(e.getMessage());
		}
	}

	@Override
	protected boolean isAllowedColumn(Column column) {
		switch (column.getColumnType()) {
		case SYSTEM:
		case CODELISTREF:
			return true;
		default:
			return false;
		}
	}

	@Override
	protected void addNewColumn(Column column) {
		newColumns.add(column);
	}

	@Override
	protected boolean isAllowedCloneableTable(Table table) {
		switch (table.getTableType()) {
		case HIERARCHICALCODELIST:
			return true;
		default:
			return false;
		}
	}

	@Override
	protected Table getTableToClone() {
		return clonedHCL;
	}

	@Override
	protected void setTableToClone(Table table) {
		clonedHCL = table;
	}

	@Override
	protected boolean isCloneWithData() {
		return withData;
	}

	@Override
	protected void setCloneWithData(boolean cloneWithData) {
		withData = cloneWithData;
	}

	@Override
	protected Collection<Column> getColumnsToRemove() {
		return columnsToRemove;
	}

	@Override
	protected void setColumnsToRemove(Collection<Column> columns) {
		columnsToRemove = Lists.newArrayList(columns);
	}

	@Override
	protected void addIndexes(String tableName, Collection<Column> columns) {
		for (Column column : columns){
			switch(column.getColumnType()){
			case SYSTEM:
				return;
			default: 
				dbWrangler.createIndex(tableName, column.getName());
			}
		}
	}

}
