package org.gcube.data.analysis.tabulardata.cube;

import java.util.Collection;

import javax.enterprise.inject.Default;
import javax.enterprise.inject.Instance;
import javax.inject.Inject;
import javax.inject.Named;
import javax.inject.Provider;
import javax.inject.Singleton;

import org.gcube.data.analysis.tabulardata.cube.exceptions.CubeManagerException;
import org.gcube.data.analysis.tabulardata.cube.exceptions.NoSuchTableException;
import org.gcube.data.analysis.tabulardata.cube.tablemanagers.TableCreator;
import org.gcube.data.analysis.tabulardata.cube.tablemanagers.TableManager;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.model.table.TableType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
@Default
public class DefaultCubeManager implements CubeManager {

	static Logger log = LoggerFactory.getLogger(DefaultCubeManager.class);

	private Provider<TableCreator> codelistCreatorProvider;
	private Provider<TableCreator> datasetCreatorProvider;
	private Provider<TableCreator> hclCreatorProvider;
	private Provider<TableCreator> unsafeTableCreatorProvider;
	private Provider<TableCreator> genericTableCreatorProvider;

	private TableManager tableManager;
	
	@Inject
	private DefaultCubeManager(TableManager tableManager, @Named("Codelist") Instance<TableCreator> codelistCreatorProvider,
			@Named("Dataset") Instance<TableCreator> datasetCreatorProvider, @Named("HCL") Instance<TableCreator> hclCreatorProvider,
			@Named("ViewTable") Instance<TableCreator> tempTableCreatorProvider, @Named("GenericTable") Instance<TableCreator> genericTableCreatorProvider) {
		super();
		this.tableManager = tableManager;
		
		this.codelistCreatorProvider = codelistCreatorProvider;
		this.datasetCreatorProvider = datasetCreatorProvider;
		this.hclCreatorProvider = hclCreatorProvider;
		this.unsafeTableCreatorProvider=tempTableCreatorProvider;
		this.genericTableCreatorProvider = genericTableCreatorProvider;
	}

	public TableCreator createDataset() {
		return datasetCreatorProvider.get();
	}

	@Override
	public Collection<Table> getTables() {
		return tableManager.getAll();
	}

	@Override
	public Collection<Table> getTables(TableType tableType) {
		return tableManager.getAll(tableType);
	}

	@Override
	public Table getTable(long id) throws NoSuchTableException {
		return tableManager.get(id);
	}

	@Override
	public void removeTable(long id) throws NoSuchTableException {
		tableManager.remove(id);
	}

	@Override
	public TableCreator createTable(TableType type) {
		switch (type) {
		case CODELIST:
			return codelistCreatorProvider.get();
		case DATASET:
			return datasetCreatorProvider.get();
		case GENERICTABLE:
			return genericTableCreatorProvider.get(); 
		case HIERARCHICALCODELIST:
			return hclCreatorProvider.get();
		case VIEWTABLE:
			return unsafeTableCreatorProvider.get();
			default:
				throw new RuntimeException("Unsupported table type: " + type);
		}
	}

	@Override
	public void updateMetadata(Table table) throws CubeManagerException {
		tableManager.updateMetadata(table);
	}

	@Override
	public void removeTableWithMetdata(long id) throws NoSuchTableException {
		tableManager.removeRaw(id);
	}

}
